/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AbstractReflector.h
 *    Abstract base class for most Reflectors.
 *
 * @author Erik Einhorn
 * @date   2010/09/24
 */

#ifndef _MIRA_ABSTRACTREFLECTOR_H_
#define _MIRA_ABSTRACTREFLECTOR_H_

#ifndef Q_MOC_RUN
#include <boost/type_traits/remove_const.hpp>
#include <boost/mpl/assert.hpp>
#endif

#include <stack>

#include <serialization/ReflectorInterface.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
namespace serialization {

///////////////////////////////////////////////////////////////////////////////

// create a detector for the reflect method
MIRA_MEMBER_DETECTOR(reflect);

///////////////////////////////////////////////////////////////////////////////

} // namespace
///@endcond

/**
 * @ingroup SerializationModule
 *
 * Abstract base class for most Reflectors.
 *
 * This class implements the reflectBase()  method of the
 * ReflectorInterface. It additionally provides the This() method for the
 * "Curiously recurring template pattern" (CRTP) to achieve static polymorphism
 * at compile time.
 *
 * Moreover, the AbstractReflector implements the invoke() method which invokes
 * the Reflector on an object of a certain class by calling the class' reflect
 * method. If the class does not contain such an intrusive reflect method, the
 * non-intrusive reflect method, that must be defined outside of the
 * class within the same namespace or in the global namespace, will be used.
 *
 * Derived Reflectors can overwrite the invokeOverwrite() method.
 * invokeOverwrite() is called directly from invoke() and ensures that the
 * invokeOverwrite() of the most derived class is called (some kind of
 * static virtual function call without performance penalty, see CRTP).
 * If a Reflector overwrites invokeOverwrite() it should always call
 * invokeOverwrite() of this base class.
 *
 * Beside the invoke() method a Reflector may be passed as parameter when
 * calling a reflect method directly. However, you usually should use the
 * invoke() method to ensure, that invokeOverwrite() method of the derived
 * classes is called.
 *
 * However, usually the final Reflectors that are used by the "end-user"
 * like the @ref Serializer and @ref Deserializer provide their own special
 * methods (e.g. serialize, deserialize, ...) for invoking the Reflector.
 * For those Reflectors these methods must be called, instead of the invoke()
 * method. These Reflectors will call invoke(), internally. See the
 * documentation of the concrete derived Reflectors for more details. For this
 * reason the invoke() method of the AbstractReflector is protected. The
 * derived concrete Reflectors may expose the method or provide their own
 * public interface (as @ref Serializer and @ref Deserializer do).
 *
 *
 * <b>Implements from base classes:</b>
 * reflectBase()
 *
 *
 * <b>Methods that can be overwritten:</b>
 * invokeOverwrite()
 *
 *
 * <b>Implementation Detail:</b>
 *
 * This class and derived classes uses the Curiously recurring template
 * pattern (CRTP) to implement static polymorphism. To do so, each class that
 * derives from ReflectorInterface or any derived class gives its own type as
 * template parameter, e.g.
 *
 *    class MyClass : public ReflectorInterface<MyClass>
 *
 * Hence, the ReflectorInterface and derived classes know about their concrete
 * derived class (that is MyClass in the above example). The This()-method then
 * can safely cast the this pointer into the concrete derived type. Therefore,
 * the Reflector and all derived classes can call methods of the final derived
 * concrete class using This()->method();
 *
 * See Wikipedia for more details!
 *
 * <b>Implementation Detail:</b>
 *
 * @see @ref SerializationPage
 */
template <typename Derived>
class AbstractReflector : public ReflectorInterface<Derived>
{

public:
	typedef serialization::VersionType VersionType;

	/// implements ReflectorInterface (for documentation see ReflectorInterface)
	template <typename T>
	VersionType requireVersion(VersionType version, VersionType minVersion, const T* caller = NULL) {
		VersionType v = This()->template version<T>(version);
		checkVersion(v, minVersion);
		return v;
	}

	MIRA_DEPRECATED("Please call as requireVersion<MyType>(v, minV) or requireVersion(v, minV, this)",
	VersionType requireVersion(VersionType version, VersionType minVersion)) {
		VersionType v = This()->version(version);
		checkVersion(v, minVersion);
		return v;
	}

	template <typename T>
	void requireVersion(VersionType requiredVersion, const T* caller = NULL) {
		requireVersion<T>(requiredVersion, requiredVersion);
	}

	MIRA_DEPRECATED("Please call as requireVersion<MyType>(v) or requireVersion(v, this)",
	void requireVersion(VersionType requiredVersion)) {
		requireVersion(requiredVersion, requiredVersion);
	}

private:

	void checkVersion(VersionType version, VersionType minVersion) {
		if(version < minVersion)
			MIRA_THROW(XIO, "Found version: " << (int)version <<
			                ", but require at least version: " << (int)minVersion);
	}

public:

	/// implements ReflectorInterface (for documentation see ReflectorInterface)
	template <typename Base>
	void reflectBase(Base& base) {
		this->invoke(base);
	}

protected:

	/**
	 * "Curiously recurring template pattern" (CRTP).
	 * Safely casts this into Derived. This is safe since when implementing
	 * derived class their type is passed as template parameter.
	 *
	 * See Wikipedia for more details on CRTP if necessary!
	 */
	Derived* This()
	{
		// the Derived was derived from us, so we can safely cast to it ...
		return static_cast<Derived*>(this);
	}

	/**
	 * Invokes this reflector on the specified object. This is like calling
	 * object.reflect(*this). However, this automatically examines if the
	 * class T contains a reflect method. In this case it calls the intrusive
	 * reflect method, otherwise it uses the non-intrusive reflect method.
	 */
	template<typename T>
	void invoke(T& object)
	{
		this->This()->invokeOverwrite(object);
	}

	/**
	 * The actual invoke implementation, that may also be overwritten in
	 * derived classes to add additional functionality.
	 * @note If you overwrite this method you should call the implementation
	 *       of the base class unless you perform your own invocation process.
	 */
	template<typename T>
	void invokeOverwrite(T& object)
	{
		using namespace serialization;

		typedef MIRA_HAS_MEMBER_TEMPLATE(T, reflect) hasMember;
		typedef typename
				boost::mpl::eval_if<hasMember,
				boost::mpl::identity< ReflectComplexIntrusive >,   // T is a class with reflect method (Type B1)
				//else
				boost::mpl::identity< ReflectComplexNonintrusive > // T is a class without reflect method (Type B2)
		>::type chosen_reflect;

		chosen_reflect::invoke(*this->This(), object);
	}

	/**
	 * For classes with reflect method call their reflect method directly.
	 */
	struct ReflectComplexIntrusive {
		template<typename T>
		static void invoke(Derived& r, T& object) {
			typedef typename boost::remove_const<T>::type TwithoutConst;
			TwithoutConst& nonconstObject = const_cast<TwithoutConst&>(object);
			nonconstObject.reflect(r);
		}
	};

	/**
	 * For classes without reflect method, where we need to look somewhere else
	 * to get the information for visiting it. To do so we call a global
	 * non-intrusive reflect-function which must be specialized for the visitor
	 * and the member type. This allows non-intrusive reflection. If the
	 * non-intrusive reflect method does not exist we will get a compiler error
	 * here!
	 */
	struct ReflectComplexNonintrusive {
		template<typename T>
		static void invoke(Derived& r, T& object) {
			// #################################################################
			// If you get a compiler error here, you do not have specified a
			// proper non-intrusive reflect method for the type T. You can
			// either provide an intrusive or non-intrusive reflect method.
			// #################################################################
			typedef typename boost::remove_const<T>::type TwithoutConst;
			reflect(r, const_cast<TwithoutConst&>(object));
		}
	};
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif // _MIRA_ABSTRACTREFLECTOR_H_
