/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCClient.h
 *    Implementation of the client side of an RPC call.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/11/10
 */

#ifndef _MIRA_RPCCLIENT_H_
#define _MIRA_RPCCLIENT_H_

#ifndef Q_MOC_RUN
#include <boost/preprocessor/repetition.hpp>
#include <boost/thread/mutex.hpp>
#endif

#include <serialization/ReflectorInterface.h>

#include <rpc/AbstractRPCClient.h>
#include <rpc/RPCSignature.h>
#include <rpc/RPCFuture.h>
#include <rpc/RPCMacros.h>


namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup RPCModule
 *
 * The RPCClient is responsible for handling the client-side of an rpc call.
 * The client side view of an rpc call is as follows:
 * -# the call() method is called and generates an RPCRequest based on the
 *    desired service and method strings and the parameters of the method.
 *    - the RPCRequest can be a realization of multiple possible types, like:
 *      binary requests, json rpc request, etc.
 *    - the RPCRequest can be thought of some kind of container that contains
 *      all information about the desired call (for binary rpc calls it is a
 *      buffer with serialized data, with serialized parameters etc.)
 *    - the result of the call() method is an RPCFuture which allows the
 *      caller to wait until the call has finished and to access the result
 *      value of the call as soon as it becomes available.
 *    - the calls are identified by IDs which are generated randomly based on UUIDs
 *    - internally a PendingResponse object is created for each call and stored
 *      in an internal map together with the call id. The PendingResponse object
 *      is kept until the RPCFuture of the call is destroyed (e.g. since the
 *      caller is not interested in the response) or until the actual response
 *      was received.
 * -# the data of the RPCRequest generated by the call method is shipped as
 *    RPCClientRequest to the remote side where it is processed as RPCServerRequest
 *    by the RPCServer and the actual call is performed. As result the RPCServer
 *    will return an RPCServerResponse.
 * -# the response is received as RPCClientResponse at the client side
 * -# the handleResponse method of the RPCClient is called with the received
 *    response. It will extract the return value (if any) and set that value
 *    to the RPCFuture in order to signal the caller that the desired call has
 *    finished and to return the result.
 *    - internally the handleResponse method looks for the corresponding
 *      PendingResponse object (that was created in step 1) using the ID of the
 *      call, which is included in the response header
 *    - the PendingResponse object is responsible for handling the response
 *
 * @see @ref ServiceRPCPage, RPCServer
 */
class RPCClient : public AbstractRPCClient
{
protected:

	/**
	 * Interface for PendingResponse.
	 * @see PendingResponse
	 */
	struct PendingResponseBase {
		virtual ~PendingResponseBase() {}
		/// @see PendingResponse
		virtual void handleResponse(void* responsePtr) = 0;
		/// @see PendingResponse
		virtual int getBackendTypeId() const = 0;
	};

	/**
	 * PendingResponse template class.
	 * This class has to be a template, since we must handle different
	 * response types (e.g. Binary responses, JSON responses, etc) with different
	 * return value types (e.g. void, float, etc.). The return types depend on
	 * the rpc method that is called. Hence, for each rpc call an PendingResponse
	 * object with the matching types is created in the call() method below.
	 * This PendingResponse object is responsible for handling the response
	 * that is returned by the RPCServer after the call was invoked and processed.
	 * The PendingResponse object also holds the boost::promise object that is
	 * used to generate the RPCFuture and that is used to set the return value
	 * in the handleResponse method.
	 */
	template <typename Backend, typename R>
	struct PendingResponse : public PendingResponseBase {

		typedef typename Backend::ClientResponse Response;

		PendingResponse() : backendTypeId(typeId<Backend>()) {}

		/**
		 * Handles the response by calling getReturn() of the RPCResponse backend.
		 * The getReturn() method will set the value to the promise object (which
		 * communicates with the RPCFuture)
		 */
		virtual void handleResponse(void* responsePtr) {
			Response* concreteResponsePtr = static_cast<Response*>(responsePtr);
			concreteResponsePtr->getReturn(promise);
		}

		/**
		 * Returns the (non portable) type id of the used backend. It is
		 * used by RPCClient::handleResponse() for type checking where it is
		 * compared with the type of the actually received response.
		 */
		virtual int getBackendTypeId() const { return backendTypeId; }

		/// The boost::promise object that is used for generating and setting the RPCFuture
		boost::promise<R> promise;
		int backendTypeId;
	};

	typedef boost::shared_ptr<PendingResponseBase> PendingResponsePtr;

public:

	#define RPCGEN_CALL_BODY_SET_PARAMETER(z,n,_) request.setParameter(p##n);

	/**
	 * Generates the call(...) methods for multiple parameter counts using
	 * boost preprocessor.
	 */
	#define RPCGEN_CALL_EX(z,n,_)                                              \
	template<typename Backend, typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>  \
	RPCFuture<R> call(typename Backend::ClientRequest& request, const std::string& service, \
	                  const std::string& method                                \
	                  BOOST_PP_REPEAT_ ## z(n,RPCGEN_CALL_PARAM_DECL,nil)) {   \
		RPCSignature s = makeRPCSignature<R                                    \
							BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>(method);   \
		std::string callId = request.generateCallID();                         \
		RPCFuture<R> future = addPendingResponse<Backend,R>(callId);           \
		request.setHeader(callId,service,s);                                   \
		BOOST_PP_REPEAT_ ## z(n,RPCGEN_CALL_BODY_SET_PARAMETER,nil)            \
		return future;                                                         \
	}
	BOOST_PP_REPEAT(BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),RPCGEN_CALL_EX, nil)


#ifdef DOXYGEN
	// this is a fake method to generate a doxygen documentation for the call()
	// methods created above

	/**
	 * Generates an RPCRequest for an RPC call.
	 * This method takes the name of the service, the name of the RPC method
	 * and a variable number of optional parameters p1 ... pn to generate an
	 * RPC request that is returned in the first parameter. The request type
	 * can be a BinaryRPCClientRequest, JsonRPCClientRequest, etc.
	 *
	 * @param[out] request The request object which will be filled with all information
	 *                     of the request
	 * @param service      The name of the service
	 * @param method       The method name of the requested call
	 * @param p1           The first optional parameter. See pn.
	 * @param pn           Variable number of optional parameters. The number
	 *                     and type depends on the requested RPC method. Both
	 *                     will be checked by the RPC server. If the parameter
	 *                     types do not match, the server will return an error.
	 * @return Returns an RPCFuture object. The type of the RPCFuture depends
	 *         on the return type of the RPC method. It allows the caller to
	 *         wait for the RPC call to finish and to retrieve the return value.
	 *         The return value for the RPCFuture will be set by handleResponse().
	 *
	 * Example:
	 * \code
	 *
	 *     RPCFuture<float> result = rpcclient.call<BinaryRPCBackend, float>(request, "myservice", "mymethod", 123, 1.23f);
	 *
	 *     result.wait(); // wait for the result
	 *
	 *     float r = result.get(); // get the result value
	 *
	 * \endcode
	 */
	template<typename Backend, typename R, typename P1, ..., typename Pn>
	RPCFuture<R> call(typename Backend::ClientRequest& request, const std::string& service,
	                  const std::string& method, P1 p1, ..., Pn pn);
#endif


	/**
	 * Adds a new "PendingResponse" object for a call (with the specified
	 * callid) that was either generated by the above call() method or by a call
	 * that was created manually (e.g. JSON calls).
	 *
	 * This method is internally used by the call() method.
	 *
	 * @return Returns an RPCFuture object. The type of the RPCFuture depends
	 *         on the return type of the RPC method. It allows the caller to
	 *         wait for the RPC call to finish and to retrieve the return value.
	 *         The return value for the RPCFuture will be set by handleResponse().
	 */
	template <typename Backend, typename R>
	RPCFuture<R> addPendingResponse(const std::string& callId)
	{
		boost::shared_ptr<PendingResponse<Backend, R>> ptr(new PendingResponse<Backend,R>());
		boost::mutex::scoped_lock lock(mMutex);
		// the following may overwrite an existing pending response, which is
		// okay, as it might be intended by the user.
		mPendingResponses[callId] = ptr;
		return RPCFuture<R>(ptr->promise.get_future(), this, callId);
	}

public:

	/**
	 * Is called after the RPCServer has sent the response of the RPC call
	 * to handle the response and to set the return value or return state of
	 * the RPCFuture that was returned by the call method.
	 */
	template <typename Backend>
	void handleResponse(typename Backend::ClientResponse& response)
	{
		std::string callId;
		response.getHeader(callId);

		boost::mutex::scoped_lock lock(mMutex);
		auto it = mPendingResponses.find(callId);

		// if call was already removed since future was destroyed, we can skip
		// handling the response and safe performance
		if(it==mPendingResponses.end())
			return;

		// check if expected response type and the given response type do match
		if(typeId<Backend>()!=it->second->getBackendTypeId())
			MIRA_THROW(XLogical, "The expected Response Backend type does not match "
			           "the given Response. Probably you sent an RPC call with a "
			           "different Request Backend that does not match to the received "
			           "Response");

		PendingResponsePtr r = it->second;
		mPendingResponses.erase(it);
		lock.unlock();

		r->handleResponse(&response);

		// the pending response r and its promise will be destroyed here,
		// which is okay, since the value or exception was set in
		// handleResponse above, hence no "broken promise" will occure
	}

	/**
	 * Is called by the RPCFutures that are created by the call() method upon
	 * the destruction of the RPCFutures. Internally, the pending responses
	 * are removed since no one is longer interested in the response.
	 */
	virtual void onDestructFuture(const std::string& callId)
	{
		boost::mutex::scoped_lock lock(mMutex);
		mPendingResponses.erase(callId);
	}

private:

	typedef std::map<std::string, PendingResponsePtr> PendingResponses;
	PendingResponses mPendingResponses;
	boost::mutex mMutex;
};


///////////////////////////////////////////////////////////////////////////////

}


#endif
