/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file JSON.h
 *    Wrappers for JSON.
 *
 * @author Tim Langner
 * @date   2011/06/21
 */

#ifndef _MIRA_JSON_H_
#define _MIRA_JSON_H_

#include <iostream>

#ifndef Q_MOC_RUN
#include <json_spirit_value.h>
#endif

#include <error/Exceptions.h>

#include <serialization/IsAtomicSerializable.h>

#include <utils/Singleton.h>

#include <stream/BinaryStream.h>

#ifndef MIRA_WINDOWS
// export the json templates to avoid recurring instantiations and therefore reduce compile time
extern template class json_spirit::Config_map<std::string>;
extern template class json_spirit::Value_impl<json_spirit::Config_map<std::string>>;
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

namespace json {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL

/**
 * Inherits from boost::true_type, if T is a JSON base type, i.e. string or
 * arithmetic type.
 */
template <typename T>
struct IsBaseType :
	boost::mpl::bool_<boost::is_same<T, std::string>::value || boost::is_arithmetic<T>::value>
{};

/**
 * Template type trait mapping C++ data types
 * to JSON base types.
 */
template <typename T>
struct TypeTrait
{
	typedef T type;
};

/**
 * Macro for defining a JSON TypeTrait easily
 */
#define MIRA_JSON_TRAIT(B,T)  \
template <>                   \
struct TypeTrait<T>           \
{                             \
	typedef B type;           \
};

MIRA_JSON_TRAIT(int, char)
MIRA_JSON_TRAIT(int, uint8)
MIRA_JSON_TRAIT(int, uint16)
MIRA_JSON_TRAIT(int, uint32)
MIRA_JSON_TRAIT(int, int8)
MIRA_JSON_TRAIT(int, int16)
MIRA_JSON_TRAIT(int, int32)
MIRA_JSON_TRAIT(double, float)
/// warning jsons internal floating point type is double so using long double
/// can lead to loss in precision and data
MIRA_JSON_TRAIT(double, long double)


/**
 * Casts a C++ type into a JSON type.
 */
template <typename T>
typename TypeTrait<T>::type cast(const T& value)
{
	return static_cast<typename TypeTrait<T>::type>(value);
}

template <typename T>
T reverse_cast(const typename TypeTrait<T>::type& value)
{
	return static_cast<T>(value);
}

///@endcond

///////////////////////////////////////////////////////////////////////////////

/**
 * JSONDefaultPrecision is a singleton that provides a mechanism to control
 * the default precision for output of floating-point JSON values.
 */
class JSONDefaultPrecision : public Singleton<JSONDefaultPrecision>
{
	friend class singleton::CreateUsingNew<JSONDefaultPrecision>;

protected:
	JSONDefaultPrecision() : mPrecision(MIRA_JSON_DEFAULT_PRECISION) {}

	/// Default value if not changed using set().
	static const unsigned int MIRA_JSON_DEFAULT_PRECISION = 3;

public:
	/// Set default precision for json::write().
	static void set(unsigned int precision) { instance().mPrecision = precision; }

	/// Reset default precision for json::write() to initial default value (\ref MIRA_JSON_DEFAULT_PRECISION).
	static void reset() { instance().mPrecision = MIRA_JSON_DEFAULT_PRECISION; }

	/// Query current default precision for json::write().
	static unsigned int get() { return instance().mPrecision; }

private:
	unsigned int mPrecision;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * A value is an abstract description of data in JSON (underlying data can
 * either be one of the JSON base data types (int, double, string) or an
 * json::Object, json::Array or null.
 */
typedef json_spirit::mValue  Value;

/**
 * A representation of an object (class, struct) in JSON.
 * An object can have multiple members that are stored as named json::Value
 * in a map.
 */
typedef json_spirit::mObject Object;

/**
 *  A representation of an array (vector) in JSON.
 *  An array can have multiple items that are represented as json::Value in a
 *  vector.
 */
typedef json_spirit::mArray  Array;

//////////////////////////////////////////////////////////////////////////////

/**
 * Writes a json::Value into a given stream using the JSON format.
 * @param value The JSON value
 * @param[in,out] ioStream The stream
 * @param formatted If true indentations, spaces and line breaks are inserted
 * @param precision Decimal precision for output of floating-point values
 *                  when writing value to stream (argument to std::setprecision).
 *                  If < 0, query JSONDefaultPrecision.
 */
MIRA_BASE_EXPORT void write(const Value& value, std::ostream& ioStream,
                            bool formatted=false, int precision = -1);

/**
 * Writes a json::Value into a string using the JSON format.
 * @param value The JSON value
 * @param formatted If true indentations, spaces and line breaks are inserted
 * when writing value to stream.
 * @param precision Decimal precision for output of floating-point values when
 *                  writing value to intermediate stream (argument to std::setprecision).
 *                  If < 0, query JSONDefaultPrecision.
 * @return string containing JSON text
 */
MIRA_BASE_EXPORT std::string write(const Value& value, bool formatted=false,
                                   int precision = -1);

/**
 * Read a json::Value from a string that contains JSON format.
 * @throw XIO when string does not contain valid JSON format.
 * @param s The string to convert
 * @param[out] oValue The returned value as reference
 */
MIRA_BASE_EXPORT void read(const std::string& s, Value& oValue);

/**
 * Read a json::Value from a stream that contains JSON format.
 * @throw XIO when stream does not contain valid JSON format.
 * @param[in,out] ioStream The stream to convert
 * @param[out] oValue The returned value as reference
 */
MIRA_BASE_EXPORT void read(std::istream& ioStream, Value& oValue);

/**
 * Read a json::Value from a string iterator range that contains JSON format.
 * @throw XIO when the range does not contain valid JSON format.
 * @param begin The iterator pointing to the start of the range
 * @param end The iterator pointing to the end of the range
 * @param[out] oValue The returned value as reference
 */
MIRA_BASE_EXPORT inline void read(std::string::const_iterator& begin,
                                  std::string::const_iterator& end,
                                  Value& oValue);

/**
 * Get a json::Value element/member from a json::Value
 * Use [index] for array access.
 * (e.g. ArrayMember[1])
 * @throw XInvalidConfig if the element could not be found or value is
 * not an Object.
 * @param[in] iValue The value to get the element from
 * @param[in] elementName The name of the element to get the json::Value from
 * @return The resulting element
 */
MIRA_BASE_EXPORT Value getElement(const Value& iValue,
                                  const std::string& elementName);

/**
 * Get a number element/member from a json::Value (e.g for plotting)
 * Use [index] for array access and [row][col] for Eigen matrices.
 * (e.g. Velocity.Vt[0][0] for Odometry JSON object)
 * @throw XInvalidConfig if the element could not be found, value is
 * not an Object or element is not of int or floating point type.
 * @param[in] iValue The value to get the element from
 * @param[in] elementName The name of the element to get the number from
 * @return floating point element
 */
MIRA_BASE_EXPORT double getNumberElement(const Value& iValue,
                                         const std::string& elementName);

//////////////////////////////////////////////////////////////////////////////

} // json


/// Imports the json::Value type into mira namespace
typedef json::Value  JSONValue;
/// Imports the json::Object type into mira namespace
typedef json::Object JSONObject;
/// Imports the json::Array type into mira namespace
typedef json::Array  JSONArray;

template <typename OStream>
inline OStream& operator<<(OStream& os, const JSONValue& value)
{
	os << json::write(value, false);
	return os;
}

inline BinaryStlOstream& operator<<(BinaryStlOstream& os, const JSONValue& value)
{
	os << json::write(value, false, 0);
	return os;
}

inline BinaryBufferOstream& operator<<(BinaryBufferOstream& os, const JSONValue& value)
{
	os << json::write(value, false, 0);
	return os;
}

template <typename IStream>
inline IStream& operator>>(IStream& is, JSONValue& value)
{
	std::string tmp;
	is >> tmp;
	json::read(tmp, value);
	return is;
}

template <>
class IsAtomicSerializable<JSONValue> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

} // mira

#endif
