/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FactoryMacros.h
 *    $Macros for registering classes$.
 *
 * @author Ronny Stricker
 * @date   2011/05/24
 */

#ifndef _MIRA_FACTORYMACROS_H_
#define _MIRA_FACTORYMACROS_H_

#ifndef Q_MOC_RUN
#include <boost/mpl/assert.hpp>
#endif

#include <platform/Typename.h>

///@cond INTERNAL
// class which will be abused as compile time error message
template< class T >
class ________________________________PLEASE_USE_THE__MIRA_TEMPLATE_CLASS_REGISTER__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________
{
	virtual int FOR_CLASS();
	virtual ~________________________________PLEASE_USE_THE__MIRA_TEMPLATE_CLASS_REGISTER__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________() {}
};

// class which will be abused as compile time error message
template< class T >
class ________________________________PLEASE_USE_THE__MIRA_CLASS_REGISTER__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________
{
	virtual int FOR_CLASS();
	virtual ~________________________________PLEASE_USE_THE__MIRA_CLASS_REGISTER__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________() {}
};

// class which will be abused as compile time error message
template< class T >
class ________________________________PLEASE_USE_THE__MIRA_NO_PUBLIC_DEFAULT_CONSTRUCTOR__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________
{
	virtual int FOR_CLASS();
	virtual ~________________________________PLEASE_USE_THE__MIRA_NO_PUBLIC_DEFAULT_CONSTRUCTOR__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________() {}
};

///@endcond

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @brief Auxiliary function to extract the class name from the given identifier.
 * If '::' is found within the given string, the function will stripe all
 * leading namespace identifiers (identified by '::')
 */
inline std::string mira_stripNameFromIdentifier( std::string const& identifier )
{
	size_t pos = std::string::npos;
	if ( identifier[identifier.length()-1] == '>' )
		pos = identifier.rfind('<');
	std::string tReturn = ( identifier.rfind("::",pos) != std::string::npos ) ?
			identifier.substr( identifier.rfind("::",pos) + 2 )
			: identifier;
	return tReturn;
}

///@cond INTERNAL

// *****************************************************************************
// *** ClassFactory MACROS (Protected)
// *****************************************************************************

/**
 * @brief Do not use this macro directly.
 * It is used by the other class factory object macros to avoid code duplication.
 */
#define MIRA_OBJECT_BASE( classIdentifier ) \
	friend class mira::ClassFactory; \
	\
	template<typename pCLASS, typename pBASECLASS> \
	friend struct mira::FactoryRegisterClassHelper; \
	\
	template<typename pCLASS> \
	friend class mira::TClass; \
\
public: \
	/** Return specialized TClass for this class */ \
	mira::TClass<classIdentifier> const& getClass() const { \
		/* This should do the trick, since all TClasses should be of the same*/\
		/* structure and size. However, if you get a serious error here, keep*/\
		/* it private because i will be really really disenchanted otherwise.*/\
		return \
			static_cast<mira::TClass<classIdentifier> const&>( internalGetClass() ); \
	} \
	\
	/** Return specialized TClass. */ \
	static mira::TClass<classIdentifier> const& CLASS() { \
		return _CLASS(); \
	} \
\
protected: \
	/** Wrap static _CLASS() function to access most derived class */ \
	virtual mira::Class& internalGetClass() const { \
		return _CLASS(); \
	} \
	static mira::TClass<classIdentifier>& _staticClassObject( \
			std::string const& identifier, \
			std::string const& name, \
			bool libLoaded);

///@endcond

// *****************************************************************************
// *** ClassFactory MACROS (Public)
// *****************************************************************************

/**
 * @brief Use this MACRO if you don't like the factory to automatically extract
 * the class name from the given identifier. You have to pass Identifier and
 * Class name separated by comma.
 * @ingroup ClassFactoryModule
 */
#define MIRA_NAMEDOBJECT( classIdentifier, className ) \
	MIRA_OBJECT_BASE( MIRA_PPARAM( classIdentifier ) ) \
	\
	/** Return the unique instance of the concrete class object */ \
	static mira::TClass<classIdentifier>& _CLASS() { \
		return _staticClassObject( mira::typeName<classIdentifier>(), \
			#className, true ); \
	} \
private: \
	/** This function is called by the class factory to add meta info */ \
	static void addMetaInfo( std::map<std::string, std::string>& metaMap ) \
	{ \
		/* No meta info given -> do nothing */ \
	}

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief Use this MACRO if you like the factory to automatically extract
 * the class name from the given identifier. You cannot add meta information
 * to your class with this MACRO. If you like to do so, use MIRA_META_OBJECT
 * instead.
 * @ingroup ClassFactoryModule
 */
#define MIRA_OBJECT( classIdentifier ) \
	MIRA_OBJECT_BASE( MIRA_PPARAM( classIdentifier ) ) \
	\
	/** Return the unique instance of the concrete class object */ \
	static mira::TClass<classIdentifier>& _CLASS() { \
		return _staticClassObject( mira::typeName<classIdentifier>(), \
			mira::mira_stripNameFromIdentifier( \
					mira::typeName<classIdentifier>() ), true ); \
	} \
private: \
	/** This function is called by the class factory to add meta info */ \
	static void addMetaInfo( std::map<std::string, std::string>& metaMap ) \
	{ \
		/* No meta info given -> do nothing */ \
	}

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief Use this MACRO if you like add additional meta information to your
 * class. You have to pass the class identifier and a list of meta information
 * enclosed in brackets. Example:
 * MIRA_META_OBJECT( ClassIdentifer, ("MetaKey1","MetaValue1")("MetaKey2","MetaValue2") );
 * @ingroup ClassFactoryModule
 */
#define MIRA_META_OBJECT( classIdentifier, MetaInfo ) \
	MIRA_OBJECT_BASE( MIRA_PPARAM( classIdentifier ) ) \
	\
	/** Return the unique instance of the concrete class object */ \
	static mira::TClass<classIdentifier>& _CLASS() { \
		return _staticClassObject( mira::typeName<classIdentifier>(), \
			mira::mira_stripNameFromIdentifier( \
					mira::typeName<classIdentifier>() ), true ); \
	} \
private: \
 	 /** This function is called by the class factory to add meta info */ \
	static void addMetaInfo( std::map<std::string, std::string>& metaMap ) \
	{ \
		/* append given meta information to the map */ \
		boost::assign::insert( metaMap ) MetaInfo; \
	}

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief Use this macro to register your class at the ClassFactory.
 * Pass your Class and the parent Class. Use the macro several times if your
 * class do have more than one parent.
 * @ingroup ClassFactoryModule
 */
#define MIRA_CLASS_REGISTER( Class, ... ) \
	/* generate error message if macro is not used in global namespace */ \
	template <> int \
	________________________________PLEASE_USE_THE__MIRA_CLASS_REGISTER__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________ \
	<Class>::FOR_CLASS() { return 0; } \
	MIRA_REGISTRAR( MIRA_PPARAM( mira::FactoryRegisterClass<Class, __VA_ARGS__> ) ) \
	mira::TClass<Class>& Class::_staticClassObject(std::string const& identifier, \
	                                               std::string const& name, \
	                                               bool libLoaded) { \
		static mira::TClass<Class> sClass(identifier, name, libLoaded); \
		return sClass; \
	}

/**
 * Use this macro if your class does not have a public default constructor
 * and should be managed by the class factory.
 */
#define MIRA_NO_PUBLIC_DEFAULT_CONSTRUCTOR(CLASS) \
	namespace mira { \
		template<> \
		class HasPublicDefaultConstructor<CLASS> : public std::false_type {}; \
	}

inline std::string replaceTemplateGeneric( std::string const& genericIdent,
			std::string const& specialIdent )
{
	size_t tFirstOccGeneric = genericIdent.find('<');
	size_t tLastOccGeneric = genericIdent.rfind('>');
	size_t tFirstOccSpecial = specialIdent.find('<');
	size_t tLastOccSpecial = specialIdent.rfind('>');

	assert( tFirstOccGeneric != std::string::npos &&
			tLastOccGeneric != std::string::npos );

	assert( tFirstOccSpecial != std::string::npos &&
			tLastOccSpecial != std::string::npos );

	std::string tReturnValue = genericIdent;

	return tReturnValue.replace( tFirstOccGeneric, tLastOccGeneric-tFirstOccGeneric,
			specialIdent.substr( tFirstOccSpecial,
			tLastOccSpecial - tFirstOccSpecial ) );
}

 ///@cond INTERNAL

#define MIRA_INTERNAL_OPEN_NAMESPACE(r,data,t) \
		namespace t {

#define MIRA_INTERNAL_CLOSE_NAMESPACE(r,data,t)	}

/// obtain number of variadic macro aruments (up to 5)
/// see http://groups.google.com/group/comp.std.c/browse_thread/thread/77ee8c8f92e4a3fb/346fc464319b1ee5?pli=1
#define MIRA_VA_NUM_ARGS(...) MIRA_VA_NUM_ARGS_IMPL(__VA_ARGS__,5,4,3,2,1)
#define MIRA_VA_NUM_ARGS_IMPL(_1,_2,_3,_4,_5,N,...) N


// see: http://www.mail-archive.com/boost@lists.boost.org/msg03318.html
#define MIRA_MAKE_SEQ(size, rel) MIRA_MAKE_SEQ_D(size, rel)
#define MIRA_MAKE_SEQ_D(size, rel) \
		BOOST_PP_CAT( \
				MIRA_MAKE_SEQ_A_ ## size rel, \
				0X0 \
		)() \
		/**/

// size 1
#define MIRA_MAKE_SEQ_A_1(a) ((a)) MIRA_MAKE_SEQ_B_1
#define MIRA_MAKE_SEQ_B_1(a) ((a)) MIRA_MAKE_SEQ_A_1

#define MIRA_MAKE_SEQ_A_10X0()
#define MIRA_MAKE_SEQ_B_10X0()

// size 2
#define MIRA_MAKE_SEQ_A_2(a, b) ((a, b)) MIRA_MAKE_SEQ_B_2
#define MIRA_MAKE_SEQ_B_2(a, b) ((a, b)) MIRA_MAKE_SEQ_A_2

#define MIRA_MAKE_SEQ_A_20X0()
#define MIRA_MAKE_SEQ_B_20X0()

// size 3
#define MIRA_MAKE_SEQ_A_3(a, b, c) ((a, b, c)) MIRA_MAKE_SEQ_B_3
#define MIRA_MAKE_SEQ_B_3(a, b, c) ((a, b, c)) MIRA_MAKE_SEQ_A_3

#define MIRA_MAKE_SEQ_A_30X0()
#define MIRA_MAKE_SEQ_B_30X0()

// size 4
#define MIRA_MAKE_SEQ_A_4(a, b, c, d) ((a, b, c, d)) MIRA_MAKE_SEQ_B_4
#define MIRA_MAKE_SEQ_B_4(a, b, c, d) ((a, b, c, d)) MIRA_MAKE_SEQ_A_4

#define MIRA_INTERNAL_PRINT_TOKEN(TOKEN)	#TOKEN

#define MIRA_DO_NOTHING(Sequence,StartId)

#define MIRA_INTERNAL_NAMESPACE_FOLD(r,data,t) \
	t::data

#define MIRA_INTERNAL_CONC_NAMESPACE_IMPL(Sequence, StartId) \
	BOOST_PP_SEQ_FOLD_RIGHT(MIRA_INTERNAL_NAMESPACE_FOLD,,BOOST_PP_SEQ_REST_N(StartId,Sequence))

#define MIRA_INTERNAL_CONC_NAMESPACE(Sequence,StartId) \
	/* check if namespace size is greater zero and delay evaluation */ \
	BOOST_PP_IF( BOOST_PP_SEQ_SIZE( BOOST_PP_SEQ_REST_N( StartId, Sequence) ), \
		/* length of namespace is greater zero */ \
		MIRA_INTERNAL_CONC_NAMESPACE_IMPL \
		/* we do not have a namespace -> do nothing */ \
		, MIRA_DO_NOTHING )(Sequence, StartId)

#define MIRA_INTERNAL_TEMPLATE_CLASS_REGISTER( r, Args, TmplSpec) \
	/* Args(0) - contains the Class itself */ \
	/* Args(1) - number of parents k */ \
	/* Args(2) - number of template parameters */ \
	/* Args(3) - address n of first namespace token in sequence Args */ \
	/* Args(4 -- 4+k) - parents */ \
	/* Args(n -- end) - namespace */ \
	MIRA_REGISTRAR( MIRA_PPARAM( mira::FactoryRegisterClass< \
		/* the folding process adds the namespace, if the namesapce sequence*/ \
		/* length is greater than zero (with delayed evaluation) */ \
		BOOST_PP_IF( BOOST_PP_SEQ_SIZE( BOOST_PP_SEQ_REST_N( \
				BOOST_PP_SEQ_ELEM(3,Args),Args) ), \
		/* length of namespace is greater zero */ \
		MIRA_INTERNAL_CONC_NAMESPACE_IMPL \
		/* we do not have a namespace -> do nothing */ \
		, MIRA_DO_NOTHING )( Args, BOOST_PP_SEQ_ELEM(3,Args) )  \
		BOOST_PP_SEQ_ELEM(0,Args)<BOOST_PP_TUPLE_REM_CTOR( BOOST_PP_SEQ_ELEM(2,Args), TmplSpec)>, \
		BOOST_PP_SEQ_ENUM(BOOST_PP_SEQ_SUBSEQ(Args,4,BOOST_PP_SEQ_ELEM(1,Args)))> ) ) \


#define MIRA_INTERNAL_TEMPLATE_CLASS_STATIC( r, Args, TmplSpec) \
	/* Args(0) contains the Class itself */ \
	/* Args(1) contains the number of template parameters */ \
	template<> mira::TClass<\
		BOOST_PP_SEQ_ELEM(0,Args)<BOOST_PP_TUPLE_REM_CTOR(BOOST_PP_SEQ_ELEM(1,Args), TmplSpec)> >& \
		BOOST_PP_SEQ_ELEM(0,Args)<BOOST_PP_TUPLE_REM_CTOR(BOOST_PP_SEQ_ELEM(1,Args), TmplSpec)>::_staticClassObject( \
											std::string const& identifier, \
											std::string const& name, \
											bool libLoaded) { \
		static mira::TemplateClass<BOOST_PP_SEQ_ELEM(0,Args)<BOOST_PP_TUPLE_REM_CTOR(BOOST_PP_SEQ_ELEM(1,Args), TmplSpec)> > \
			sClass(identifier, name, libLoaded); \
		return sClass; \
	} \


#define MIRA_TMPL_SPEC_CONCAT(r, data, i, elem) "," BOOST_PP_STRINGIZE(elem)

#define TAIL_NAME( Seq ) \
	BOOST_PP_SEQ_FOR_EACH_I( MIRA_TMPL_SPEC_CONCAT, , Seq)

#define MIRA_INTERNAL_TEMPLATE_IDENTIFIER( r, Args, TmplSpec) \
	/* Args(0) contains the Class itself */ \
	/* Args(1) contains the number of template parameters */ \
	/* Args(2-n) contains the namespace */ \
	template<> \
	/* the folding process adds the namespace */ \
	std::string const& TemplateClass<MIRA_INTERNAL_CONC_NAMESPACE(Args,2) \
		BOOST_PP_SEQ_ELEM(0,Args)<BOOST_PP_TUPLE_REM_CTOR(BOOST_PP_SEQ_ELEM(1,Args), TmplSpec)> >::getIdentifier() const { \
		/* we should use a combination of the generic and the specialized */ \
		/* here since the generic string should include the namespace.*/ \
		static std::string tIdent = replaceTemplateGeneric( \
			TClass<MIRA_INTERNAL_CONC_NAMESPACE(Args,2) \
				BOOST_PP_SEQ_ELEM(0,Args)<BOOST_PP_TUPLE_REM_CTOR(BOOST_PP_SEQ_ELEM(1,Args), TmplSpec)> >::getIdentifier(), \
				"<" \
				BOOST_PP_STRINGIZE(BOOST_PP_SEQ_HEAD(BOOST_PP_TUPLE_TO_SEQ(BOOST_PP_SEQ_ELEM(1,Args),TmplSpec))) \
				/* use for_each_i here since recursion of fore_each is not allowed */ \
				BOOST_PP_SEQ_FOR_EACH_I( MIRA_TMPL_SPEC_CONCAT, , BOOST_PP_SEQ_TAIL(BOOST_PP_TUPLE_TO_SEQ(BOOST_PP_SEQ_ELEM(1,Args), TmplSpec))) \
				">" ); \
		return tIdent; \
	} \
	\
	template<> \
	/* the folding process adds the namespace */ \
	std::string const& TemplateClass<MIRA_INTERNAL_CONC_NAMESPACE(Args,2) \
		BOOST_PP_SEQ_ELEM(0,Args)<BOOST_PP_TUPLE_REM_CTOR(BOOST_PP_SEQ_ELEM(1,Args), TmplSpec)> >::getName() const { \
		static std::string tName = mira_stripNameFromIdentifier( \
				getIdentifier() ); \
		return tName; \
	} \

///@endcond

/**
 * @brief Use this macro to register your template class at the ClassFactory.
 * You should use this function if your template has only one template parameter.
 * Please note that the syntax is different from the regular MIRA_CLASS_REGISTER.
 * Therefore, you have to pass the Class (without any template arguments and its
 * Parents as well as the Namespace of your class (given as boost preprocessor
 * sequence) and the template specialization parameters.
 * E.g. if your class is defined as mira::subspace::MyClass<T> : public mira::Base1,
 * public mira::subspace::Base2 and you like to register your class for T=int and
 * T=char, the macro should look something like the following:
 * MIRA_TEMPLATE_CLASS_REGISTER((mira)(subspace),MyClass,(int)(char),mira::Base1,mira::subspace::Base2)
 */
#define MIRA_TEMPLATE_CLASS_REGISTER( Namespace, Class, TmplSpec, ... ) \
		MIRA_TEMPLATE_CLASS_REGISTER_IMPL( Namespace, Class, 1, MIRA_MAKE_SEQ(1,TmplSpec), __VA_ARGS__ )

/**
 * @brief Use this macro to register your template class at the ClassFactory.
 * You have to use this function if your template has more than one template parameter.
 * Please note that the syntax is different from the regular MIRA_CLASS_REGISTER.
 * Therefore, you have to pass the Class (without any template arguments and its
 * Parents as well as the Namespace of your class (given as boost preprocessor
 * sequence) and the template specialization parameters.
 * E.g. if your class is defined as mira::subspace::MyClass<T1,T2> : public mira::Base1,
 * public mira::subspace::Base2 and you like to register your class for (T1=int,T2=char) and
 * (T1=char,T2=uint), the macro should look something like the following:
 * MIRA_VARTEMPLATE_CLASS_REGISTER((mira)(subspace),MyClass,2,(int,char)(char,uint),mira::Base1,mira::subspace::Base2)
 */
#define MIRA_VARTEMPLATE_CLASS_REGISTER( Namespace, Class, NumTmplPrm, TmplSpec, ... ) \
		MIRA_TEMPLATE_CLASS_REGISTER_IMPL( Namespace, Class, NumTmplPrm, MIRA_MAKE_SEQ(NumTmplPrm,TmplSpec), __VA_ARGS__ )

 ///@cond INTERNAL

#define MIRA_TMPL_SIG_FROM_SEQ( NumTmplPrm, SeqId, Sequence )\
	BOOST_PP_TUPLE_REM_CTOR( NumTmplPrm, BOOST_PP_SEQ_ELEM(SeqId,Sequence) )

#define MIRA_TEMPLATE_CLASS_REGISTER_IMPL( Namespace, Class, NumTmplPrm, TmplSpec, ... ) \
	/* generate error message if macro is not used in global namespace */ \
	template <> int \
	________________________________PLEASE_USE_THE__MIRA_TEMPLATE_CLASS_REGISTER__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________\
	<MIRA_INTERNAL_CONC_NAMESPACE(Namespace,0)Class \
	<MIRA_TMPL_SIG_FROM_SEQ(NumTmplPrm,0,TmplSpec)> >::FOR_CLASS() { return 0; } \
	/* the following lines build up a sequence for the internal register macro*/\
	/* structure: (Class)(Number of Parents)(Address of first namespace token */\
	/* in the sequence)(Parents as sequence)(Namespace as sequence) */ \
	BOOST_PP_SEQ_FOR_EACH(MIRA_INTERNAL_TEMPLATE_CLASS_REGISTER, \
		(Class) \
		(MIRA_VA_NUM_ARGS(__VA_ARGS__))(NumTmplPrm)(BOOST_PP_ADD(MIRA_VA_NUM_ARGS(__VA_ARGS__),4))\
			BOOST_PP_TUPLE_TO_SEQ(MIRA_VA_NUM_ARGS(__VA_ARGS__),(__VA_ARGS__)) \
			Namespace,TmplSpec)  \
	\
	/* we can only specialize the the template in the corresponding namespace */\
	/* check if length of namespace is greater than zero */ \
	BOOST_PP_IF( BOOST_PP_SEQ_SIZE( Namespace ), \
 	/* length of namespace is greater zero */ \
	BOOST_PP_SEQ_FOR_EACH(MIRA_INTERNAL_OPEN_NAMESPACE,~,Namespace) \
	/* length of namespace is zero -> do nothing */ \
	, ) \
		/* we have to combine all parameters in a boost pp sequence in order */ \
		/* to pass them to the internal macro */ \
		/* Register the class at the factory */ \
		BOOST_PP_SEQ_FOR_EACH(MIRA_INTERNAL_TEMPLATE_CLASS_STATIC, \
			(Class)(NumTmplPrm), TmplSpec) \
	/* check if length of namespace is greater than zero */ \
	BOOST_PP_IF( BOOST_PP_SEQ_SIZE( Namespace ), \
	/* length of namespace is greater zero */ \
	BOOST_PP_SEQ_FOR_EACH(MIRA_INTERNAL_CLOSE_NAMESPACE,~,Namespace) \
	/* length of namespace is zero -> do nothing */ \
	, ) \
	\
	namespace mira { \
		/* Specialize the functions to obtain the template identifiers */ \
		BOOST_PP_SEQ_FOR_EACH(MIRA_INTERNAL_TEMPLATE_IDENTIFIER, \
			(Class)(NumTmplPrm)Namespace,TmplSpec) \
	}

///@endcond

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_FACTORYMACROS_H_ */

