/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Factory.h
 *    $Header file containing base classes to enable class creation using a
 *    class factory$
 *
 * @author Ronny Stricker, Erik Einhorn
 * @date   2010/01/20
 */

#ifndef _MIRA_FACTORY_H_
#define _MIRA_FACTORY_H_

#include <map>
#include <cstdarg>

#ifndef Q_MOC_RUN
#include <boost/type_traits/is_abstract.hpp>
#include <boost/mpl/eval_if.hpp>
#include <boost/assign.hpp>
#include <boost/preprocessor/seq.hpp>
#include <boost/preprocessor/tuple.hpp>
#endif

#include <utils/Singleton.h>
#include <utils/PParam.h>
#include <utils/Registrar.h>

#include <thread/Thread.h>

#include <factory/FactoryMacros.h>
#include <factory/Class.h>
#include <factory/TClass.h>
#include <factory/TemplateClass.h>
#include <factory/Object.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

// *****************************************************************************
// *** ClassFactory
// *****************************************************************************

/**
 * @brief What should i say, the class factory.
 * The class factory is capable to construct instances of registered classes
 * using the factory macros. Class objects can be accessed using the identifier 
 * or the meta information of the class.
 * @ingroup ClassFactoryModule
 */
class MIRA_BASE_EXPORT ClassFactory : public LazySingleton<ClassFactory>
{
	friend class ClassProxy;
	friend class LightFactoryMutexGetter;
	friend class VacantClass;
public:

	ClassFactory() { mRoot.mIdentifier = "mira::Object"; }

	~ClassFactory();

	/**
	 * @brief Create new instance of the class defined by class identifier.
	 * The created class is casted to the class type defined by CLASS.
	 * @throw XBadCast If dynamic cast to desired class type fails
	 * @throw XInvalidParameter If the given identifier is unknown
	 * @throw XLogical If the class creation fails.
	 */
	template<typename CLASS>
	static CLASS* newInstance( std::string const& classIdentifier );

	/**
	 * @brief Create new instance of the class defined by class identifier.
	 * Additional constructor parameters can be passed using the va_list.
	 * The created class is casted to the class type defined by CLASS.
	 * @throw XBadCast If dynamic cast to desired class type fails
	 * @throw XInvalidParameter If the given identifier is unknown
	 * @throw XLogical If the class creation fails.
	 */
	template<typename CLASS>
	static CLASS* newInstance( std::string const& classIdentifier,
	                           int paramCount, ... );

	template<typename CLASS>
	static CLASS* newInstance( std::string const& classIdentifier,
	                           int paramCount, std::va_list list );

	/**
	 * @brief Return true if a class with the desired identifier is registered.
	 */
	static bool isClassRegistered( std::string const& classIdentifier )
	{
		return instance().mRoot.isClassRegistered( classIdentifier );
	}

	/**
	 * @brief Return the Class object for the desired Class.
	 * @see Class::getClassByIdentifier()
	 */
	static ClassProxy getClassByIdentifier(std::string const& classIdentifier )
	{
		return instance().mRoot.getClassByIdentifier( classIdentifier );
	}

	/**
	 * @brief Return list of Class objects matching the meta criterion.
	 * @see Class::getClassByMeta(std::string const& metaKey,
	 * std::string const& metaValue )
	 */
	static std::vector<ClassProxy> getClassByMeta(std::string const& metaKey,
	                                              std::string const& metaValue )
	{
		return instance().mRoot.getClassByMeta( metaKey, metaValue );
	}

	/**
	 * @brief Return list of Class objects returning true for the given
	 * comparison function.
	 * @see Class::getClassByMeta( T funcPtr, std::vector<ClassProxy>& pResult )
	 */
	template <class T>
	static std::vector<ClassProxy> getClassByMeta( T funcPtr )
	{
		return instance().mRoot.getClassByMeta( funcPtr );
	}

	/**
	 * @brief Return list of registered classes.
	 */
	static std::map<std::string, ClassProxy > getDerivedClasses()
	{
		return instance().mRoot.getDerivedClasses();
	}

	/**
	 * @brief Register Class.
	 * You normally don't have to call this function. Use the Macros instead.
	 * @throws XLogical If class identifier is already in use
	 */
	static void registerClass( boost::shared_ptr<Class> iClass );

	/**
	 * @brief Register Class.
	 * You normally don't have to call this function. Use the Macros instead.
	 * @throws XLogical If class identifier is already in use
	 * @throws XLogical If base class and class do have the same identifiers.
	 */
	static void registerClass( boost::shared_ptr<Class> iClass,
			boost::shared_ptr<Class> baseClass );
	
	/**
	 * @brief Unregister Class.
	 * Remove the class from the list of known classes and reduce the class tree.
	 */
	static void unregisterClass( Class* iClass );

	/**
	 * @brief Register Parent - Child relationship.
	 * You normally don't have to call this function. Use the Macros instead.
	 * @throws XInvalidParameter if one ore more identifiers are unknown
	 */
	static void postRegisterBaseClasses( std::string const& iClass,
			std::vector< std::string> const& parents );

	/**
	 * @brief Propagate all children to indirect parents and add meta information
	 * of parent classes to children.
	 * You normally don't have to call this function. Use the Macros instead.
	 */
	static void finalizePostRegister();

protected:
	/**
	 * @brief Auxiliary function to propagate child classes to indirect parents.
	 */
	void propagateChild( ClassProxy& child, Class& parent );

	/**
	 * @brief Internal function to register a class.
	 * Used by registerClass() with and without parent.
	 */
	bool internalClassRegister( boost::shared_ptr<Class> iClass );
	
private:
	TClass<Object> mRoot;	/**< The almighty root class, the fountain, the
						genesis class */
	boost::recursive_mutex mThreadMutex;	/**< assure thread save access */
	std::map<std::string, boost::shared_ptr<Class> > mClasses;
};

///@cond INTERNAL

// *****************************************************************************
// *** FactoryRegisterClass
// *****************************************************************************

struct FactoryNullDeleter
{
	void operator()(void const *) const
	{ }
};

template<typename TClassP, typename Base>
struct FactoryRegisterClassHelper
{
	static void invoke() {
		boost::shared_ptr<TClass<TClassP> > tClass( &(TClassP::_CLASS()) , FactoryNullDeleter() );
		boost::shared_ptr<Class> tClassPtr = boost::dynamic_pointer_cast<Class>( tClass );
		boost::shared_ptr<TClass<Base> > tBase( &(Base::_CLASS()), FactoryNullDeleter() );
		boost::shared_ptr<Class> tBasePtr = boost::dynamic_pointer_cast<Class>( tBase );
		mira::ClassFactory::instance().registerClass(
				tClassPtr, tBasePtr );
	}
};

template<typename TClassP>
struct FactoryRegisterClassHelper<TClassP, mira::Object>
{
	static void invoke() {
		boost::shared_ptr<TClass<TClassP> > tClass( &(TClassP::_CLASS()), FactoryNullDeleter() );
		boost::shared_ptr<Class> tClassPtr = boost::dynamic_pointer_cast<Class>( tClass );
		mira::ClassFactory::instance().registerClass( tClassPtr );
	}
};

template<typename Class>
struct FactoryRegisterClassHelper<Class, bool>
{
	static void invoke() {}
};

/**
 * @brief This class simply registers the given class at the factory.
 * Used within the MIRA_CLASS_REGISTER macro.
 */
template<typename Class,
         typename Base0,
         typename Base1=bool, typename Base2=bool,
         typename Base3=bool, typename Base4=bool>
class FactoryRegisterClass
{
public:
	FactoryRegisterClass()
	{
		FactoryRegisterClassHelper<Class,Base0>::invoke();
		FactoryRegisterClassHelper<Class,Base1>::invoke();
		FactoryRegisterClassHelper<Class,Base2>::invoke();
		FactoryRegisterClassHelper<Class,Base3>::invoke();
		FactoryRegisterClassHelper<Class,Base4>::invoke();
	}
};

///@endcond

// *****************************************************************************
// *** Implementation of ClassFactory
// *****************************************************************************

// this looks a bit like code duplication but we cannot use the newInstance
// function of mRoot since mRoot is of type TClass. The implementation of the
// template newInstance() version for TClass is quite useless -> therefore
// we need this function...
template<typename CLASS>
inline CLASS* mira::ClassFactory::newInstance( std::string const& classIdentifier )
{
	Object* tBase = instance().mRoot.newInstance( classIdentifier );
	return mira_factoryDynamicCast<CLASS>( tBase );
}

template<typename CLASS>
inline CLASS* mira::ClassFactory::newInstance( std::string const& classIdentifier,
                                               int paramCount, ... )
{
	std::va_list ap;
	va_start(ap, paramCount);
	Object* tObject = instance().mRoot.newVAInstance( classIdentifier,
	                                                paramCount, ap );
	return mira_factoryDynamicCast<CLASS>( tObject );
}

template<typename CLASS>
inline CLASS* mira::ClassFactory::newInstance( std::string const& classIdentifier,
	                           int paramCount, std::va_list list )
{
	Object* tObject = instance().mRoot.newVAInstance( classIdentifier,
	                                                paramCount, list );
	return mira_factoryDynamicCast<CLASS>( tObject );
}


// *****************************************************************************
// *** Implementation of PseudoClass from Object.h
// *****************************************************************************

inline Object* PseudoClass::newInstance(std::string const& pChildIdentifier ) const
{
	return ClassFactory::newInstance<Object>(pChildIdentifier);
}

template <class CLASS>
inline CLASS* PseudoClass::newInstance(std::string const& pChildIdentifier ) const
{
	return ClassFactory::newInstance<CLASS>(pChildIdentifier);
}

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* FACTORY_H_ */
