/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeChannelInfo.h
 *    Informations about a channel in a tape.
 *
 * @author Tim Langner
 * @date   2012/01/01
 */

#ifndef _MIRA_TAPECHANNELINFO_H_
#define _MIRA_TAPECHANNELINFO_H_

#include <map>

#include <serialization/adapters/Qt/QColor>

#include <fw/TapeVisitor.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class TapeDataRenderer;

class TapeChannelMessage
{
public:
	TapeChannelMessage(bool inMemory = false)
	{
		if (inMemory)
		{
			mData.reset(Buffer<uint8>());
			mFrameID.reset("");
			mSequenceID.reset(0);
			mCompressed.reset(false);
		}
	}

	TapeChannelMessage(TapeVisitor::iterator& it) :
		mIterator(it)
	{}

	const Buffer<uint8>& getData()
	{
		if (mData)
			return *mData;
		return mIterator->data;
	}

	void setData(const Buffer<uint8>& data)
	{
		mData.reset(data);
	}

	bool getCompressed()
	{
		if (mCompressed)
			return *mCompressed;
		return mIterator->compressed;
	}

	void setCompressed(bool compress)
	{
		mCompressed.reset(compress);
	}

	const std::string& getFrameID()
	{
		if (mFrameID)
			return *mFrameID;
		return mIterator->frameID;
	}

	void setFrameID(const std::string& frameID)
	{
		mFrameID.reset(frameID);
	}

	uint32 getSequenceID()
	{
		if (mSequenceID)
			return *mSequenceID;
		return mIterator->sequenceID;
	}

	void setSequenceID(uint32 sequenceID)
	{
		mSequenceID.reset(sequenceID);
	}

protected:

	TapeVisitor::iterator mIterator;
	boost::optional<Buffer<uint8>> mData;
	boost::optional<std::string> mFrameID;
	boost::optional<uint32> mSequenceID;
	boost::optional<bool> mCompressed;
};

class TapeChannelInfo
{
public:
	typedef std::map<int64, TapeChannelMessage> DataMap;

	TapeChannelInfo() :
		selected(false)
	{}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Color", color, "Color for this channel");
		r.property("Renderer", renderer, "Renderer settings");
	}

	std::string name;
	std::string type;
	TypeMetaPtr meta;
	MetaTypeDatabase metadb;
	bool selected;
	QColor color;
	DataMap data;
	boost::shared_ptr<TapeDataRenderer> renderer;
};

typedef std::map<std::string, TapeChannelInfo> TapeChannelInfoMap;

///////////////////////////////////////////////////////////////////////////////

}

#endif 
