/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file InstallOverviewDialog.C
 *    GUI for install overview.
 *
 * @author Ronny Stricker
 * @date   2011/10/06
 */

#include "gui/dialogs/InstallOverviewDialog.h"

#include <QLayout>
#include <QTableWidget>
#include <QTableWidgetItem>
#include <QLabel>
#include <QPushButton>
#include <QFileDialog>
#include <QHeaderView>
#include <QCheckBox>

#include "core/Package.h"
#include "core/Repository.h"
#include "core/Tools.h"
#include "gui/IconProvider.h"

using namespace std;

namespace mira {

MiraTblItem::MiraTblItem( Package const* package, Database::Action const& action,
		Database* database, QWidget* parent ) : QWidget( parent ), mPackage(package)
{
	QHBoxLayout* layout = new QHBoxLayout( this );
	layout->setMargin(4);

	//////////////////////////////
	// first column - Check box
	QVBoxLayout* vLayout = new QVBoxLayout();

	mChecked = new QCheckBox( this );
	mChecked->setCheckState( Qt::Checked );

	vLayout->addWidget( mChecked );

	QSpacerItem* spacer = new QSpacerItem(10,10,QSizePolicy::Fixed, QSizePolicy::MinimumExpanding);
	vLayout->addSpacerItem( spacer );

	layout->addLayout( vLayout );

	//////////////////////////////
	// second column - Icon

	vLayout = new QVBoxLayout();

	QLabel* tLabel = new QLabel();
	tLabel->setPixmap( IconProvider::instance().getIcon( action ) );
	vLayout->addWidget( tLabel );

	spacer = new QSpacerItem(10,10,QSizePolicy::Fixed, QSizePolicy::MinimumExpanding);
	vLayout->addSpacerItem( spacer );

	layout->addLayout( vLayout );

	//////////////////////////////
	// third column - Name, Version, Source, Destination

	vLayout = new QVBoxLayout();

	QString tText = "<b>"+QString::fromStdString( package->mName )+"</b> "
			+ QString::fromStdString( package->mVersion.str() );

	if ( action & Database::INSTALL )
		tText += " (Repository: " + QString::fromStdString(
				database->getRepoNameFromUrl(package->mCurrentRepo) ) + ")";

	QLabel* label = new QLabel( tText );

	vLayout->addWidget( label );

	mPathWidget = new PathWidget( package->mName, package->mLocalPath,
			(action & Database::INSTALL) ? true : false, NULL );

	if (action == Database::INSTALL) {
		// check if a valid branch information can be obtained for item
		// otherwise the user have to review the path
		RepositoryPtr repo = database->getRepoFromUrl( package->mCurrentRepo );
		assert( repo );
		string matchedURL;
		repo->getMountDirForPackage( package->mRepoSubPath, &matchedURL );
		mPathWidget->setValid(!matchedURL.empty());
	} else
		mPathWidget->setValid(true);

	vLayout->addWidget( mPathWidget );

	connect( mPathWidget, SIGNAL( pathUpdated() ), this, SIGNAL( pathChanged() ) );
	connect( mChecked, SIGNAL( toggled(bool) ), this, SIGNAL( activationChanged(bool) ) );


	layout->addLayout( vLayout );

}

bool MiraTblItem::enabled() const
{
	return mChecked->checkState() == Qt::Checked;
}

bool MiraTblItem::valid() const
{
	return mPathWidget->mValid;
}

Path const& MiraTblItem::path() const
{
	return mPathWidget->mPath;
}

///////////////////////////////////////////////////////////////////////////////

PathWidget::PathWidget( std::string iName, Path iPath, bool changeable,
		QWidget* parent = NULL ) : QWidget( parent ), mPackageName( iName),
				mPath( iPath ), mValid( true ), mChangeable( changeable )
{
	QHBoxLayout* layout = new QHBoxLayout( this );
	layout->setMargin(0);

	box = new QComboBox( this );
	box->setMinimumHeight( 23 );

	foreach( Path const& path, PathProvider::miraPaths() ) {
		box->addItem( QString::fromStdString( path.string() ) );
	}

	label = new QLabel( this );
	label->setMinimumHeight( 20 );
	button = new QPushButton( "...", this );
	button->setMaximumWidth( 20 );
	button->setMinimumHeight( 20 );

	layout->addWidget( box );
	layout->addWidget( label );
	layout->addWidget( button );

	updateColor();

	connect( button, SIGNAL( clicked() ), this, SLOT( showPathDialog() ) );
	connect( box, SIGNAL( currentIndexChanged( int ) ), this, SLOT( dropPathSelected() ) );
	connect( this, SIGNAL( pathUpdated() ), this, SLOT( updateColor() ) );

	setBackgroundRole( QPalette::Window );
	setAutoFillBackground( true );

	setEditable( mChangeable );
}

PathWidget::~PathWidget()
{

}

void PathWidget::setEditable( bool editable )
{
	button->setEnabled( editable );
	box->setEnabled( editable );
	mChangeable = editable;
}

int getPathIndex( Path const& path )
{
	int tReturn = -1;
	PathVector paths = PathProvider::miraPaths();
	for ( uint32 i=0; i< paths.size(); ++i ) {
		if ( paths[i] == path )
			return i;
	}
	return tReturn;
}

void PathWidget::updateColor()
{
	if ( PathProvider::isMiraSubPath( mPath ) ) {
		Path miraPath = PathProvider::getAssociatedMiraPath( mPath );
		int pathIndex = getPathIndex( miraPath );
		if ( box->currentIndex() != pathIndex )
			box->setCurrentIndex( pathIndex );

		label->setText( QString::fromStdString(
				mPath.string().substr( miraPath.string().length() ) ) );
	}
	else {
		selectCustom();
	}

	if (!mChangeable)
		return;

	QPalette palette = this->palette();
	if ( PathProvider::isMiraSubPath( mPath ) && mValid ) {
		palette.setColor( QPalette::Window, QColor(222,255,222) );
		box->setStyleSheet("QComboBox { background-color: rgb(222,255,222);"
				"color: black; }");
	}
	else if ( mValid ) {
		palette.setColor( QPalette::Window, QColor(255,255,222) );
		box->setStyleSheet("QComboBox { background-color: rgb(255,255,222);"
				"color: black; }");
	}
	else {
		palette.setColor( QPalette::Window, QColor(255,222,222) );
		box->setStyleSheet("QComboBox { background-color: rgb(255,222,222);"
				"color: black; }");
	}
	setPalette( palette );
}

void PathWidget::dropPathSelected()
{
	mPath = box->currentText().toStdString() + label->text().toStdString();
	updateColor();
}

void PathWidget::selectCustom()
{
	// try to find combobox item for custom path first of all
	int index = box->findText( QString::fromStdString( mPath.string() ) );
	if ( index == -1 ) {
		// another custom item is already added -> replace
		if ( box->count() > int( PathProvider::miraPaths().size() ) ) {
			box->setItemText( box->count()-1, QString::fromStdString( mPath.string() ) );
		}
		else {
			// append item otherwise
			box->addItem( QString::fromStdString( mPath.string() ) );
		}
		box->setCurrentIndex( box->count() - 1 );
	}
	else
		box->setCurrentIndex( index );
	label->setText( "" );
}

void PathWidget::showPathDialog()
{
	QString tPath = QFileDialog::getExistingDirectory( this,
			"select destination",
			QString::fromStdString( mPath.string() ) );
	if ( !tPath.isEmpty() ) {
		mPath = tPath.toStdString();
		mValid = true;
		emit pathUpdated();
	}
}

///////////////////////////////////////////////////////////////////////////////

InstallOverviewDialog::InstallOverviewDialog(
		std::vector<Package*> const uninstallSequence,
		std::vector<Package*> const installSequence,
		Database* database, QWidget *parent)
	: QDialog( parent ), mDatabase( database )
{
	assert( database );

	Ui::InstallOverviewDialog::setupUi(this);

	// Prepare the preview table
	int row = 0;
	size_t totalCnt = uninstallSequence.size() + installSequence.size();
	overviewTable->setRowCount(totalCnt);
	overviewTable->setUpdatesEnabled( false );

	// First: Add all uninstall operations
	foreach(const Package* pkg, uninstallSequence) {
		MiraTblItem* tItem =
				new MiraTblItem( pkg, Database::UNINSTALL, database, overviewTable );
		connect( tItem, SIGNAL( pathChanged() ), this, SLOT( updateOKButton() ) );
		connect( tItem, SIGNAL( activationChanged(bool) ), this, SLOT( updateOKButton() ) );

		overviewTable->setCellWidget( row, 0, tItem );
		row++;
	}

	// Second: Add all install operations
	foreach(const Package* pkg, installSequence) {
		MiraTblItem* tItem =
				new MiraTblItem( pkg, Database::INSTALL, database, overviewTable );
		connect( tItem, SIGNAL( pathChanged() ), this, SLOT( updateOKButton() ) );
		connect( tItem, SIGNAL( activationChanged(bool) ), this, SLOT( updateOKButton() ) );
		overviewTable->setCellWidget( row, 0, tItem );
		row++;
	}

	// Finalize the table
	overviewTable->verticalHeader()->setResizeMode(QHeaderView::ResizeToContents);
	overviewTable->horizontalHeader()->setResizeMode(0,QHeaderView::Stretch);
	overviewTable->setUpdatesEnabled( true );
	updateOKButton();
}

InstallOverviewDialog::~InstallOverviewDialog()
{
}

void InstallOverviewDialog::accept()
{
	// write selected paths back to database
	for(int r = 0; r < overviewTable->rowCount(); r++) {
		QWidget* widget = overviewTable->cellWidget( r, 0 );
		MiraTblItem* tblItem = dynamic_cast<MiraTblItem*>( widget );
		assert( tblItem );

		// Find the associated "action plan" entry for this package.
		Database::ActionPlan::iterator iter =
				mDatabase->mActionPlan.find((Package*)tblItem->getPackage());
		if (iter == mDatabase->mActionPlan.end())
			continue;

		// Update the selected install path
		if (iter->second & Database::INSTALL) {
			iter->first->mLocalPath = tblItem->path();
		}

		// Removed inactive actions from "action plan"
		if ((!tblItem->enabled()) &&
			(iter->second & (Database::INSTALL | Database::UNINSTALL)))
		{
			mDatabase->mActionPlan.erase(iter);
		}
	}

	QDialog::accept();
}

void InstallOverviewDialog::updateOKButton()
{
	bool ok = true;

	for ( int row=0; row<overviewTable->rowCount();row++ ) {
		// check if item is active
		QWidget* widget = overviewTable->cellWidget( row, 0 );
		MiraTblItem* item = dynamic_cast<MiraTblItem*>( widget );
		assert( item );
		if ( item->enabled() ) {
			// check if path of item is valid
			if ( !item->valid() ) {
				ok = false;
				break;
			}
		}
	}
	buttonBox->button( QDialogButtonBox::Ok )->setEnabled( ok );
}

///////////////////////////////////////////////////////////////////////////////

} // namespace
