/*
 * Copyright (C) 2019 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GitlabRepository.C
 *
 * @author Thomas Bauer
 * @date   2019/11/21
 */

#include <boost/filesystem/operations.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/regex.hpp>

#include "core/GitlabRepository.h"
#include "core/PackageParser.h"
#include "app/GitlabAPI.h"
#include "core/Package.h"
#include <regex>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

GitlabRepository::GitlabRepository() : 
	Repository(), group_ids(), online_projects(),
	gitlab("https://gitlab.metralabs.com"), mGit(), tags_only(false) 
{
        credential.realm = "https://gitlab.metralabs.com";
}

GitlabRepository::GitlabRepository(const std::string& iName, const std::string& iUrl) : 
Repository(), group_ids(), online_projects(), gitlab(iUrl), mGit(), tags_only(false) {
	//debug switch, turn of in release
	//gitlab.checkSSLCerts(false);
	credential.realm = iUrl;
}

int GitlabRepository::findGroups(const std::string& path) {
	int count = 0; 

	if(gitlab.loggedIn()) {
		std::vector<gitlab::Group> allGroups;

		// Get all groups from Gitlab. Since there seems to be a maximum of
		// 64kByte in a request, we get the groups in pages of 100pcs.
		int page = 0;
		while(true) {
			auto result = gitlab.groups("", true, page);
			if (result.size() == 0)
				break;
			allGroups.insert(allGroups.end(), result.begin(), result.end());
			page++;
		}

		foreach(auto r, allGroups) {
			if(boost::starts_with(r.full_path, path)) {
				group_ids.insert(std::make_pair(r.full_path, r.id));
				count++;
			}
		}
	}
	return count;
}

bool GitlabRepository::login(bool force) {
	if((force or credential.user.empty() or credential.password.empty()) and prompt != NULL) {
		prompt->getLogin(credential);
	}

	return gitlab.loggedIn() or gitlab.login(credential.user, credential.password);
}

void GitlabRepository::deepExamine( std::vector<Package*>& oPackages,
                                    boost::function<void (std::string const&,int,int)> progressListener )
{
	if(not gitlab.loggedIn()) {
		// if we are not logged in we provide a login dialog
		login();

		// if we are still not logged in, something is wrong
		if(not gitlab.loggedIn())
			MIRA_THROW(XRuntime, "Failed to login on GitLab server. Check username and password!");
	}

	// we stuff the oPackages reference with all packages we find in _our_ Repository url
	// therefore we look at the path given by the url (e.g. metralabs/mira-pkg)
	auto searchPath = splitUrl(url)[UrlPath];

	if(boost::starts_with(searchPath, "/") and searchPath.length() > 1)
		searchPath = searchPath.substr(1);

	findGroups(searchPath+"/");

	int counter = 0;
	// Download infos for all projects
	foreach(auto id, group_ids) {
		try{
			auto t = gitlab.groupProjects(id.second);
			online_projects.insert(online_projects.end(), t.begin(), t.end());
			if(progressListener) {
				progressListener("Get Projects from group: " + id.first, ++counter, group_ids.size());
			}
		} catch (XHTTP404& ex) {
			MIRA_LOG(WARNING) << "GitLab: Got 404 for group with id=" << id.first << ": " << ex.what();
		}
	}

	// look for *.package and mountdir.xml files
	counter = 0;
	foreach(auto p, online_projects) {
		try {
			auto tPackage = createPackage(p);
			if(tPackage.size() > 0) {
				oPackages.insert(oPackages.end(), tPackage.begin(), tPackage.end());
			}

			if(progressListener) {
				progressListener("Project: " + p.name, ++counter, online_projects.size());
			}
		} catch (XHTTP404& ex) {
			MIRA_LOG(WARNING) << "GitLab: Got 404 while processing " << p.name << ": " << ex.what();
		} catch (Exception& ex) {
			MIRA_LOG(WARNING) << "GitLab: Error while processing " << p.name << ": " << ex.what();
		} catch (std::exception& ex) {
			MIRA_LOG(WARNING) << "GitLab: Error while processing " << p.name << ": " << ex.what();
		}
	}
}

std::vector<Package*> GitlabRepository::createPackage(const gitlab::Project& p) {
	std::vector<Package*> ret;
	std::vector<gitlab::Tag> tags;

	if(tags_only) { // we're only interessted in tags
		tags = gitlab.tags(p.id);
	} else { // otherwise we look for master
		gitlab::Tag t;
		t.name = "master";
		t.message = "";
		tags.push_back(t);
	}

	boost::regex version_regex("[0-9]+\\.[0-9]+\\.[0-9]+");

	foreach(auto tag, tags) {
		boost::smatch match;
		bool regexMatch = boost::regex_search(tag.name, match, version_regex);

		if(tag.name == "master" || regexMatch) {
			auto files = gitlab.listFiles(p.id, "", tag.name);
			foreach(auto f, files) {
				if(boost::ends_with(f.name, ".package") or boost::ends_with(f.name, ".extpackage")) {
					auto filecontent = gitlab.rawFile(p.id, f.name, tag.name);
					auto path = splitUrl(p.web_url)[UrlPath];
					auto path_prefix = splitUrl(url)[UrlPath];
					// check if the path of repo url is a prefix of web_url path and remove if so
					if(boost::starts_with(path, path_prefix)) {
						path = path.substr(path_prefix.length());
					}
					ret.push_back(indexWebPackage(path + ".git/" + f.name, this, &filecontent));
				} else if(f.name == "mountdir.xml") {
					// Always use the "mountdir.xml" from master to avoid problems
					// with invalid mountdir.xml in older tags.
					try {
						auto filecontent = gitlab.rawFile(p.id, f.name, "master");
						parseMountDirXml(p.web_url + ".git/" + f.name, filecontent);
					} catch(Exception& ex) {
						MIRA_LOG(WARNING) << "Failed to get mountdir for package " << p.name << " from 'master' at " << f.name << ": " << ex.what();
					}
				}
			}
		}
	}
	return ret;
}

std::string GitlabRepository::catFile( Url const& url ) {
	// functionallity can be achieved with GitlabAPI::rawFile();
	MIRA_THROW(XNotImplemented, "catFile: this function is not implemented and shall not be used.");
	return std::string();
}

bool GitlabRepository::isResponsibleFor( mira::Path const& localPath ) {
	// we can not be sure, that localPath is really a path w/out a filename
	auto real_path = localPath.string();
	if(boost::filesystem::is_regular_file(localPath)) {
		real_path = localPath.parent_path().string();
	}
	return mGit.isGitRepository(real_path);
}

mira::Path GitlabRepository::getRelativePathFor( mira::Path const& localPath ) {
	// we can not be sure, that localPath is really a path w/out a filename
	std::string ret = "";
	if(isResponsibleFor(localPath)) {
		auto real_path = localPath.string();
		if(is_regular_file(localPath)) {
			real_path = localPath.parent_path().string();
		}
		auto remote = mGit.remoteURL(real_path);
		if(boost::starts_with(remote, url)) {
			ret = remote.substr(url.length()) + "/" + localPath.filename().string();
		}
		MIRA_LOG(NOTICE) << "getRelativepathfor(): Relative path for " << localPath << " is " << ret;
	}
	return mira::Path(ret);
}

void GitlabRepository::install( Package const& package, mira::Path const& destPath ) {
	prepareCheckoutDirectory(destPath, package.mType & Package::SOURCE);

	// this is a special case, it cloud happen that a package contains to package files, so if the path with the package file
	// already exists, do nothing.
	if(boost::filesystem::exists(destPath) and boost::filesystem::exists(mira::Path(destPath.string() + "/" + package.mFileName))) {
		MIRA_LOG(NOTICE) << "Path " << destPath.string() << " and " << package.mFileName << " already exists.";
		return; 
	}

	// Ensure, that the remote URL ends with ".git". This is very important
	// for git client older than version 2.0. These clients are not able to
	// handle the redirect messages from GitLab, which redirects the remote
	// URLs from /path/to/package to /path/to/package.git.
	std::string cloneURL = url + package.mRepoSubPath;
	if (!boost::ends_with(cloneURL, ".git"))
		cloneURL += ".git";

	try {
		mGit.clone(cloneURL, destPath.string(), credential);
	} catch (XAccessViolation& e) {
		if(login(true)) {
			mGit.clone(cloneURL, destPath.string(), credential);
		} else {
			MIRA_RETHROW(e, "Could open login dialog.");
		}
	}
	if(tags_only) {
		if(not mGit.detach_head(destPath.string(), "refs/tags/" + package.mVersion.str())) {
			MIRA_THROW(XRuntime, "Could not checkout version " + package.mVersion.str() + ". \n Repository is on master. \n You might want to reindex.");
		} else {
			mGit.checkout_head(destPath.string(), true);
		}
	}
}

void GitlabRepository::uninstall( Package const& package ) {
	Path p(package.mLocalPath);
	if(boost::filesystem::exists(p) and isResponsibleFor(p)) {
		auto status = mGit.status(p.string());
		auto unpushed = mGit.needPush(p.string());
		if(status.size() == 0 and unpushed.size() == 0) {
			MIRA_LOG(NOTICE) << "Removing libs for " << package.mName;
			removeGeneratedLibraryFiles(package);
			MIRA_LOG(NOTICE) << "Removing " << package.mLocalPath;
			remove_all(p);
		} else if(prompt != NULL) {
			std::string pending_changes;
			std::string pending_push;
			if(status.size() > 0) {
				pending_changes = "The following changes are not commited:\n";
				foreach(const auto& p, status) {
					pending_changes += GitUtils::statusCode(p.second) + ": " + p.first + "\n";
				}
			}
			if(unpushed.size() > 0) {
				pending_push = "The following branches are not synced with origin:\n";
				for(auto it = unpushed.cbegin(); it != unpushed.cend(); it++) {
					pending_push += *it + "\n"; 
				}

			}
			prompt->showErrorMessage("Uninstall is blocked by the following issues :\n" + pending_changes + pending_push);
		}
	} else if(not isResponsibleFor(p)) {
		MIRA_LOG(WARNING) << "GitlabRepository is not responsible for " << p;
		MIRA_LOG(NOTICE) << "exists: " <<boost::filesystem::exists(p) << "is responsible for " << p << ": " << isResponsibleFor(p);
	} else {
		MIRA_LOG(WARNING) << p << " does not exist.";
	}
}

///////////////////////////////////////////////////////////////////////////////

} // namespace mira

MIRA_CLASS_SERIALIZATION(mira::GitlabRepository, mira::Repository);
