/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

#include <iostream>

#include <boost/algorithm/string.hpp>
#include <boost/program_options.hpp>
#include <boost/filesystem/operations.hpp>

#include <QTextCodec>

#include <platform/Process.h>
#include <platform/Console.h>
#include <utils/PathFinder.h>
#include <xml/XMLDom.h>
#include <error/LogTxtStreamSink.h>

#include "core/Database.h"
#include "core/Repository.h"
#include "core/MIRAPackage.h"
#include "app/MIRAPackageGUI.h"
#include "app/MIRAPackageApp.h"
#include "app/CmdLinePromptProvider.h"
#include "app/CmdLineOptions.h"
#include "platform/Environment.h"

using namespace mira;
using namespace std;
namespace po = boost::program_options;

typedef std::vector< std::vector<std::string> >  CredentialsVector;


po::options_description defineCommands()
{
	po::options_description tCmdOptions( "Allowed options" );
	tCmdOptions.add_options()
			("help", "Produce help message.")
			("debug", "Enable debug output generation.")
			("reindex,R", "Rebuild the index of all branches and repositories." )
			("clearrepos,C", "Clear the repository list." )
			("addrepo,A", po::value< vector< string > >()->multitoken(),
					"Adds another source repository location. Please provide "
					"repository name and url in succession.")
			("addurl", po::value< string >(),
					"Adds another repository. Please provide an URL to a "
					"repository information file (*.repo).")
			("list,L", po::value< string >()->implicit_value("all"),
					"Show the list of available packages. Optional arguments:\n"
					"  all - list all packages (available and installed)\n"
					"  installed - list installed packages only\n"
					"  available - show available packages only")
			("export", po::value< string >()->implicit_value("packages.mpe"),
					"Export package and repository configuration to the given file.")
			("import", po::value< string >()->implicit_value("packages.mpe"),
					"Import(add) package and repository configuration from the given file.")
			("details", po::value< string >(),
					"List the details of a given package.")
			("install,I", po::value< vector< string > >()->multitoken(),
					"Install the given packages (and dependent packages)." )
			("deinstall,D", po::value< vector< string > >()->multitoken(),
					"Uninstall the given packages. If AllPackages is given as "
					"package parameter, the program tries to delete all "
					"installed packages.")
			("mirapath,P", po::value<string>(),
					"Select the mirapath for checkout.")
			("deppath", po::value<string>(),
					"Select install path for required dependencies occuring "
					"during install (default: deppath=mirapath).")
			("auth,U",
					"request username and password just ONCE and use it "
					"for all calls to svn.")
			("credential", po::value<std::vector<std::string>>()->multitoken(),
					"Tripel [Repositroyname, Username, Password] "
					"Set username and password for the given repository. "
					"Given Bash Variablename for password. "
					"This options works only in CmdLineMode.")
			("noninteractive", "In commandline modus, disable userinput.");
	return tCmdOptions;
}

po::options_description hiddenCommands()
{
	po::options_description tCmdOptions( "Hidden options" );
	tCmdOptions.add_options()
			("hidden", "show all options (hidden included)")
			("dependencies", po::value< string >(),
					"list dependencies of given package.")
			("installSequence", po::value< vector< string > >()->multitoken(),
					"return install sequence for the given packages.");
	return tCmdOptions;
}

// add default repositories if database cannot be loaded
bool loadDatabase( MIRAPackage* core, const vector<string>& defaultRepos )
{
	try {
		core->load();
	} catch( Exception& ex ) {
		cout << ex.what() << endl;
		// we should add the default repositories since no package
		// database can be found
		if ( defaultRepos.size() > 0 ) {
			std::cout << "No mirapackage.xml file found. Adding default repositories."<<std::endl;
			foreach( string repoUrl, defaultRepos ) {
				RepositoryPtr repo = Repository::createFromURL(repoUrl);
				core->addRepository( repo );
			}
		}
		return false;
	}
	return true;
}

int main(int argc, char *argv[])
{
	apr_initialize();

	QTextCodec::setCodecForLocale(QTextCodec::codecForName("UTF-8"));
#if QT_VERSION <= 0x050000
	QTextCodec::setCodecForCStrings( QTextCodec::codecForName("UTF-8") );
#endif

	// add some default repositories to gather package information from if
	// no database xml file can be found
	std::vector<std::string> tDefaultRepos;
	//tDefaultRepos.push_back("ftp://mira-project.org/repos/MIRA-main/src/MIRA-main.repo");
	//tDefaultRepos.push_back("ftp://mira-project.org/repos/MetraLabs-public/src/MetraLabs-public.repo");


	// define command line options
	po::options_description tVisibleOptions = defineCommands();
	po::options_description tAllOptions = tVisibleOptions;
	tAllOptions.add( hiddenCommands() );

	// parse command line
	po::parsed_options parsed_options(NULL);
	po::variables_map tCmd;
	try {
		parsed_options = po::command_line_parser(argc, argv).options(tAllOptions).run();
		po::store(parsed_options, tCmd );
		po::notify( tCmd );
	}
	catch ( po::error& tException ) {
		cout << tException.what() << "\n\n";
		cout << tVisibleOptions << "\n";
		return 1;
	}

	// evaluate command line options
	if ( tCmd.count("help") ) {
		usage( tDefaultRepos );
		cout << tVisibleOptions << "\n";
		return 1;
	}

	if ( tCmd.count("hidden") ) {
		usage( tDefaultRepos );
		cout << tAllOptions << "\n";
		return 1;
	}

	MIRA_LOGGER.registerSink( LogTxtStreamSink( std::cout ) );
	if ( tCmd.count("debug") )
		MIRA_LOGGER.setSeverityLevel(NOTICE);
	else
		MIRA_LOGGER.setSeverityLevel(WARNING);
	
	int tAppReturn = 0;
	// start UI if no command line parameters (except 'debug') are given
	try {
		if (tCmd.size() == 0 || ( tCmd.size() == 1 && tCmd.count("debug"))) {
			bool execGUI = true;
			MIRAPackageApp app( argc, argv );
			try {
				checkForMIRAPath();
			}
			catch (Exception&) {
				// no MIRAPath found -> start wizard?
				if ( QMessageBox::question(NULL,"start wizard","cannot found MIRA_PATH"
						" variable, do you wish to start the mirawizard?",
						QMessageBox::Yes | QMessageBox::No) == QMessageBox::Yes ) {
					Process::createProcess( "mirawizard").wait();

				}
				else {
					execGUI = false;
				}
			}

			if ( execGUI ) {
				MIRAPackageGUI checkoutGUI;

				loadDatabase( &checkoutGUI, tDefaultRepos );

				checkoutGUI.checkForInstalled();
				checkoutGUI.mDB.buildTagIndex();
				checkoutGUI.show();
				tAppReturn = app.exec();
				checkoutGUI.save();
			}
		}
		else {

			// try to load the package database
			// if loading fails we should generate a new database from the default
			// repositories
			bool tForceReindex = false;

			QCoreApplication qtCoreApp(argc, argv);

			CmdLinePromptProvider provider;
			MIRAPackage checkoutCore;
			if(tCmd.count("noninteractive") == 0) {
				checkoutCore.mDB.setRepositoryPromptProvider( &provider );
			}
			vector<string> tEmptyList;
			if ( !loadDatabase( &checkoutCore,
					((tCmd.count("addrepo") > 0) ||
					 (tCmd.count("clearrepos") > 0) ||
					 (tCmd.count("addurl") > 0)) ? tEmptyList : tDefaultRepos ) ) {
				tForceReindex = true;
			}

			if(tCmd.count("credential") > 0) {
				CredentialsVector cv_listing;
				foreach(auto o, parsed_options.options) {
					if(o.string_key == "credential") {
						cv_listing.push_back(o.value);
					}
				}
				foreach(auto cred, cv_listing) {
					if(cred.size() == 3) {
						auto repoptr = checkoutCore.mDB.getRepoFromName(cred[0]);
						if(repoptr != NULL) {
							repoptr->credential.user = cred[1];
							try {
								repoptr->credential.password = resolveEnvironmentVariable(cred[2]);
							} catch (XInvalidConfig& e) {
								repoptr->credential.password = cred[2];
							}
						} else {
							MIRA_LOG(WARNING) << "No repository named " << cred[0];
						}
					} else {
						MIRA_LOG(WARNING) << "credential takes exactly 3 arguments.";
					}
				} 
			}

			if ( tCmd.count("clearrepos") ) {
				checkoutCore.clearRepositories();
				checkoutCore.save();
			}

			if( tCmd.count("addrepo") ) {
				vector<string> tRepo = tCmd["addrepo"].as< vector< string > >();
				if ( tRepo.size() != 2 ) {
					usage( tDefaultRepos );
					cout << tVisibleOptions << "\n";
					return 1;
				}

				checkoutCore.addRepository( tRepo[0], "", tRepo[1], "", 0, "autoDetect" );
				checkoutCore.save();
			}

			if ( tCmd.count("addurl") ) {
				string tURL = tCmd["addurl"].as< string >();
				RepositoryPtr repo = Repository::createFromURL(tURL);
				checkoutCore.addRepository(repo);
				checkoutCore.save();
			}

			if( tCmd.count("reindex") || tForceReindex ) {
				checkoutCore.reindexWebRepositories( false );
				checkoutCore.checkForInstalled();
				checkoutCore.save();
			}

			if( tCmd.count("mirapath") ) {
				string tPath = tCmd["mirapath"].as< string >();
				checkoutCore.mInstallRoot = tPath;
			}
			
			if ( tCmd.count("deppath") ) {
				checkoutCore.mDepInstallPath = tCmd["deppath"].as< string >();
				cout << "DepInstallpath: " << checkoutCore.mDepInstallPath << endl;
			}

			// check which packages are installed on the system
			checkoutCore.checkForInstalled();

			if ( tCmd.count("list") ) {
				string tCommand = tCmd["list"].as< string >();
				if ( !listPackages( tCommand, checkoutCore ) ) {
					usage( tDefaultRepos );
					cout << tVisibleOptions << "\n";
					return 1;
				}
			}

			if ( tCmd.count("export") ) {
				string tFile = tCmd["export"].as< string >();
				if ( !exportPackages( tFile, checkoutCore ) ) {
					usage( tDefaultRepos );
					cout << tVisibleOptions << "\n";
					return 1;
				}
			}

			if ( tCmd.count("import") ) {
				string tFile = tCmd["import"].as< string >();
				if ( !importPackages( tFile, checkoutCore ) ) {
					usage( tDefaultRepos );
					cout << tVisibleOptions << "\n";
					return 1;
				}
			}

			if ( tCmd.count("details") ) {
				string tPackage = tCmd["details"].as< string >();
				showPackageDetails( tPackage, checkoutCore );
			}

			if ( tCmd.count("install") ) {
				vector<string> tPackages = tCmd["install"].as< vector< string > >();
				installPackages( tPackages, checkoutCore );
			}

			if ( tCmd.count("deinstall") ) {
				vector<string> tPackages = tCmd["deinstall"].as< vector< string > >();
				deinstallPackages( tPackages, checkoutCore );
			}

			// process hidden options
			if ( tCmd.count("dependencies") ) {
				string tPackageName = tCmd["dependencies"].as< string >();
				listDependencies( tPackageName, checkoutCore );
			}

			if ( tCmd.count("installSequence") ) {
				vector<string> tPackageNames = tCmd["installSequence"].as< vector< string > >();
				listInstallSequence( tPackageNames, checkoutCore );
			}
		}
	}
	catch ( exception& ex)
	{
		cout << "Oh no!" << endl;
		cout << ex.what() << endl;
		tAppReturn = -1;
	}

	apr_terminate();
	return tAppReturn;
}
