/*
 * Copyright (C) 2019 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GitlabAPI.C
 *
 * @author Thomas Bauer
 * @date   2019/11/22
 */

#include <app/GitlabAPI.h>

#include <error/Exceptions.h>
#include <error/LoggingCore.h>

#include <serialization/JSONSerializer.h>
#include <serialization/Serialization.h>

#include <utils/Singleton.h>
#include <utils/MakeString.h>

#include <boost/algorithm/string.hpp>

namespace mira { namespace gitlab {

///////////////////////////////////////////////////////////////////////////////

class LibCurlSingleton : public LazySingleton<LibCurlSingleton> {
public:
	LibCurlSingleton() {
		auto e = curl_global_init(CURL_GLOBAL_DEFAULT);
		if(e != 0) {
			MIRA_THROW(XRuntime, "GitlabAPI: Could not init libcurl library!");
		}
		MIRA_LOG(NOTICE) << "Initialized libcurl version: " << curl_version();
	}

	virtual ~LibCurlSingleton() {
		curl_global_cleanup();
	}
};

///////////////////////////////////////////////////////////////////////////////

GitlabAPI::GitlabAPI(const std::string& host) : instance(host), loggedin(false), curl_handle(NULL), access_token() {
	LibCurlSingleton::instance();

	curl_handle = curl_easy_init();
	if(curl_handle == NULL) {
		MIRA_THROW(XRuntime, "GitlabAPI: Could not access curl library!");
	}

	auto tmp = splitUrl(instance);

	if(not tmp[UrlHost].empty()) {
		if(tmp[UrlScheme] == "http") {
			MIRA_LOG(NOTICE) << "Change http to http_s_!";
		}
		instance = "https://" + tmp[UrlHost];
	} else {
		MIRA_THROW(XIO, "No host given: " << instance);
	}
}

GitlabAPI::GitlabAPI(const GitlabAPI& other) : instance(other.instance), loggedin(other.loggedin), curl_handle(NULL), access_token(other.access_token) {
	curl_handle = curl_easy_init();
	if(curl_handle == NULL) {
		MIRA_THROW(XRuntime, "GitlabAPI: Could not access curl library!");
	}
}

GitlabAPI& GitlabAPI::operator=(const GitlabAPI& rhs) {
	instance = rhs.instance;
	loggedin = rhs.loggedin;
	access_token = rhs.access_token;

	curl_handle = curl_easy_init();
	if(curl_handle == NULL) {
		MIRA_THROW(XRuntime, "GitlabAPI: Could not access curl library!");
	}
	return *this;
}


GitlabAPI::~GitlabAPI() {
	if(curl_handle != NULL) {
		curl_easy_cleanup(curl_handle);
	}
}

bool GitlabAPI::login(const std::string& username, const std::string& password) {
	if(not loggedin) {
		// https://docs.gitlab.com/ee/api/oauth2.html#resource-owner-password-credentials-flow
		// https://tools.ietf.org/html/rfc6749#section-4.3

		// We need to replace some special characters.
		std::string esc_pwd = std::string(curl_easy_escape(curl_handle, password.c_str(), password.length()));

		auto resp = post("/oauth/token", "grant_type=password&username=" + username + "&password=" + esc_pwd);

		JSONValue auth_response;
		json::read(resp, auth_response);
		JSONDeserializer dserial(auth_response);

		gitlab::Error e;

		try {
			dserial.deserialize(e);
			MIRA_THROW(XRuntime, e.error << e.error_description);
		} catch(XRuntime) {
			try {
				dserial.deserialize(access_token);
				loggedin = true;
			} catch (XRuntime) {
				MIRA_LOG(WARNING) << "Could not login. No access token returned from server " << instance;
			}
		}
	}
	return loggedin;
}

bool GitlabAPI::provideApplicationToken(const std::string& token) {
	if(not loggedin) {
		access_token.access_token=token;
		access_token.token_type="bearer";
		loggedin=true;
	}
	return loggedin;
}

std::vector<Project> GitlabAPI::projects(unsigned int per_page) {
	std::vector<Project> ret;
	if(loggedIn()) {
		curl_slist* headers = NULL;
		long http_status = 0;

		headers = curl_slist_append(headers, access_token.authorizationHeader().c_str());

		auto resp = request(apiVersion() + "/projects?per_page=", http_status, headers);

		JSONValue proj_resp;

		json::read(resp, proj_resp);

		JSONDeserializer ds(proj_resp);
		ds.deserialize(ret);
	}
	return ret;
}

std::vector<Project> GitlabAPI::groupProjects(unsigned int groupID, unsigned int per_page, bool subgroups) {
	std::vector<Project> ret;
	if(loggedIn()) {
		curl_slist* headers = NULL;
		long http_status = 0;
		headers = curl_slist_append(headers, access_token.authorizationHeader().c_str());

		auto resp = request(MakeString() << apiVersion() << "/groups/" << groupID << "/projects?per_page=" << per_page, http_status, headers);

		JSONValue proj_resp;

		json::read(resp, proj_resp);

		JSONDeserializer ds(proj_resp);
		try {
			ds.deserialize(ret);
		} catch (XRuntime) {
			Error e;
			ds.deserialize(e);
			if(boost::starts_with(e.error, "404")) {
				MIRA_THROW(XHTTP404, e.error);
			} else {
				MIRA_THROW(XNetworkError, e.error << ": " << e.error_description);
			}
		}
	}
	return ret;
}

std::vector<Namespace> GitlabAPI::namespaces(const std::string& search) {
	std::vector<Namespace> ret;
	if(loggedIn()) {
		curl_slist* headers = NULL;
		long http_status = 0; 
		headers = curl_slist_append(headers, access_token.authorizationHeader().c_str());

		std::string request_path = "/namespaces";

		if(not search.empty()) {
			request_path += "?search=" + search;
		}

		auto resp = request(apiVersion() + request_path, http_status, headers);

		JSONValue ns_resp;

		json::read(resp, ns_resp);

		JSONDeserializer ds(ns_resp);
		ds.deserialize(ret);
	}
	return ret;
}

std::vector<Group> GitlabAPI::groups(const std::string& search, bool all, unsigned int page ) {
	std::vector<Group> ret;
	if(loggedIn()) {
		curl_slist* headers = NULL;
		long http_status = 0; 
		headers = curl_slist_append(headers, access_token.authorizationHeader().c_str());

		std::string request_path = "/groups";

		if(all) {
			request_path += MakeString() << "?all_available=true&per_page=100&page="  << page;
		} else if(not search.empty()) {
			request_path += MakeString() << "?search=" + search + "&per_page=100&page=" << page;
		}

		auto resp = request(apiVersion() + request_path, http_status, headers);

		JSONValue groups_resp;

		json::read(resp, groups_resp);

		JSONDeserializer ds(groups_resp);
		ds.deserialize(ret);
	}
	return ret;
}

std::vector<Tag> GitlabAPI::tags(unsigned int project_id, const std::string& search) {
	std::vector<Tag> ret;
	if(loggedIn()) {
		curl_slist* headers = NULL;
		long http_status = 0; 
		headers = curl_slist_append(headers, access_token.authorizationHeader().c_str());

		std::string request_path = MakeString() << "/projects/" << project_id << "/repository/tags";

		if(not search.empty()) {
			request_path += "?search=" + search; 
		}

		auto resp = request(apiVersion() + request_path, http_status, headers);

		JSONValue tags_resp;

		json::read(resp, tags_resp);

		JSONDeserializer ds(tags_resp);
		ds.deserialize(ret);
	}
	return ret;
}

std::string GitlabAPI::rawFile(unsigned int project_id, std::string path, const std::string& ref) {
	curl_slist* headers = NULL;
	long http_status = 0;
	headers = curl_slist_append(headers, access_token.authorizationHeader().c_str());

	// url encode path
	if(boost::starts_with(path, "/") and path.length() > 1) {
		path = path.substr(1);
	}
	if(path.empty()) {
		MIRA_THROW(XIO, "No path to file.");
	}

	std::string request_path = MakeString() << "/projects/" << project_id << "/repository/files/"
		<< encodePath(path) << "/raw?ref=" << ref;

	auto resp = request(apiVersion() + request_path, http_status, headers);

	if(http_status == 404) {
		MIRA_THROW(XHTTP404, "File not found.");
	} else if(http_status > 200) {
		std::string msg = MakeString() << "Got HTTP " << http_status;
		MIRA_THROW(XNetworkError, msg.c_str());
	}

	return resp;
}

std::vector<Tree> GitlabAPI::listFiles(unsigned int project_id, const std::string& path, const std::string& ref) {
	std::vector<Tree> ret;
	curl_slist* headers = NULL;
	long http_status = 0;
	headers = curl_slist_append(headers, access_token.authorizationHeader().c_str());

	std::string request_path = MakeString() << "/projects/"  << project_id << "/repository/tree" << encodePath(path) << "?ref=" << ref;

	auto resp = request(apiVersion() + request_path, http_status, headers);

	JSONValue tree_resp;
	json::read(resp, tree_resp);

	JSONDeserializer ds(tree_resp);

	try {
		ds.deserialize(ret);
	} catch(XRuntime) {
		try {
			Error e;
			ds.deserialize(e);
			MIRA_THROW(XNetworkError, e.error << " " << e.error_description);
		} catch (XRuntime& e) {
			Message m;
			ds.deserialize(m);
			if(boost::starts_with(m.message, "404")) {
				MIRA_THROW(XHTTP404, m.message);
			} else {
				MIRA_RETHROW(e, "at GitlabAPI::listFiles()");
			}
		}
	}

	return ret;
}

std::string GitlabAPI::request(const std::string& path, long& http_status, const curl_slist* header) {

	std::string ret = "";
	ret.reserve(CURL_MAX_HTTP_HEADER);

	if(curl_handle != NULL) {
		auto url = instance + path;
		long _http_status = 0;
		curl_easy_setopt(curl_handle, CURLOPT_HTTPHEADER, header);
		curl_easy_setopt(curl_handle, CURLOPT_URL, (instance + path).c_str());
		curl_easy_setopt(curl_handle, CURLOPT_HTTPGET, 1);
		curl_easy_getinfo(curl_handle, CURLINFO_RESPONSE_CODE, &http_status);

		// activate cookies for this handle
		//curl_easy_setopt(curl_handle, CURLOPT_COOKIEFILE, "");

		// save returned data
		curl_easy_setopt(curl_handle, CURLOPT_WRITEFUNCTION, write_callback);
		curl_easy_setopt(curl_handle, CURLOPT_WRITEDATA, static_cast<void*>(&ret));

		auto result = curl_easy_perform(curl_handle);
		http_status = _http_status;

		if(not result == CURLE_OK) {
			std::string error_buffer(curl_easy_strerror(result));
			MIRA_LOG(ERROR) << "Request error: " << error_buffer;
			// todo rework in release 
			MIRA_THROW(XNetworkError, "Request Error: " << error_buffer);
		}

		// keep going with https://curl.haxx.se/libcurl/c/cookie_interface.html

	} else {
		MIRA_LOG(ERROR) << "Curl handle is not initialized!";
	}

	return ret;
}

std::string GitlabAPI::post(const std::string& path, const std::string& data,  const curl_slist* header) {
	std::string ret; 

	if(curl_handle != NULL) {
		curl_easy_setopt(curl_handle, CURLOPT_URL, (instance + path).c_str());
		curl_easy_setopt(curl_handle, CURLOPT_HTTPPOST, 1);
		curl_easy_setopt(curl_handle, CURLOPT_POSTFIELDS, data.c_str());

		// save received data
		curl_easy_setopt(curl_handle, CURLOPT_WRITEFUNCTION, write_callback);
		curl_easy_setopt(curl_handle, CURLOPT_WRITEDATA, static_cast<void*>(&ret));

		auto result = curl_easy_perform(curl_handle);
		if(result != CURLE_OK) {
			MIRA_LOG(ERROR) << "Could not perform post request: " << curl_easy_strerror(result);
		}
	} else {
		MIRA_LOG(ERROR) << "Curl handle is not initialized!";
	}
	return ret;
}

// *** DEBUG OPTION ***
void GitlabAPI::setVerbose(bool verbose) {
	if(verbose) {
		curl_easy_setopt(curl_handle, CURLOPT_VERBOSE, 1);
	} else {
		curl_easy_setopt(curl_handle, CURLOPT_VERBOSE, 0);
	}
}

void GitlabAPI::checkSSLCerts(bool check) {
	if(check) {
		curl_easy_setopt(curl_handle, CURLOPT_SSL_VERIFYPEER, 1);
	} else {
		curl_easy_setopt(curl_handle, CURLOPT_SSL_VERIFYPEER, 0);
	}
}

//std::string GitlabAPI::post(...) {}

size_t write_callback(char* ptr, size_t size, size_t nmemb, void* userdata) {
	auto buffer = static_cast<std::string*>(userdata);

	// if buffer.capacity is insufficent, this will reserve additional space
	buffer->reserve(nmemb);

	if(ptr != NULL) {
		buffer->append(ptr, nmemb);
	}

	return nmemb;
}

///////////////////////////////////////////////////////////////////////////////

} // namespace gitlab

} // namespace mira

MIRA_CLASS_SERIALIZATION(mira::XNetworkError, mira::XRuntime);
MIRA_CLASS_SERIALIZATION(mira::XHTTP404,      mira::XNetworkError);

