/*
 * Copyright (C) 2019 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GitUtils.C
 *
 * @author Thomas Bauer
 * @date   2019/12/02
 */

#include <boost/filesystem/operations.hpp>

#include <error/Exceptions.h>
#include <error/Logging.h>
#include <utils/UUID.h>

#include "app/GitUtils.h"

namespace mira {

///////////////////////////////////////////////////////////////////////////////

GitUtils::GitUtils() {
	git_libgit2_init();
}

GitUtils::~GitUtils() {
	git_libgit2_shutdown();
}

bool GitUtils::check_lg2(int err) {
	if(err != 0) {
		const git_error* e = giterr_last();
		if(e != NULL) {
			if(e->klass == GIT_ERROR_NET) {
				MIRA_THROW(XAccessViolation, "Authentication Error: " << e->message);
			} else {
				MIRA_LOG(ERROR) << "libgit2 error: " << e->message;
				MIRA_THROW(XRuntime, "libgit2 error: " << e->message << " " << e->klass);
			}
		} else {
			MIRA_THROW(XRuntime, "An error occured but could not retrieve libgit2 error message.");
		}
		return false;
	}
	return true;
}

///////////////////////////////////////////////////////////////////////////////
//							Callback Functions
///////////////////////////////////////////////////////////////////////////////

static int credential_cb(git_cred **out, const char* url, const char* username_from_url, unsigned int allowed_types, void* payload) {
	int error = 1;
	if(payload != NULL) {
		auto c = static_cast<const Credential*>(payload);
		if(allowed_types & GIT_CREDTYPE_USERPASS_PLAINTEXT and not c->user.empty()) {
			error = git_cred_userpass_plaintext_new(out, c->user.c_str(), c->password.c_str());
		}
	}
	return error;
}

static int status_cb(const char* path, unsigned int status_flags, void* payload) {
	reinterpret_cast<std::map<std::string, unsigned int>*>(payload)->insert({std::string(path), status_flags});
	return 0;
}

///////////////////////////////////////////////////////////////////////////////

bool GitUtils::clone(const std::string& httpUrl, const std::string& path, const Credential& cred) {
	bool ret = false;

	Credential c(cred);

	if(not(httpUrl.empty() or path.empty())) {
		git_repository* repo = NULL; 
		git_clone_options opts;
		check_lg2(git_clone_init_options(&opts, GIT_CLONE_OPTIONS_VERSION));
		if(not cred.password.empty()) {
			opts.fetch_opts.callbacks.credentials = &credential_cb;
			opts.fetch_opts.callbacks.payload = static_cast<void*>(&c);
		}

		if(check_lg2(git_clone(&repo, httpUrl.c_str(), path.c_str(), &opts))) {
			ret = true;
		}
		git_repository_free(repo);
	}
	return ret;
}

/** @brief checkout a cretain reference.
 */
bool GitUtils::detach_head(const std::string& path, const std::string& ref) {
	bool ret = false;
	git_repository* repo = NULL;
	if(check_lg2(git_repository_open(&repo, path.c_str()))) {
		git_object* gobj = NULL;
		// get id of e.g. ref/tags/1.0.1
		if( check_lg2(git_revparse_single(&gobj, repo, ref.c_str()))
		and check_lg2(git_repository_set_head_detached(repo, git_object_id(gobj)))) {
			ret = true;
		}
		if(gobj != NULL) {
			git_object_free(gobj);
		}
	}
	if(repo != NULL) {
		git_repository_free(repo);
	}
	return ret;
}

bool GitUtils::checkout_head(const std::string& path, bool force) {
	bool ret = false;
	git_repository* repo = NULL;
	if(check_lg2(git_repository_open(&repo, path.c_str()))) {
		git_checkout_options opts;
		git_checkout_init_options(&opts, GIT_CHECKOUT_OPTIONS_VERSION);
		if(force) {
			opts.checkout_strategy = GIT_CHECKOUT_FORCE;
		} else {
			opts.checkout_strategy = GIT_CHECKOUT_SAFE;
		}
		check_lg2(git_checkout_head(repo, &opts));
	}
	if(repo != NULL) {
		git_repository_free(repo);
	}
	return ret;
}

bool GitUtils::isGitRepository(const std::string& path) {
	// checks wheter path can be opened
	return (git_repository_open_ext(NULL, path.c_str(), GIT_REPOSITORY_OPEN_NO_SEARCH, NULL) == 0);
}

bool GitUtils::checkURL(const std::string& URL) {
	git_repository* r = NULL;
	git_remote* remote = NULL;
	bool canConnect = false;

	UUID uuid = boost::uuids::random_generator()();
	std::string tmp_path = "/tmp/" + toString(uuid);

	if(git_repository_init(&r, tmp_path.c_str(), 1) == GIT_OK && 
		git_remote_create(&remote, r, "test", URL.c_str()) == GIT_OK && 
		git_remote_connect(remote, GIT_DIRECTION_FETCH, NULL, NULL, NULL) == GIT_OK) {
		canConnect = true; 
	} else {
		const git_error* e = git_error_last();
		MIRA_LOG(DEBUG) << "LibGit2 Error Class: " << e->klass << " : " << e->message;
	}
	if(remote != NULL) {
		git_remote_free(remote);
	}
	if(r != NULL) {
		git_repository_free(r);
	}

	try {
		boost::filesystem::remove_all(boost::filesystem::path(tmp_path));
	} catch (...) {}

	return canConnect;
}

std::map<std::string, unsigned int> GitUtils::status(const std::string& path) {
	std::map<std::string, unsigned int> ret;
	git_repository* repo = NULL;

	if(check_lg2(git_repository_open(&repo, path.c_str()))) {
		git_status_foreach(repo, status_cb, &ret);
	}

	if(repo != NULL) {
		git_repository_free(repo);
	}
	return ret;
}

std::map<std::string, std::string> GitUtils::getBranchMap(const std::string& path) {
	std::map<std::string, std::string> ret;
	git_repository* repo = NULL;
	git_branch_iterator* iter = NULL;
	git_branch_t branch_t;

	if(check_lg2(git_repository_open(&repo, path.c_str()))) {
		git_branch_iterator_new(&iter, repo, GIT_BRANCH_LOCAL);
		git_reference* gref = NULL;
		git_reference* gref_up = NULL;

		while(git_branch_next(&gref, &branch_t, iter) == 0) {
			if(branch_t == GIT_BRANCH_LOCAL && gref) {
				const char* local_branch_name;
				const char* remote_branch_name;

				git_branch_name(&local_branch_name, gref);
				git_branch_upstream(&gref_up, gref);
				
				if(gref_up != NULL) {
					git_branch_name(&remote_branch_name, gref_up);
					ret.insert({std::string(local_branch_name), std::string(remote_branch_name)});
				} else {
					ret.insert({std::string(local_branch_name), std::string()});
				}
			}
		}

		if(iter != NULL) {
			git_branch_iterator_free(iter);
		}
		if(gref != NULL) {
			git_reference_free(gref);
		}
		if(gref_up != NULL) {
			git_reference_free(gref_up);
		}
	}

	if(repo != NULL) {
		git_repository_free(repo);
	}

	return ret;
}

std::vector<std::string> GitUtils::needPush(const std::string& path, const std::string& remote) {
	std::vector<std::string> ret;
	git_repository* repo = NULL;
	git_revwalk* walker = NULL;
	auto branches = getBranchMap(path);

	if(check_lg2(git_repository_open(&repo, path.c_str()))) {
		git_revwalk_new(&walker, repo);
		foreach(auto b, branches) {
			if(b.second != "") {
				std::string rbranch("refs/remotes/" + remote + "/" + b.second);
				std::string lbranch("refs/heads/" + b.first);
				if(git_revwalk_push_ref(walker, rbranch.c_str()) == GIT_OK &&
					git_revwalk_hide_ref(walker, lbranch.c_str()) == GIT_OK) { 
					git_oid id;

					int count = 0;
					while (!git_revwalk_next(&id, walker)) {
						++count;
					}
					if(count != 0) {
						ret.push_back(b.first);
					}
				}
			} else {
				ret.push_back(b.first);
			}
		}
	}

	if(walker != NULL) {
		git_revwalk_free(walker);
	}
	if(repo != NULL) {
		git_repository_free(repo);
	}
	return ret;
}

std::string GitUtils::remoteURL(const std::string& path, const std::string& remotename) {
	std::string ret;
	git_repository* repo = NULL;
	if(check_lg2(git_repository_open(&repo, path.c_str()))) {
		git_remote* remote = NULL;
		if(check_lg2(git_remote_lookup(&remote, repo, remotename.c_str()))) {
			ret = git_remote_url(remote);
		}
		if(remote != NULL) {
			git_remote_free(remote);
		}
	}
	if(repo != NULL) {
		git_repository_free(repo);
	}
	return ret;
}

std::string GitUtils::statusCode(const unsigned int sc) {
	std::string ret = "??";
	if(sc == GIT_STATUS_WT_NEW) {
		ret = "new file";
	} else if(sc == GIT_STATUS_WT_MODIFIED) {
		ret = "modified";
	} else if(sc == GIT_STATUS_WT_DELETED) {
		ret = "deleted";
	} else if(sc == GIT_STATUS_WT_RENAMED) {
		ret = "renamed";
	} else if(sc == GIT_STATUS_WT_TYPECHANGE) {
		ret = "typechange";
	}
	return ret;
}

///////////////////////////////////////////////////////////////////////////////

} // namespace mira
