/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelView.C
 *    Implementation of ChannelView.
 *
 * @author Erik Einhorn
 * @date   2010/12/01
 */

#include <boost/algorithm/string.hpp>

#include <utils/StringAlgorithms.h>
#include <serialization/Serialization.h>
#include <fw/Framework.h>

#include <views/GUIMimeTypes.h>
#include <views/ChannelView.h>

#include <widgets/CollapsibleTreeHeader.h>
#include <widgets/TreeViewFilter.h>

#include <QMimeData>
#include <QVBoxLayout>
#include <QMouseEvent>
#include <QDrag>
#include <QApplication>
#include <QMenu>
#include <QHeaderView>
#include <QPushButton>
#include <QToolTip>

///////////////////////////////////////////////////////////////////////////////
MIRA_CLASS_SERIALIZATION(mira::ChannelView, mira::ViewPart);

///////////////////////////////////////////////////////////////////////////////
namespace mira {

///////////////////////////////////////////////////////////////////////////////

class ChannelTreeWidget : public QTreeWidget
{


public:

	class Item : public QTreeWidgetItem
	{
		void init(const std::string& name)
		{
			setText(0, name.c_str());
			setFlags((Qt::ItemFlags)(Qt::ItemIsSelectable | Qt::ItemIsDragEnabled | Qt::ItemIsEnabled));
			for(int i=1; i<=8; ++i)
				setText(i, "");
		}

	public:
		Item(QTreeWidget* parent, const std::string& name, bool ns) :
			QTreeWidgetItem(parent), isNamespace(ns)
		{
			init(name);
		}

		Item(QTreeWidgetItem* parent, const std::string& name, bool ns) :
			QTreeWidgetItem(parent), isNamespace(ns)
		{
			init(name);
		}

		virtual bool operator< (const QTreeWidgetItem& o) const
		{
			const Item& other = dynamic_cast<const Item&>(o);
			if(isNamespace != other.isNamespace)
				return isNamespace;
			return QTreeWidgetItem::operator<(o);
		}

		bool isNamespace;
		std::string channelID;
	};

public:

	ChannelTreeWidget(QWidget* parent) : QTreeWidget(parent)
	{
		setColumnCount(9);
		QStringList l;
		l << "Name" << "Type" << "Typed" << "TypeMeta" << "Updates" << "Last Update" << "Pub" << "Sub" << "Slots";
		setHeaderLabels(l);
		QHeaderView* header = new CollapsibleTreeHeader(Qt::Horizontal,this);
		setHeader( header );
		dragPixmap = QIcon(":/icons/Channel.png").pixmap(32, 32);
	}

	void mousePressEvent(QMouseEvent *event)
	{
		if (event->button() == Qt::LeftButton)
			dragStartPosition = event->pos();

		QTreeWidget::mousePressEvent(event);
	}

	void mouseMoveEvent(QMouseEvent *event)
	{
		if (!(event->buttons() & Qt::LeftButton))
			return;
		if ((event->pos() - dragStartPosition).manhattanLength() < QApplication::startDragDistance())
			return;

		// get item that the user is dragging
		Item* item = dynamic_cast<Item*>(itemAt(dragStartPosition));
		if(item==NULL)
			return;

		if(item->channelID.empty()) // item does not contain a channel id
			return;

		QDrag *drag = new QDrag(this);
		QMimeData *mimeData = new QMimeData;

		mimeData->setData(MIRA_MIME_CHANNEL, QByteArray(item->channelID.c_str()) );
		drag->setMimeData(mimeData);
		drag->setPixmap(dragPixmap);

		//Qt::DropAction dropAction = drag->start();
		Qt::DropAction dropAction = drag->exec();
	}

	bool viewportEvent (QEvent* event)
	{
		if(event->type()==QEvent::ToolTip)
			return updateTooltip(true);

		return QTreeWidget::viewportEvent(event);
	}

	bool updateTooltip(bool showIfHidden=false)
	{
		if(!underMouse())
			return false;

		if(!QToolTip::isVisible() && !showIfHidden)
			return false;

		QPoint pos = QCursor::pos();
		QPoint p = viewport()->mapFromGlobal(pos);
		Item* item = dynamic_cast<Item*>(itemAt(p));
		if(!item)
			return false;

		QToolTip::showText(pos, item->toolTip(0), this);
		return true;
	}

public:

	QPoint dragStartPosition;
	QPixmap dragPixmap;

};

///////////////////////////////////////////////////////////////////////////////

struct ChannelView::UI : public QWidget
{
	UI(QWidget* parent) :
		QWidget(parent)
	{
		QVBoxLayout* verticalLayout = new QVBoxLayout(this);
		verticalLayout->setContentsMargins(0,3,0,0);
		verticalLayout->setSpacing(0);

		tree = new ChannelTreeWidget(this);
		tree->header()->setStretchLastSection(false);
		tree->setColumnCount(9);
		tree->setColumnWidth(0,200);
		tree->setColumnWidth(1,120);
		tree->setColumnWidth(2,120);
		tree->setColumnWidth(3,20);
		tree->setColumnWidth(4,90);
		tree->setColumnWidth(5,50);
		tree->setColumnWidth(6,30);
		tree->setColumnWidth(7,30);
		tree->setColumnWidth(8,40);
		tree->header()->setResizeMode(0,QHeaderView::Stretch);
		for (int i=1; i<=8; ++i)
			tree->header()->setResizeMode(i,QHeaderView::Interactive);

		mFilter = new TreeViewFilter(tree,this);
		verticalLayout->addWidget(mFilter);
		verticalLayout->addWidget(tree);

		tree->header()->installEventFilter(this);
		mResizeModeTimerId = 0;
	}

	bool eventFilter(QObject* obj, QEvent* ev)
	{
		if (obj == tree->header() && ev->type() == QEvent::Resize) {
			setFrontStretch(true);
		}
		return false;
	}

	void setFrontStretch(bool stretch)
	{
		if (mResizeModeTimerId)
		{
			killTimer(mResizeModeTimerId);
			mResizeModeTimerId = 0;
		}

		QHeaderView* hv = tree->header();

		if (stretch) {
			hv->setStretchLastSection(false);                // order is important, do not allow more than 1 stretching
			hv->setResizeMode(0, QHeaderView::Stretch);      // column at a time to keep the current width while changing

			mResizeModeTimerId = startTimer(1000);           // reset to "standard" interaction scheme (stretch last column) after 1s
		} else {
			hv->setResizeMode(0, QHeaderView::Interactive);
			hv->setStretchLastSection(true);
		}
	}

	void timerEvent(QTimerEvent *event)
	{
		if (event->timerId() != mResizeModeTimerId)
			return;

		setFrontStretch(false);
	}

	ChannelTreeWidget* tree;

	TreeViewFilter* mFilter;

	int mResizeModeTimerId;
};

ChannelView::ChannelView() :
	mUI(NULL)
{
	mShowType = new QAction("Show data type", this);
	mShowType->setCheckable(true);
	connect(mShowType, SIGNAL(toggled(bool)), this, SLOT(updateChannelTree()));

	mShowIsTyped = new QAction("Show if channel is typed", this);
	mShowIsTyped->setCheckable(true);
	connect(mShowIsTyped, SIGNAL(toggled(bool)), this, SLOT(updateChannelTree()));

	mShowTypeMeta = new QAction("Show type meta", this);
	mShowTypeMeta->setCheckable(true);
	connect(mShowTypeMeta, SIGNAL(toggled(bool)), this, SLOT(updateChannelTree()));

	mShowUpdateCount = new QAction("Show update count", this);
	mShowUpdateCount->setCheckable(true);
	connect(mShowUpdateCount, SIGNAL(toggled(bool)), this, SLOT(updateChannelTree()));

	mShowLastUpdate = new QAction("Show last update time", this);
	mShowLastUpdate->setCheckable(true);
	connect(mShowLastUpdate, SIGNAL(toggled(bool)), this, SLOT(updateChannelTree()));

	mShowPublishersSubscribersCount = new QAction("Show subscriber/publisher count", this);
	mShowPublishersSubscribersCount->setCheckable(true);
	connect(mShowPublishersSubscribersCount, SIGNAL(toggled(bool)), this, SLOT(updateChannelTree()));

	mShowSlotCount =  new QAction("Show slot count", this);
	mShowSlotCount->setCheckable(true);
	connect(mShowSlotCount, SIGNAL(toggled(bool)), this, SLOT(updateChannelTree()));

	mShowInternalPublishersSubscribers = new QAction("Include internal subscribers/publishers in tooltip", this);
	mShowInternalPublishersSubscribers->setCheckable(true);
	connect(mShowInternalPublishersSubscribers, SIGNAL(toggled(bool)), this, SLOT(updateChannelTree()));

	mHideFilterBarAct = new QAction("Hide filter bar automatically",this);
	mHideFilterBarAct->setCheckable(true);
}

QWidget* ChannelView::createPartControl()
{
	mUI=new UI(this);

	QMenu* menu = getViewMenu();
	menu->addAction(mShowType);
	menu->addAction(mShowIsTyped);
	menu->addAction(mShowTypeMeta);
	menu->addAction(mShowUpdateCount);
	menu->addAction(mShowLastUpdate);
	menu->addAction(mShowPublishersSubscribersCount);
	menu->addAction(mShowSlotCount);
	menu->addAction(mShowInternalPublishersSubscribers);
	menu->addAction(mHideFilterBarAct);

	QAction* showFilterAction = new QAction(this);
	showFilterAction->setShortcutContext(Qt::WidgetWithChildrenShortcut);
	showFilterAction->setShortcut(QString("Ctrl+F"));
	addAction(showFilterAction);
	connect(showFilterAction, SIGNAL(triggered(bool)), mUI->mFilter, SLOT(showFilter()));

	connect(mHideFilterBarAct, SIGNAL(toggled(bool)), mUI->mFilter, SLOT(setAutoHide(bool)));
	mUI->mFilter->setAutoHide(mHideFilterBarAct->isChecked());

	mUpdateTimerId = startTimer(2000);
	updateChannelTree();

	return mUI;
}

void ChannelView::timerEvent(QTimerEvent *e)
{
	if(e->timerId()==mUpdateTimerId) {
		updateChannelTree();
	}

	ViewPart::timerEvent(e);
}

void ChannelView::updateChannelTree()
{
	if(mUI==NULL)
		return;

	std::set<std::string> channels = MIRA_FW.getChannelManager().getChannels();

	Time localNow = Time::now().toLocal();

	// find new channels and insert them
	foreach(const std::string& channel, channels)
	{
		if(mChannelItems.count(channel)==0) // new channel?
			insertChannel(channel);
		else
		{
			const std::string typeName = MIRA_FW.getChannelManager().getTypename(channel);

			mChannelItems[channel].first->setText(1, QString("%1").arg(typeName.c_str()));
			mChannelItems[channel].first->setCheckState(2, MIRA_FW.getChannelManager().getTypeId(channel)>=0?Qt::Checked:Qt::Unchecked);

			TypeMetaPtr meta = MIRA_FW.getChannelManager().getTypeMeta(channel);
			mChannelItems[channel].first->setText(3, QString("%1").arg(meta ? meta->toString().c_str() : ""));

			uint64 nrChanges = MIRA_FW.getChannelManager().getNrOfDataChanges(channel);
			uint64 diff = nrChanges - mChannelItems[channel].second;
			mChannelItems[channel].first->setText(4, QString("%1 (%2 Hz)")
					.arg(nrChanges)
					.arg(diff/2.0f));
			mChannelItems[channel].second = nrChanges;
			Time last = MIRA_FW.getChannelManager().getLastSlotTime(channel);
			if (!last.isValid())
				mChannelItems[channel].first->setText(5, "no data");
			else {
				last = last.toLocal();
				if(last.date().day() == localNow.date().day()) // if same day, then show time only
					mChannelItems[channel].first->setText(5, toString(last.time_of_day()).c_str());
				else
					mChannelItems[channel].first->setText(5, toString(last).c_str());
			}

			uint32 nrPublishers = MIRA_FW.getChannelManager().getNrPublishers(channel);
			uint32 nrSubscribers = MIRA_FW.getChannelManager().getNrSubscribers(channel);

			mChannelItems[channel].first->setText(6, QString("%1").arg(nrPublishers));
			mChannelItems[channel].first->setText(7, QString("%1").arg(nrSubscribers));
			mChannelItems[channel].first->setText(8,
					QString("%1").arg(MIRA_FW.getChannelManager().getNrOfSlots(channel)));

			QString tooltipText = QString::fromStdString(channel) + " (" + QString::fromStdString(typeName) + ")\n" +
					QString("Updates: %1 (%2 Hz)").arg(nrChanges).arg(diff/2.0f);

			if (nrPublishers > 0) {
				std::set<std::string> publishers
					= MIRA_FW.getChannelManager().getPublishers(channel, mShowInternalPublishersSubscribers->isChecked());
				tooltipText += QString("\nPublishers:");
				foreach (const std::string& s, publishers)
					tooltipText += QString("\n  ") + QString::fromStdString(s);
			}
			if (nrSubscribers > 0) {
				std::set<std::string> subscribers
					= MIRA_FW.getChannelManager().getSubscribers(channel, mShowInternalPublishersSubscribers->isChecked());
				tooltipText += QString("\nSubscribers:");
				foreach (const std::string& s, subscribers)
					tooltipText += QString("\n  ") + QString::fromStdString(s);
			}
			
			mChannelItems[channel].first->setToolTip(0, tooltipText);
		}
	}

	mUI->tree->updateTooltip();

	bool columnsChanged = mUI->tree->isColumnHidden(1) == mShowType->isChecked();
	columnsChanged |= mUI->tree->isColumnHidden(2) == mShowIsTyped->isChecked();
	columnsChanged |= mUI->tree->isColumnHidden(3) == mShowTypeMeta->isChecked();
	columnsChanged |= mUI->tree->isColumnHidden(4) == mShowUpdateCount->isChecked();
	columnsChanged |= mUI->tree->isColumnHidden(5) == mShowLastUpdate->isChecked();
	columnsChanged |= mUI->tree->isColumnHidden(6) == mShowPublishersSubscribersCount->isChecked();
	columnsChanged |= mUI->tree->isColumnHidden(8) == mShowSlotCount->isChecked();

	if (columnsChanged) {
		mUI->setFrontStretch(true);
	}

	mUI->tree->setColumnHidden(1, !mShowType->isChecked());
	mUI->tree->setColumnHidden(2, !mShowIsTyped->isChecked());
	mUI->tree->setColumnHidden(3, !mShowTypeMeta->isChecked());
	mUI->tree->setColumnHidden(4, !mShowUpdateCount->isChecked());
	mUI->tree->setColumnHidden(5, !mShowLastUpdate->isChecked());
	mUI->tree->setColumnHidden(6, !mShowPublishersSubscribersCount->isChecked());
	mUI->tree->setColumnHidden(7, !mShowPublishersSubscribersCount->isChecked());
	mUI->tree->setColumnHidden(8, !mShowSlotCount->isChecked());

	mUI->tree->sortItems(0,Qt::AscendingOrder);
}

void ChannelView::insertChannel(const std::string& channelID)
{
	std::vector<std::string> s;
	boost::split(s, channelID, boost::is_from_range('/','/'));

	if(s.size()<=1)
		return;

	// search for existing toplevel item where we need to add the
	// channel
	QTreeWidgetItem* root = NULL;
	for(int i=0; i<mUI->tree->topLevelItemCount(); ++i)
	{
		QTreeWidgetItem* it= mUI->tree->topLevelItem(i);
		if(it->text(0).toLocal8Bit().data() == s[1]) {
			root = it;
			break;
		}
	}

	if(root==NULL) { // nothing was found -> create new top level item
		root = new ChannelTreeWidget::Item(mUI->tree,s[1],true);
		mUI->tree->insertTopLevelItem(0, root);
		root->setExpanded(true);
		// set italic font per default
		QFont font = root->font(0);
		font.setItalic(true);
		root->setFont(0,font);
		root->setIcon(0, QIcon(":/icons/Namespace.ico"));
	}


	// now search the sub-items recursively
	QTreeWidgetItem* parent = root;
	for(size_t i=2; i<s.size(); ++i)
	{
		// check if we already have a item with the name s[i]
		QTreeWidgetItem* item = NULL;
		for(int j=0; j<parent->childCount(); ++j)
		{
			QTreeWidgetItem* it = parent->child(j);
			if(it->text(0).toLocal8Bit().data() == s[i]) {
				item = it;
				break;
			}
		}

		if(item==NULL) { // not found -> create new item
			item =  new ChannelTreeWidget::Item(parent,s[i],true);
			parent->addChild(item);
			item->setExpanded(false); // collapse sub namespaces by default
			// set italic font per default
			QFont font = item->font(0);
			font.setItalic(true);
			item->setFont(0,font);
			item->setIcon(0, QIcon(":/icons/Namespace.ico"));
		}

		parent = item; // prepare next round
	}

	// when we reach here parent contains the leaf-node for our channelID
	ChannelTreeWidget::Item* item = dynamic_cast<ChannelTreeWidget::Item*>(parent);
	assert(item!=NULL);

	mChannelItems.insert(std::make_pair(channelID,std::make_pair(item,0)));

	QFont font = item->font(0);
	font.setItalic(false);
	item->setFont(0,font);
	item->setIcon(0, QIcon(":/icons/Channel.png"));
	item->channelID = channelID;
	item->isNamespace = false;
	item->setToolTip(0, channelID.c_str());
}

///////////////////////////////////////////////////////////////////////////////
}
///////////////////////////////////////////////////////////////////////////////
