/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

#include <boost/version.hpp>
#include <boost/test/unit_test.hpp>
#if (BOOST_VERSION >= 107100)
#include <boost/test/tools/floating_point_comparison.hpp>
#else
#include <boost/test/floating_point_comparison.hpp>
#endif

#include <robot/DifferentialProbabilisticMotionModel.h>

using namespace mira;

robot::DifferentialProbabilisticMotionModel model;

PoseCov2 pose;
std::vector<Pose2> particles;

void step(Pose2 delta)
{
	robot::DifferentialProbabilisticMotionModel::MotionParameters p =
			model.calculateMotionParameters(Pose2(0.0f,0.0f,0.0f), delta);

	// particel filter model
	for(std::size_t i=0; i<particles.size(); ++i)
		particles[i] = model.sampleMotion(particles[i], p);

	// gaussian model
	pose = model.gaussianMotion(pose, p);
}

void computeMeanAndCov(const std::vector<Pose2>& particles, Eigen::Vector2f& mean, Eigen::Matrix2f& cov)
{
	mean = Eigen::Vector2f::Zero();
	cov  = Eigen::Matrix2f::Zero();

	for(std::size_t i=0; i<particles.size(); ++i)
		mean.block<2,1>(0,0) += particles[i].t;

	mean *= 1.0f / particles.size();

	for(std::size_t i=0; i<particles.size(); ++i)
	{
		Eigen::Vector2f x = particles[i].t;
		for(int j=0; j<2; ++j)
		for(int k=0; k<2; ++k)
			cov(j,k) += (x(j)-mean(j))*(x(k)-mean(k));
	}
	cov *= 1.0f / particles.size();
}

void runTest(float tscale)
{
	std::cout << std::endl;
	std::cout << std::endl;
	std::cout << "*** Running test with scale " << tscale << " ***" << std::endl;
	std::cout << std::endl;

	model.alphaTransTrans = 0.0001f;
	model.alphaTransRot = 0.0001f;
	model.alphaRotTrans = 0.005f;
	model.alphaRotRot = 0.01f;

	particles.clear();
	particles.resize(10000);

	pose = PoseCov2(0.0f,0.0f,0.0f, Eigen::Matrix3f::Identity()*0.0f);

	// forward
	for(int i=0; i<32/tscale; ++i)
		step(Pose2(0.05f*tscale,0.0f,0.0f));

	// left turn
	for(int i=0; i<16/tscale; ++i)
		step(Pose2(0.0f,0.0f,half_pi<float>()/(16.0f/tscale)));

	// forward
	for(int i=0; i<32/tscale; ++i)
		step(Pose2(0.05f*tscale,0.0f,0.0f));

	// right turn
	for(int i=0; i<16/tscale; ++i)
		step(Pose2(0.0f,0.0f,-half_pi<float>()/(16.0f/tscale)));

	// forward
	for(int i=0; i<32/tscale; ++i)
		step(Pose2(0.05f*tscale,0.0f,0.0f));
}

struct Reference
{
	Eigen::Vector2f mean;
	Eigen::Matrix2f cov;
};

Reference refParticles;
Reference refPose;

void checkParticles(const std::string& label, const std::vector<Pose2>& particles, const Reference& ref)
{
	std::cout << "** Performing Checks for " << label << "** " << std::endl;

	// particles
	Eigen::Vector2f mean;
	Eigen::Matrix2f cov;
	computeMeanAndCov(particles, mean, cov);

	std::cout << "Mean:" << std::endl;
	std::cout << mean << std::endl;

	std::cout << std::endl;

	std::cout << "Cov:" << std::endl;
	std::cout << cov << std::endl;

	std::cout << std::endl;

	std::cout << "Checking if indifferent to sampling interval" << std::endl;
	float prec = 15.0f; // we need a quite large confidence interval, since the particle filters sometimes differ significantly
	BOOST_CHECK_CLOSE(mean(0),ref.mean(0), prec);
	BOOST_CHECK_CLOSE(mean(1),ref.mean(1), prec);
	BOOST_CHECK_CLOSE(cov(0,0),ref.cov(0,0), prec);
	BOOST_CHECK_CLOSE(cov(0,1),ref.cov(0,1), prec);
	BOOST_CHECK_CLOSE(cov(1,1),ref.cov(1,1), prec);

	std::cout << std::endl;
}


void checkGaussian(const std::string& label, const PoseCov2& pose, const Reference& ref)
{
	std::cout << "** Performing Checks for " << label << "** " << std::endl;

	std::cout << "Mean:" << std::endl;
	std::cout << pose.t << std::endl;

	std::cout << std::endl;

	std::cout << "Cov:" << std::endl;
	std::cout << pose.cov.block<2,2>(0,0) << std::endl;

	std::cout << std::endl;

	std::cout << "Checking if indifferent to sampling interval" << std::endl;
	float prec = 1.0f;
	BOOST_CHECK_CLOSE(pose.t(0),ref.mean(0), prec);
	BOOST_CHECK_CLOSE(pose.t(1),ref.mean(1), prec);
	BOOST_CHECK_CLOSE(pose.cov(0,0),ref.cov(0,0), prec);
	BOOST_CHECK_CLOSE(pose.cov(0,1),ref.cov(0,1), prec);
	BOOST_CHECK_CLOSE(pose.cov(1,1),ref.cov(1,1), prec);

	std::cout << std::endl;
}

void compareParticlesWithGaussian(const std::vector<Pose2>& particles, const PoseCov2& pose)
{
	std::cout << "** Comparing mean and cov of particle samples with Gaussian - Both should almost be equal" << std::endl;

	Eigen::Vector2f mean;
	Eigen::Matrix2f cov;
	computeMeanAndCov(particles, mean, cov);

	float prec = 15.0f; // we need a quite large confidence interval, since the particle filters sometimes differ significantly
	BOOST_CHECK_CLOSE(pose.t(0),mean(0), prec);
	BOOST_CHECK_CLOSE(pose.t(1),mean(1), prec);
	BOOST_CHECK_CLOSE(pose.cov(0,0),cov(0,0), prec);
	BOOST_CHECK_CLOSE(pose.cov(0,1),cov(0,1), prec);
	BOOST_CHECK_CLOSE(pose.cov(1,1),cov(1,1), prec);
}

BOOST_AUTO_TEST_CASE(testModels)
{
	runTest(1.0f);
	// build up the references
	computeMeanAndCov(particles, refParticles.mean, refParticles.cov);

	refPose.mean = pose.t;
	refPose.cov = pose.cov.block<2,2>(0,0);

	checkParticles("Particles", particles, refParticles);
	checkGaussian("Gaussian", pose, refPose);
	compareParticlesWithGaussian(particles,pose);

	runTest(2.0f);
	checkParticles("Particles", particles, refParticles);
	checkGaussian("Gaussian", pose, refPose);
	compareParticlesWithGaussian(particles,pose);

	runTest(0.5f);
	checkParticles("Particles", particles, refParticles);
	checkGaussian("Gaussian", pose, refPose);
	compareParticlesWithGaussian(particles,pose);
}
