/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RangeScanVisualization.C
 *    3D visualization plugin for miracenter to visualize RangeScan data.
 *
 * @author Tim Langner, Christian Martin
 * @date   2011/01/02
 */

#include <serialization/Serialization.h>
#include <serialization/SetterNotify.h>
#include <serialization/PropertyHint.h>
#include <serialization/DefaultInitializer.h>
#include <image/Color.h>

#include <transform/Pose.h>
#include <math/LinearInterpolator.h>
#include <math/Saturate.h>

#include <robot/RangeScan.h>

#include <visualization/ColormapProperty.h>
#include <visualization/Visualization3DBasic.h>
#include <visualization/3d/DynamicPoints.h>
#include <visualization/3d/TextObject.h>

#include <gui/HighlightProperty.h>

#include <widgets/OgreUtils.h>

#include <OGRE/OgreManualObject.h>
#include <OGRE/OgreSceneNode.h>
#include <OGRE/OgreSceneManager.h>
#include <OGRE/OgreQuaternion.h>
#include <OGRE/OgreMaterialManager.h>

using namespace Eigen;
using namespace std;

namespace mira { namespace robot {

//////////////////////////////////////////////////////////////////////////////

/**
 * 3D visualization plugin for miracenter to visualize RangeScan data.
 */
class RangeScanVisualization3D :  public Visualization3DBasic<robot::RangeScan>
{
	MIRA_META_OBJECT(mira::robot::RangeScanVisualization3D,
			("Name", "Range Scan")
			("Description", "3D Visualization of 2D range scans")
			("Category", "Robot Datatypes"))

	typedef Visualization3DBasic<robot::RangeScan> Base;

public:
	struct Face
	{
		Vector3f p1,p2;
		Vector3f n;
		bool smooth;
	};

	/// Range scan rendering mode.
	enum RenderMode
	{
		BAND = 0,  ///< Show the range scan as a band.
		FAN,       ///< Show the range scan as a fan.
		POINTS,    ///< Show the range scan as a list of points.
		CONES      ///< Show the range scan as individual cones.
	};

	/// Range scan color mode.
	enum ColorMode
	{
		DEFAULT_COLOR = 0, ///< use default color for all scans
		COLOR_CHANNEL,     ///< use color from separate color channel
		VALID,             ///< use valid field for color
		REFLECTANCE,       ///< use reflectance values
		CERTAINTY          ///< use certainty values
	};

	/// The default constructor
	RangeScanVisualization3D() : Base("Range"),
		mScanObject(NULL),
		mPoints(NULL),
		mText(NULL),
		mChosenColormap(NULL),
		mHighlight(&RangeScanVisualization3D::drawLast, this)
	{
		MIRA_INITIALIZE_THIS;
	}

	/// The destructor.
	virtual ~RangeScanVisualization3D()
	{
		if(getSite()==NULL)
			return;

		if ( mScanObject != NULL )
			getSite()->getSceneManager()->destroyManualObject(mScanObject);

		if (!mMaterial.isNull())
			Ogre::MaterialManager::getSingletonPtr()->remove(mMaterial->getName());

		delete mText;
	}

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Base);
		r.property("RenderMode", mRenderMode,
		           setterNotify(mRenderMode, &RangeScanVisualization3D::drawLast, this),
		           "Specify the render mode", FAN,
		           mira::PropertyHints::enumeration("Band;Fan;Points;Cones"));
		r.property("ColorMode", mColorMode,
		           setterNotify(mColorMode, &RangeScanVisualization3D::drawLast, this),
		           "Specify the color mode", VALID,
		           mira::PropertyHints::enumeration("Default;ColorChannel;Valid;Reflectance;Certainty"));
		r.property("Color", mColor,
		           setter<Ogre::ColourValue>(&RangeScanVisualization3D::setColor, this),
		           "The color", Ogre::ColourValue(0.2f, 0.5f, 0.7f, 1.0f));
		r.property("ColorInvalid", mColorInvalid,
		           setterNotify(mColorInvalid, &RangeScanVisualization3D::drawLast, this),
		           "The color for invalid values.",
		           Ogre::ColourValue(0.0f, 0.0f, 0.0f, 1.0f));
		r.property("ColorInvalidUser", mColorInvalidUser,
				setterNotify(mColorInvalidUser, &RangeScanVisualization3D::drawLast, this),
				   "The color for user defined invalid values.",
				   Ogre::ColourValue(1.0f, 0.0f, 0.0f, 1.0f));
		r.property("ColorAboveMax", mColorAboveMax,
		           setterNotify(mColorAboveMax, &RangeScanVisualization3D::drawLast, this),
		           "The color for values above maximum.",
		           Ogre::ColourValue(1.0f, 1.0f, 0.0f, 1.0f));
		r.property("ColorBelowMin", mColorBelowMin,
		           setterNotify(mColorBelowMin, &RangeScanVisualization3D::drawLast, this),
		           "The color for values below minimum.",
		           Ogre::ColourValue(0.0f, 1.0f, 1.0f, 1.0f));
		r.property("ColorMasked", mColorMasked,
		           setterNotify(mColorMasked, &RangeScanVisualization3D::drawLast, this),
		           "The color for masked values.",
		           Ogre::ColourValue(1.0f, 1.0f, 1.0f, 1.0f));
		r.property("Colormap", mColormap,
		           setter<ColormapProperty>(&RangeScanVisualization3D::setColormap, this),
		           "The color palette for 'Reflectance' and 'Certainty'",
		           ColormapProperty("mira::GrayscaleColormap"));
		r.property("Max Range", mMaxRange,
		           setterNotify(mMaxRange, &RangeScanVisualization3D::drawLast, this),
		           "The max. range that is visualized. The transparency is increased up to this "
		           "range to fade the scan out (disabled if <=0)", 0.0f);
		r.property("Highlight", mHighlight,
		           "Select a single scan to highlight", HighlightProperty(&RangeScanVisualization3D::drawLast, this));
		r.property("Alpha", mAlpha,
			setter<float>(&RangeScanVisualization3D::setAlpha, this),
			"The alpha factor used for controlling the opacity of the range scan", 0.5f,
			mira::PropertyHints::limits<float>(0.0f, 1.0f));

		channelProperty(r, "Colors", mColorChannel, "", NOT_REQUIRED);
	}

public: // implementation of Visualization3D and Visualization

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{
		mScanObject = getSite()->getSceneManager()->createManualObject("RangeScanVisualization3D"+toString(this));
		node->attachObject(mScanObject);

		mPoints = new DynamicPoints();
		node->attachObject(mPoints);

		mMaterial = Ogre::MaterialManager::getSingleton().create("RangeScanVisualization3D"+toString(this)+"/Material", "MIRAMaterials");
		mMaterial->setReceiveShadows(false);
		mMaterial->getTechnique(0)->setLightingEnabled(true);
		mMaterial->getTechnique(0)->setCullingMode(Ogre::CULL_NONE);
		mMaterial->getTechnique(0)->setAmbient(mColor * 0.5);
		mMaterial->getTechnique(0)->setDiffuse(mColor);
		mMaterial->setSceneBlending(Ogre::SBT_TRANSPARENT_ALPHA);
		mMaterial->setDepthWriteEnabled(false);

		mPoints->setMaterial("TransparentNoLight");

		mText = new TextObject("-", mgr, node);
		mText->setTextAlignment(TextObject::H_LEFT, TextObject::V_BELOW);
		mText->setCharacterHeight(mHighlight.textsize());
		mText->setColor(mHighlight.color());
		mText->showOnTop();
		mText->setVisible(false);
	}

	void setAlpha(float alpha)
	{
		mAlpha = saturate<float>(alpha, 0.0f, 1);
		drawLast();
	}

	void setColor(const Ogre::ColourValue& color)
	{
		mColor = color;
		if ( !mMaterial.isNull() )
		{
			mMaterial->getTechnique(0)->setAmbient(mColor * 0.5);
			mMaterial->getTechnique(0)->setDiffuse(mColor);
		}
		drawLast();
	}

	void setColormap(const ColormapProperty& palette)
	{
		mColormap = palette;
		mColormap.setFilter(ColormapProperty::filterContinuous);

		mChosenColormap = dynamic_cast<const ContinuousColormap*>(&mColormap.getColormap());

		drawLast();
	}

	Ogre::ColourValue getColorForScan(const RangeScan& scan, std::size_t i)
	{
		// By default or fall-back use the standard color
		Ogre::ColourValue color = mColor;

		switch(mColorMode) {
		case DEFAULT_COLOR:
			break;
		case COLOR_CHANNEL:
			color = i < mColors.size() ?  mColors[i] : mColor;
			break;
		case VALID:
			if (i < scan.valid.size()) {
				if (scan.valid[i] == RangeScan::Invalid)
					color = mColorInvalid;
				else
				if (scan.valid[i] == RangeScan::InvalidUser)
					color = mColorInvalidUser;
				else
				if (scan.valid[i] == RangeScan::AboveMaximum)
					color = mColorAboveMax;
				else
				if (scan.valid[i] == RangeScan::BelowMinimum)
					color = mColorBelowMin;
				else
				if (scan.valid[i] == RangeScan::Masked)
					color = mColorMasked;
			}
			break;
		case REFLECTANCE:
			if(mChosenColormap != NULL) {
				float value =
					(i < scan.reflectance.size()) ? scan.reflectance[i] : 0.0f;
				Color::RGBA c = mChosenColormap->getf(value);
				color = Ogre::ColourValue(c.r, c.g, c.b, c.a);
			}
			break;
		case CERTAINTY:
			if(mChosenColormap != NULL) {
				float value =
					(i < scan.certainty.size()) ? scan.certainty[i] : 0.0f;
				Color::RGBA c = mChosenColormap->getf(value);
				color = Ogre::ColourValue(c.r, c.g, c.b, c.a);
			}
			break;
		}

		// Update the colour with our alpha value
		color.a *= mAlpha;

		return color;
	}

	void updateColors()
	{
		try {
			Channel<std::vector<Color::RGBA>> ch = mColorChannel.getChannel();
			auto r = ch.read();

			mColors.clear();
			foreach(const Color::RGBA& c, r->value()) {
				mColors.push_back(Ogre::ColourValue(c.r, c.g, c.b, c.a));
			}

		} catch(...) {
			mColors.clear();
		}
	}

protected:
	void dataChanged(ChannelRead<robot::RangeScan> scan);

protected:
	void drawLast();
	Vector3f computeRangeScan(const robot::RangeScan& scan, size_t i, float maxRange=0.0f);
	void drawFan(const robot::RangeScan& scan);
	void drawBand(const robot::RangeScan& scan);
	void drawPoints(const robot::RangeScan& scan);
	void drawCones(const robot::RangeScan& scan);
	void drawCone(const robot::RangeScan& scan, uint i,
	              const Ogre::ColourValue & color);
	void displayData(const robot::RangeScan& scan, uint i);

	ChannelProperty<std::vector<Color::RGBA>> mColorChannel;
	std::vector<Ogre::ColourValue> mColors;

	RenderMode mRenderMode;
	ColorMode mColorMode;
	Ogre::ManualObject* mScanObject;
	DynamicPoints *mPoints;
	TextObject* mText;
	Ogre::MaterialPtr mMaterial;
	Ogre::ColourValue mColor;
	Ogre::ColourValue mColorInvalid;
	Ogre::ColourValue mColorInvalidUser;
	Ogre::ColourValue mColorAboveMax;
	Ogre::ColourValue mColorBelowMin;
	Ogre::ColourValue mColorMasked;
	ColormapProperty mColormap;
	float mMaxRange;
	float mAlpha;
	const ContinuousColormap* mChosenColormap;
	HighlightProperty mHighlight;
};

//////////////////////////////////////////////////////////////////////////////

void RangeScanVisualization3D::dataChanged(ChannelRead<robot::RangeScan> scan)
{
	updateColors();

	mScanObject->clear();
	mPoints->clear(); mPoints->update();
	RenderMode mode = (RenderMode)mRenderMode;
	switch (mode)
	{
		case BAND: drawBand(scan); break;
		case FAN: drawFan(scan); break;
		case POINTS: drawPoints(scan); break;
		case CONES: drawCones(scan); break;
		default: MIRA_LOG(ERROR) << "Unknown draw mode " << mode; return;
	}

	mHighlight.setWrapIndex((int)scan->range.size());

	if (mHighlight.isEnabled() && scan->range.size())
	{
		drawCone(scan, mHighlight.index(), mHighlight.color());
		displayData(scan, mHighlight.index());
	}
	else
		mText->setVisible(false);
}

void RangeScanVisualization3D::drawLast()
{
	try
	{
		if (!mDataChannel.isValid())
			return;
		dataChanged(mDataChannel.getChannel().read());
	}
	catch(Exception&)
	{
	}
}

Vector3f RangeScanVisualization3D::computeRangeScan(const robot::RangeScan& scan, size_t i, float maxRange)
{
	float alpha0 = scan.startAngle;
	float dalpha = scan.deltaAngle;
	float alpha = alpha0 + dalpha * i;
	float r = scan.range[i];
	if(maxRange>0.0f && r>maxRange)
		r = maxRange;

	return Vector3f(r*cosf(alpha), r*sinf(alpha), 0);
}

void RangeScanVisualization3D::drawPoints(const robot::RangeScan& scan)
{
	for(size_t i=0;i<scan.range.size();++i)
	{
		Vector3f p = computeRangeScan(scan, i);
		mPoints->addPoint(Ogre::Vector3(p(0), p(1), p(2)), getColorForScan(scan, i));
	}
	mPoints->update();

}

inline Ogre::ColourValue endColor(Ogre::ColourValue baseColor, const RangeScan& scan, std::size_t i, float maxRange)
{
	if(maxRange<=0.0f)
		return baseColor;

	float r = scan.range[i];
	if(r>maxRange)
		r = maxRange;

	float alpha = lerp(baseColor.a, 0.0f, r/maxRange);
	return Ogre::ColourValue(baseColor.r, baseColor.g, baseColor.b, alpha);
}

void RangeScanVisualization3D::drawFan(const RangeScan& scan)
{
	// Handle range scans with only one scan line
	if (scan.range.size() == 1)
	{
		drawCone(scan, 0, getColorForScan(scan, 0));
	}
	else
	{
		mScanObject->begin("TransparentNoLightTwoSided",
		                   Ogre::RenderOperation::OT_TRIANGLE_FAN);
		mScanObject->position(0, 0, 0);
		Ogre::ColourValue oldColor = getColorForScan(scan, 0);
		mScanObject->colour(oldColor);
		for(size_t i=0; i<scan.range.size(); ++i)
		{
			Vector3f p = computeRangeScan(scan, i, mMaxRange);
			Ogre::ColourValue newColor  = getColorForScan(scan, i);
			mScanObject->position(p(0), p(1), p(2));
			mScanObject->colour(endColor(oldColor, scan, i, mMaxRange));
			if (newColor != oldColor)
			{
				mScanObject->end();
				mScanObject->begin("TransparentNoLightTwoSided",
				                   Ogre::RenderOperation::OT_TRIANGLE_FAN);
				mScanObject->position(0, 0, 0);
				mScanObject->colour(newColor);
				mScanObject->position(p(0), p(1), p(2));
				mScanObject->colour(endColor(newColor, scan, i, mMaxRange));
				oldColor = newColor;
			}
		}
		mScanObject->end();
	}
}

void RangeScanVisualization3D::drawCone(const RangeScan& scan, uint i,
                                        const Ogre::ColourValue & color)
{
	// mid of the cone
	float alpha = scan.startAngle + i*scan.deltaAngle.value();
	// "left" side of the cone
	float alpha0 = alpha - scan.coneAngle.value()/2.0f;
	// "right" side of the cone
	float alpha1 = alpha0 + scan.coneAngle;

	float r = scan.range[i];
	if(mMaxRange>0.0f && r>mMaxRange)
		r = mMaxRange;

	Vector3f p1(r*cosf(alpha0), r*sinf(alpha0), 0);
	Vector3f p2(r*cosf(alpha1), r*sinf(alpha1), 0);
	Vector3f aOffsetL(-scan.aperture*sinf(alpha),  scan.aperture*cosf(alpha), 0);
	Vector3f aOffsetR( scan.aperture*sinf(alpha), -scan.aperture*cosf(alpha), 0);

	mScanObject->begin("TransparentNoLightTwoSided",
	                   Ogre::RenderOperation::OT_TRIANGLE_STRIP);
	mScanObject->position(aOffsetL(0), aOffsetL(1), 0);
	mScanObject->colour(color);
	mScanObject->position(p2(0)+aOffsetL(0), p2(1)+aOffsetL(1), p2(2));
	mScanObject->colour(endColor(color, scan, i, mMaxRange));
	mScanObject->position(aOffsetR(0), aOffsetR(1), 0);
	mScanObject->colour(endColor(color, scan, i, mMaxRange));
	mScanObject->position(p1(0)+aOffsetR(0), p1(1)+aOffsetR(1), p1(2));
	mScanObject->colour(endColor(color, scan, i, mMaxRange));
	mScanObject->end();
}

void RangeScanVisualization3D::drawCones(const RangeScan& scan)
{
	for(size_t i=0; i<scan.range.size(); ++i)
		drawCone(scan, i, getColorForScan(scan, i));
}

void RangeScanVisualization3D::drawBand(const robot::RangeScan& scan)
{
	const float max_rdiff = 1.0f;
	const float hwidth = 0.15f * 0.5f;

	mScanObject->begin(mMaterial->getName(),
	                   Ogre::RenderOperation::OT_TRIANGLE_LIST);

	std::vector<Face> faces;
	for(size_t i=1;i<scan.range.size();++i)
	{
		if (scan.valid.size()>0 && scan.valid[i] != RangeScan::Valid)
			continue;
		Face f;
		// compute both points
		float r1 = scan.range[i-1];
		float r2 = scan.range[i];
		f.p1 = computeRangeScan(scan, i-1);
		f.p2 = computeRangeScan(scan, i);

		if(r1>0.0f && r2>0.0f && std::abs(r1-r2) < max_rdiff)
		{
			// face is "smooth", i.e. is not part of a large discontinuity
			f.smooth=true;

			// compute the normal
			Vector3f d1;
			d1 = f.p2 - f.p1;
			Vector3f normal;

			normal <<  d1(1)*hwidth,
			          -d1(0)*hwidth,
			           0.0f;

			normal.normalize();
			f.n = normal;
		} else
			f.smooth=false;

		// add the face
		faces.push_back(f);
	}

	for(size_t i=0; i<faces.size(); ++i)
	{
		const Face& f = faces[i];
		if (!f.smooth)
			continue;
		Vector3f n0, n1;

		if(i>0 && faces[i-1].smooth)
			n0 = 0.5f * (f.n + faces[i-1].n);
		else
			n0 = f.n;

		if(i<faces.size()-1 && faces[i+1].smooth)
			n1 = 0.5f * (f.n + faces[i+1].n);
		else
			n1 = f.n;

		mScanObject->position(f.p1(0), f.p1(1), f.p1(2)-hwidth);
		mScanObject->normal(n0(0), n0(1), n0(2));
		mScanObject->position(f.p2(0), f.p2(1), f.p2(2)-hwidth);
		mScanObject->normal(n1(0), n1(1), n1(2));
		mScanObject->position(f.p2(0), f.p2(1), f.p2(2)+hwidth);
		mScanObject->normal(n1(0), n1(1), n1(2));

		mScanObject->position(f.p2(0), f.p2(1), f.p2(2)+hwidth);
		mScanObject->normal(n1(0), n1(1), n1(2));
		mScanObject->position(f.p1(0), f.p1(1), f.p1(2)+hwidth);
		mScanObject->normal(n0(0), n0(1), n0(2));
		mScanObject->position(f.p1(0), f.p1(1), f.p1(2)-hwidth);
		mScanObject->normal(n0(0), n0(1), n0(2));
	}
	mScanObject->end();
}

void RangeScanVisualization3D::displayData(const robot::RangeScan& scan, uint i)
{
	mText->setCharacterHeight(mHighlight.textsize());
	mText->setColor(mHighlight.color());
	mText->setVisible(isEnabled());

	std::stringstream ss;
	ss << "range=" << scan.range[i];
	if (scan.valid[i] >= RangeScan::InvalidUser)
		ss << "\nvalid=InvalidUser";
	else
		switch (scan.valid[i])
		{
			case RangeScan::Valid        : ss << "\nvalid=Valid";
						                   break;
			case RangeScan::BelowMinimum : ss << "\nvalid=BelowMinimum";
			             			       break;
			case RangeScan::AboveMaximum : ss << "\nvalid=AboveMaximum";
						                   break;
			case RangeScan::Invalid      : ss << "\nvalid=Invalid";
						                   break;
			case RangeScan::Masked       : ss << "\nvalid=Masked";
						                   break;
			default                      : ss << "\nvalid=Unknown";
			                               break;
		}

	if (scan.certainty.size() > i)
		ss << "\ncertainty=" << scan.certainty[i];

	if (scan.reflectance.size() > i)
		ss << "\nreflectance=" << scan.reflectance[i];

	mText->setCaption(ss.str());
}
//////////////////////////////////////////////////////////////////////////////


}} // namespace

MIRA_CLASS_SERIALIZATION(mira::robot::RangeScanVisualization3D, mira::Visualization3D);
