/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file URDF.C
 *    Implementation of URDF.h.
 *
 * @author Tim Langner
 * @date   2011/12/05
 */

#include <model/URDF.h>

#include <transform/Pose.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

std::string loadNameFromURDF(const XMLDom::const_iterator& urdf)
{
	return urdf.get_attribute<std::string>("name", "");
}

Pose3 loadOriginFromURDF(const XMLDom::const_iterator& urdf)
{
	Pose3 pose;
	XMLDom::const_iterator origin = urdf.find("origin");
	if (origin != urdf.end())
	{
		XMLDom::const_attribute_iterator xyz = origin.find_attribute("xyz");
		if (xyz != origin.attribute_end())
		{
			std::stringstream ss(xyz.value());
			ss >> pose.x() >> pose.y() >> pose.z();
		}
		XMLDom::const_attribute_iterator rpy = origin.find_attribute("rpy");
		if (rpy != origin.attribute_end())
		{
			std::stringstream ss(rpy.value());
			float r, p, y;
			ss >> r >> p >> y;
			pose.r = quaternionFromYawPitchRoll(y, p, r);
		}
	}
	return pose;
}

Point3f loadAxisFromURDF(const XMLDom::const_iterator& urdf)
{
	Point3f point(1.0f, 0.0f, 0.0f);
	XMLDom::const_iterator axis = urdf.find("axis");
	if (axis != urdf.end())
	{
		XMLDom::const_attribute_iterator xyz = axis.find_attribute("xyz");
		if (xyz != axis.attribute_end())
		{
			std::stringstream ss(xyz.value());
			ss >> point.x() >> point.y() >> point.z();
		}
	}
	return point;
}

///////////////////////////////////////////////////////////////////////////////

MaterialPtr loadMaterialFromURDF(const XMLDom::const_iterator& urdf)
{
	MaterialPtr material(new Material());
	material->name = urdf.get_attribute<std::string>("name");
	bool hasContent = false;
	// optional color
	XMLDom::const_iterator colorTag = urdf.find("color");
	if (colorTag != urdf.end())
	{
		hasContent = true;
		std::stringstream ss(colorTag.get_attribute<std::string>("rgba"));
		ss >> material->color.r >> material->color.g >> material->color.b >> material->color.a;
	}
	else
		material->color = Color::RGBA(1.0f, 1.0f, 1.0f, 1.0f);
	// optional texture
	XMLDom::const_iterator textureTag = urdf.find("texture");
	if (textureTag != urdf.end())
	{
		hasContent = true;
		material->texture = textureTag.get_attribute<std::string>("filename");
	}
	else
		material->texture = "";
	return hasContent?material:MaterialPtr();
}

///////////////////////////////////////////////////////////////////////////////

GeometryPtr loadGeometryFromURDF(const XMLDom::const_iterator& urdf)
{
	XMLDom::const_iterator geometry = urdf.find("geometry");
	if (geometry == urdf.end())
		MIRA_THROW(XInvalidConfig, "No 'geometry' tag found");
	XMLDom::const_iterator shape = geometry.begin();
	if (*shape == "box")
	{
		BoxPtr box(new Box());
		std::stringstream ss(shape.get_attribute<std::string>("size"));
		ss >> box->size.width() >> box->size.height() >> box->size.depth();
		return box;
	}
	else if (*shape == "cylinder")
	{
		CylinderPtr cylinder(new Cylinder());
		cylinder->radius = shape.get_attribute<float>("radius");
		cylinder->length = shape.get_attribute<float>("length");
		return cylinder;
	}
	else if (*shape == "sphere")
	{
		SpherePtr sphere(new Sphere());
		sphere->radius = shape.get_attribute<float>("radius");
		return sphere;
	}
	else if (*shape == "mesh")
	{
		MeshPtr mesh(new Mesh());
		mesh->filename = shape.get_attribute<std::string>("filename");
		mesh->scale = shape.get_attribute<float>("scale", 1.0f);
		return mesh;
	}
	else
		MIRA_THROW(XInvalidConfig, "Unknown shape '" << *shape << "'");
	return GeometryPtr();
}

///////////////////////////////////////////////////////////////////////////////

InertialPtr loadInertialFromURDF(const XMLDom::const_iterator& urdf)
{
	InertialPtr inertial(new Inertial());
	// optional origin
	inertial->origin = loadOriginFromURDF(urdf);
	return inertial;
}

///////////////////////////////////////////////////////////////////////////////

CollisionPtr loadCollisionFromURDF(const XMLDom::const_iterator& urdf)
{
	CollisionPtr collision(new Collision());
	// optional name
	collision->name = loadNameFromURDF(urdf);
	// optional origin
	collision->origin = loadOriginFromURDF(urdf);
	// required geometry
	collision->geometry = loadGeometryFromURDF(urdf);
	return collision;
}

///////////////////////////////////////////////////////////////////////////////

VisualPtr loadVisualFromURDF(const XMLDom::const_iterator& urdf)
{
	VisualPtr visual(new Visual());
	// optional name
	visual->name = loadNameFromURDF(urdf);
	// optional origin
	visual->origin = loadOriginFromURDF(urdf);
	// required geometry
	visual->geometry = loadGeometryFromURDF(urdf);
	// optional material
	XMLDom::const_iterator materialTag = urdf.find("material");
	if (materialTag != urdf.end())
	{
		visual->materialName = materialTag.get_attribute<std::string>("name");
		visual->material = loadMaterialFromURDF(materialTag);
	}
	return visual;
};

///////////////////////////////////////////////////////////////////////////////

LinkPtr loadLinkFromURDF(const XMLDom::const_iterator& urdf)
{
	LinkPtr link(new Link());
	link->name = urdf.get_attribute<std::string>("name");
	// optional inertial
	XMLDom::const_iterator inertialTag = urdf.find("inertial");
	if (inertialTag != urdf.end())
		link->inertial = loadInertialFromURDF(inertialTag);


	XMLDom::const_iterator ci = urdf.begin();
	for ( ; ci != urdf.end(); ++ci)
	{
		// optional visual
		if (*ci == "visual")
			link->visuals.push_back(loadVisualFromURDF(ci));
		// optional collision
		if (*ci == "collision")
			link->collisions.push_back(loadCollisionFromURDF(ci));
	}

	return link;
}

///////////////////////////////////////////////////////////////////////////////

JointPtr loadJointFromURDF(const XMLDom::const_iterator& urdf)
{
	JointPtr joint(new Joint());
	joint->name = urdf.get_attribute<std::string>("name");
	std::string type = urdf.get_attribute<std::string>("type");
	if (type == "revolute")
		joint->type = Joint::REVOLUTE;
	else if (type == "continuous")
		joint->type = Joint::CONTINUOUS;
	else if (type == "prismatic" || type == "planar")
		joint->type = Joint::PRISMATIC;
	else if (type == "fixed")
		joint->type = Joint::FIXED;
	else if (type == "floating")
		joint->type = Joint::FLOATING;
	else
		MIRA_THROW(XInvalidConfig, "Joint '" << joint->name << "' has invalid type");
	// optional origin
	joint->origin = loadOriginFromURDF(urdf);
	// required parent
	XMLDom::const_iterator parentTag = urdf.find("parent");
	if (parentTag == urdf.end())
		MIRA_THROW(XInvalidConfig, "Joint '" << joint->name << "' has no parent tag");
	joint->parent = parentTag.get_attribute<std::string>("link");
	// required child
	XMLDom::const_iterator childTag = urdf.find("child");
	if (childTag == urdf.end())
		MIRA_THROW(XInvalidConfig, "Joint '" << joint->name << "' has no child tag");
	joint->child = childTag.get_attribute<std::string>("link");
	// optional axis
	joint->axis = loadAxisFromURDF(urdf);
	// optional limit for continuous, floating and fixed
	XMLDom::const_iterator limitTag = urdf.find("limit");
	if (limitTag == urdf.end() && 
		(joint->type == Joint::REVOLUTE || joint->type == Joint::PRISMATIC))
		MIRA_THROW(XInvalidConfig, "Missing limit tag for revolute/prismatic joint '" << joint->name << "'");
	if (limitTag != urdf.end())
	{
		if (limitTag.find_attribute("lower") != limitTag.attribute_end())
			joint->lowerLimit = limitTag.get_attribute<float>("lower");
		if (limitTag.find_attribute("upper") != limitTag.attribute_end())
			joint->upperLimit = limitTag.get_attribute<float>("upper");
	}
	// TODO calibration, dynamics,...
	return joint;
}

///////////////////////////////////////////////////////////////////////////////

void loadModelFromURDF(const XMLDom::const_iterator& urdf, RigidModel& oModel)
{
	oModel.clear();

	std::set<std::string> linkNames;

	if (*urdf != "robot")
		MIRA_THROW(XInvalidConfig, "URDF does not contain 'robot' root node");
	oModel.name = urdf.get_attribute<std::string>("name");
	XMLDom::const_iterator i = urdf.begin();
	for ( ; i != urdf.end(); ++i)
	{
		if (*i == "material")
		{
			MaterialPtr m = loadMaterialFromURDF(i);
			if (oModel.materials.count(m->name) != 0)
				MIRA_THROW(XInvalidConfig, "Material must be unique ('" << m->name << "')");
			oModel.materials[m->name] = m;
		}
		else if (*i == "link")
		{
			LinkPtr l = loadLinkFromURDF(i);
			if (!linkNames.insert(l->name).second)
				MIRA_THROW(XInvalidConfig, "Link must be unique ('" << l->name << "')");
			oModel.links.push_back(l);
			if (l->visuals.size() == 1)
			{
				// look for a material in the visual component of the link
				// that is either a named reference to a global material or
				// a new material that get inserted into the global material list
				// checks also for undefined material
				VisualPtr v = l->visuals[0];
				// if visual has a material
				if (v && !v->materialName.empty())
				{
					// check if its already contained in the global material list
					if (oModel.materials.count(v->materialName))
					{
						// reference to the global material
						v->material = oModel.materials.find(v->materialName)->second;
					}
					else // otherwise its a new one
					{
						// if the material is fully defined (not just a name reference)
						// add a reference to the global material list
						if (v->material)
							oModel.materials[v->materialName] = v->material;
						else // material is a reference to an undefined material
							MIRA_THROW(XInvalidConfig, "Material '" << v->materialName << "' is undefined");
					}
				}
			}
		}
		else if (*i == "joint")
		{
			JointPtr j = loadJointFromURDF(i);
			if (oModel.joints.count(j->name) != 0)
				MIRA_THROW(XInvalidConfig, "Joint must be unique ('" << j->name << "')");
			oModel.joints[j->name] = j;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}}
