/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IRigidModelProvider.h
 *    Interface for rigid model provider services.
 *
 * @author Christof Schröter
 * @date   2019/11/20
 */

#ifndef _MIRA_IRIGIDMODELPROVIDER_H_
#define _MIRA_IRIGIDMODELPROVIDER_H_

#include <serialization/IgnoreMissing.h>

#include <fw/Framework.h>
#include <fw/ServiceProperty.h>

#include <model/RigidModel.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

/**
 * Interface for rigid model provider services
 */
class IRigidModelProvider
{
public:
	
	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("IRigidModelProvider");
		r.method("getRigidModel", &IRigidModelProvider::getRigidModel, this,
		         "Return the current rigid model");
	}

	/// The destructor
	virtual ~IRigidModelProvider() {}

	/**
	 * Return the used rigid model pointer
	 */
	virtual model::RigidModelPtr getRigidModel() = 0;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * IRigidModelConsumer is a unit that has a ServiceProperty 'ModelProvider'
 * for an IRigidModelProvider service
 */
class IRigidModelConsumer
{
public:

	// Subclasses should set mModelProvider themselves if they expect a different
	// (derived) interface
	IRigidModelConsumer() : mModelProvider("IRigidModelProvider") {}
	virtual ~IRigidModelConsumer() {}

	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("ModelProvider", mModelProvider,
		           setter(&IRigidModelConsumer::setModelProvider, this),
		           "The name of the service that provides a rigid model "
		           "(if empty, a suitable service should be chosen automatically)",
		           serialization::IgnoreMissing());

		r.interface("IRigidModelConsumer");
	}

	// special reflect for XML deserialization, with a fallback if it is configured as plain string
	void reflect(XMLDeserializer& r)
	{
		try {
			r.property("ModelProvider", mModelProvider,
			           setter(&IRigidModelConsumer::setModelProvider, this),
			           "The name of the service that provides a rigid model "
			           "(if empty, a suitable service should be chosen automatically)",
			           serialization::IgnoreMissing());
		} catch (const XMemberNotFound_NoDefault&) {
			// workaround for old configs where ModelProvider was a string property
			MIRA_LOG(WARNING) << "IRigidModelConsumer failed reading property ModelProvider "
			                     "as ServiceProperty. Fallback: trying as string.";
			std::string service;
			r.property("ModelProvider", service, "", "");
			MIRA_LOG(WARNING) << "Expected ServiceProperty for IRigidModelConsumer's ModelProvider "
			                     "property, found string '" << service << "'. Please update your config!";
			mModelProvider = service;
			setModelProvider(mModelProvider);
		}
	}

	// setRobotModelProvider must be implemented and must set mModelProvider
	virtual void setModelProvider(ServiceProperty service) = 0;

protected:
	ServiceProperty mModelProvider;
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif
