/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Geometry.h
 *    Geometric representation of a model component.
 *
 * @author Tim Langner
 * @date   2011/05/09
 */

#ifndef _MIRA_MODEL_GEOMETRY_H_
#define _MIRA_MODEL_GEOMETRY_H_

#include <transform/Pose.h>

#include <geometry/Rect.h>
#include <geometry/Polygon.h>

#include <model/RigidModelExports.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for all geometric representations of a rigid model part (link).
 * Geometric representations allow to compute a footprint and a bounding
 * box e.g. for collision detection or navigation tasks.
 */
class MIRA_RIGID_MODEL_EXPORT Geometry : public Object
{
	MIRA_OBJECT(Geometry)
public:

	virtual ~Geometry() {}

public:

	/**
	 * Computes the footprint of the geometry primitive depending on
	 * the pose of the primitive that is specified by its origin.
	 * The footprint is the projection of the primitives outline
	 * onto the xy-plane. The location of the origin within the
	 * primitive depends on the actual primitive.
	 */
	virtual Polygon2f getFootprint(const RigidTransform3f& origin) const = 0;

	/**
	 * Computes the axis aligned bounding box of the geometry
	 * primitive depending on the pose of the primitive that is
	 * specified by its origin. The location of the origin within
	 * the primitive depends on the actual primitive.
	 */
	virtual Box3f getBoundingBox(const RigidTransform3f& origin) const = 0;

};

/// pointer to a geometric representation
typedef boost::shared_ptr<Geometry> GeometryPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Box representation
 * The box is axis aligned and its origin is at the center of the box.
 */
class MIRA_RIGID_MODEL_EXPORT Box : public Geometry
{
	MIRA_OBJECT(Box)
public:

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Size", size, "The size of the box");
	}

	/// Implementation of Geometry::getFootprint
	virtual Polygon2f getFootprint(const RigidTransform3f& origin) const;
	/// Implementation of Geometry::getBoundingBox
	virtual Box3f getBoundingBox(const RigidTransform3f& origin) const;

	Size3f size;///< The size of the box
};

typedef boost::shared_ptr<Box> BoxPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Cone representation
 * The origin of the cone is located at the center of its base circle. The
 * apex is located at 'length' on the z-axis, i.e. the cones axis coincides
 * with the z-axis.
 */
class MIRA_RIGID_MODEL_EXPORT Cone : public Geometry
{
	MIRA_OBJECT(Cone)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Radius", radius, "The radius of the cone");
		r.property("Length", length, "The length of the cone");
	}

	/// Implementation of Geometry::getFootprint
	virtual Polygon2f getFootprint(const RigidTransform3f& origin) const;
	/// Implementation of Geometry::getBoundingBox
	virtual Box3f getBoundingBox(const RigidTransform3f& origin) const;

	float radius;///<The radius of the cone
	float length;///<The length of the cone
};

typedef boost::shared_ptr<Cone> ConePtr;

//////////////////////////////////////////////////////////////////////////////

/**
 * Cylinder representation
 * The origin is in the center of the cylinder.
 * The cylinder axis coincides with the z-axis.
 */
class MIRA_RIGID_MODEL_EXPORT Cylinder : public Geometry
{
	MIRA_OBJECT(Cylinder)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Radius", radius, "The radius of the cylinder");
		r.property("Length", length, "The length of the cylinder");
	}

	/// Implementation of Geometry::getFootprint
	virtual Polygon2f getFootprint(const RigidTransform3f& origin) const;
	/// Implementation of Geometry::getBoundingBox
	virtual Box3f getBoundingBox(const RigidTransform3f& origin) const;

	float radius;///<The radius of the cylinder
	float length;///<The length of the cylinder
};

typedef boost::shared_ptr<Cylinder> CylinderPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Spherical representation
 * The origin of the sphere is at its center.
 */
class MIRA_RIGID_MODEL_EXPORT Sphere : public Geometry
{
	MIRA_OBJECT(Sphere)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Radius", radius, "The radius of the sphere");
	}

	/// Implementation of Geometry::getFootprint
	virtual Polygon2f getFootprint(const RigidTransform3f& origin) const;
	/// Implementation of Geometry::getBoundingBox
	virtual Box3f getBoundingBox(const RigidTransform3f& origin) const;

	float radius;///<The radius of the sphere
};

typedef boost::shared_ptr<Sphere> SpherePtr;

//////////////////////////////////////////////////////////////////////////////

/**
 * Mesh representation
 * The vertices of the mesh are given relative to its origin.
 */
class MIRA_RIGID_MODEL_EXPORT Mesh : public Geometry
{
	MIRA_OBJECT(Mesh)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Filename", filename, "The filename of the mesh");
		r.property("Scale", scale, "The scale factor of the mesh", 1.0f);
	}

	/// Implementation of Geometry::getFootprint
	virtual Polygon2f getFootprint(const RigidTransform3f& origin) const;
	/// Implementation of Geometry::getBoundingBox
	virtual Box3f getBoundingBox(const RigidTransform3f& origin) const;

	std::string filename;///<The filename of the mesh
	float scale;///<The scale factor of the mesh
};

typedef boost::shared_ptr<Mesh> MeshPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Polygon representation
 * The vertices of the polygon are given relative to its origin.
 */
class MIRA_RIGID_MODEL_EXPORT Polygon : public Geometry
{
	MIRA_OBJECT(Polygon)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Polygon", polygon, "The polygon geometry");
	}

	/// Implementation of Geometry::getFootprint
	virtual Polygon2f getFootprint(const RigidTransform3f& origin) const;
	/** 
	 * The polygon is 2D, hence its bounding box would be 2D too, which is
	 * is still empty in 3D.
	 * @return always an invalid empty box
	 */
	virtual Box3f getBoundingBox(const RigidTransform3f& origin) const;

	/**
	 * Returns the minimum radius of the polygon that describes a circle that
	 * touches the innermost point of the polygon.
	 */
	float getMinRadius() const
	{
		Rect2f rect;
		boost::geometry::envelope(polygon, rect);
		float minX = std::min(std::abs(rect.minCorner.x()), std::abs(rect.maxCorner.x()));
		float minY = std::min(std::abs(rect.minCorner.y()), std::abs(rect.maxCorner.y()));
		return std::min(minX, minY);
	}

	/**
	 * Returns the maximum radius of the polygon that describes a circle that
	 * touches the outermost point of the polygon.
	 */
	float getMaxRadius() const
	{
		Rect2f rect;
		boost::geometry::envelope(polygon, rect);
		float maxX = std::max(std::abs(rect.minCorner.x()), std::abs(rect.maxCorner.x()));
		float maxY = std::max(std::abs(rect.minCorner.y()), std::abs(rect.maxCorner.y()));
		return hypotf(maxX, maxY);
	}

	Polygon2f polygon; ///<The polygon geometry
};

typedef boost::shared_ptr<Polygon> PolygonPtr;

//////////////////////////////////////////////////////////////////////////////

}}

#endif
