/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AnglePropertyDelegate.h
 *    A delegate for angle properties.
 *
 * @author Tim Langner
 * @date   2011/03/29
 */

#ifndef _MIRA_ANGLEPROPERTYDELEGATE_H_
#define _MIRA_ANGLEPROPERTYDELEGATE_H_

#include <QDoubleSpinBox>

#include <widgets/PropertyEditor.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_AngleSpinBoxBase : public QDoubleSpinBox
{
	Q_OBJECT
public:
	PropertyDelegate_AngleSpinBoxBase(PropertyNode* iProperty, QWidget* parent) :
		QDoubleSpinBox(parent), property(iProperty) 
	{
		setFrame(false);
	}

public slots:
	virtual void slotSetValue(double value) = 0;

public:
	PropertyNode* property;
};

///////////////////////////////////////////////////////////////////////////////

template <typename AngleType>
class PropertyDelegate_AngleSpinBox : public PropertyDelegate_AngleSpinBoxBase
{
public:
	PropertyDelegate_AngleSpinBox(PropertyNode* iProperty, QWidget* parent) :
		PropertyDelegate_AngleSpinBoxBase(iProperty, parent) 
	{
		connect(this, SIGNAL(valueChanged(double)),
		        this, SLOT(slotSetValue(double)));
		setMinimum(AngleType::convertToSerialized(AngleType::lower()));
		setMaximum(AngleType::convertToSerialized(AngleType::upper()));

		if(AngleType::turn()>10)
			setSingleStep(1);
		else
			setSingleStep(0.1);

		const TypedPropertyNode<AngleType>* p = property->toTyped<AngleType>();
		assert(p!=NULL);
		setValue((double)p->get().serializedValue());
	}

	virtual void slotSetValue(double value)
	{
		TypedPropertyNode<AngleType>* p = property->toTyped<AngleType>();
		AngleType angle;
		angle.setSerializedValue(value);
		p->set(angle);
	}
};

///////////////////////////////////////////////////////////////////////////////

template <typename AngleType>
class PropertyDelegate_Angle : public PropertyEditor::Delegate
{
public:
	virtual SupportedTypes supportedTypes() const
	{
		return makeSupportedType<AngleType>(SupportedType::TEXT |
		                                    SupportedType::EDITOR);
	}

	virtual std::string getText(const PropertyNode* property)
	{
		const TypedPropertyNode<AngleType>* p = property->toTyped<AngleType>();
		assert(p!=NULL);
		return toString(p->get().serializedValue());
	}
	
	virtual QWidget* createEditor(PropertyNode* property, QWidget* parent)
	{
		return new PropertyDelegate_AngleSpinBox<AngleType>(property, parent);
	}
};

///////////////////////////////////////////////////////////////////////////////

#define MIRA_PROPERTY_ANGLE(type)                                         \
class PropertyDelegate_##type : public PropertyDelegate_Angle<type>       \
{                                                                         \
	MIRA_OBJECT(PropertyDelegate_##type)                                  \
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif
