/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LightSource.C
 *    Additional light source that can be added to the scene.
 *
 * @author Erik Einhorn
 * @date   2014/07/08
 */

#include <OGRE/OgreLight.h>
#include <OGRE/OgreSceneManager.h>

#include <visualization/Visualization3D.h>

#include <serialization/SetterNotify.h>
#include <serialization/DefaultInitializer.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>

using namespace mira;

namespace mira { 

///////////////////////////////////////////////////////////////////////////////

/**
 * Additional light source that can be added to the scene.
 */
class LightSource : public Visualization3D
{
MIRA_META_OBJECT(LightSource, // TODO: add correct name, category and description
			("Name", "Light Source")
//			("Category", "Geomtery")
			("Description", "Additional light source that can be added to the scene"))


public:

	LightSource() : mLight(NULL), mSpotlightParams(this)
	{
		MIRA_INITIALIZE_THIS;
	}

	virtual ~LightSource()
	{
		if(mLight)
			mSceneManager->destroyLight(mLight);
	}

	struct Pose
	{
		Pose() : x(0.0f), y(0.0f), z(0.0f), yaw(0.0f), pitch(0.0f) {}


		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.property("X", x, "");
			r.property("Y", y, "");
			r.property("Z", z, "");
			r.property("Yaw", yaw, "", PropertyHints::step(10.0f));
			r.property("Pitch", pitch, "", PropertyHints::step(10.0f));
		}

		float x,y,z,yaw,pitch;
	};

	struct SpotlightParams
	{
		SpotlightParams(LightSource* l) : ls(l) {
			outerAngle=45.0f;
			falloff=1.0f;
		}

		template<typename Reflector>
		void reflect(Reflector& r)
		{
			//r.property("Inner Angle", x, "Angle covered by the bright inner cone [deg]", );
			r.property("Cone Angle", outerAngle, setterNotify(outerAngle, &LightSource::changedSpotlight, ls), "Angle covered by the splotlights cone ", 45.0f, PropertyHints::step(10.0f) | PropertyHints::limits(0.0f, 180.0f));
			r.property("Falloff", falloff, setterNotify(falloff, &LightSource::changedSpotlight, ls), "The rate of falloff within the splotlight cone. 1.0 means a linear falloff, less means slower falloff, higher means faster falloff.", 1.0f);
		}

		//float innerAngle; // NOTE: The inner cone applicable only to Direct3D, it'll always treat as zero in OpenGL, therefore its left out here
		float outerAngle;
		float falloff;

		LightSource* ls;
	};


	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization3D);
		r.property("Color", mLightColor, setterNotify(mLightColor, &LightSource::changedLight, this), "The color of the default light", Ogre::ColourValue::White);
		r.property("Pose", mPose, "The position and direction of the light source", Pose());
		r.property("Type", mType, setterNotify(mType, &LightSource::changedLight, this), "The type of the light source", Ogre::Light::LT_POINT, PropertyHints::enumeration("Point Light;Directional;Spotlight"));
		r.property("Spotlight", mSpotlightParams, "Parameters for spotlight", SpotlightParams(this));
	}

	virtual void setupScene(IVisualization3DSite* site)
	{
		mSceneManager = site->getSceneManager();
		mLight = mSceneManager->createLight();
		refreshLight();
	}

	virtual Ogre::SceneNode* getNode()
	{
		return NULL;
	}

	void setEnabled(bool enabled) {
		Visualization3D::setEnabled(enabled);
		if(!enabled && mLight) {
			mSceneManager->destroyLight(mLight);
			mLight=NULL;
		}
		if(enabled && !mLight) {
			mLight = mSceneManager->createLight();
			refreshLight();
		}
	}

	void changedLight() {
		if(!mLight)
			return;

		mLight->setType(mType);
		mLight->setDiffuseColour(mLightColor);
		mLight->setSpecularColour(mLightColor);
	}

	void changedSpotlight() {
		if(!mLight)
			return;
		if(mLight->getType()!=Ogre::Light::LT_SPOTLIGHT)
			return;
		mLight->setSpotlightRange(Ogre::Radian(0.0f), Ogre::Degree(mSpotlightParams.outerAngle), mSpotlightParams.falloff);
	}

	void refreshLight() {
		changedLight();
		changedSpotlight();
	}

	virtual void update(Duration dt) {
		/*RigidTransform3f d = getAuthority().template getTransform<RigidTransform3f>(
				mDataFrameID, mDataTimestamp,
				getSite()->getCameraFrame(), Time(),
				getSite()->getFixedFrame());*/

		//OgreUtils::setTransform(mNode, d);

		if(!mLight)
			return;

		// default light direction is along x-axis
		// this direction can be altered by the yaw and pitch parameters of the pose
		Eigen::Quaternionf q = quaternionFromYawPitchRoll(deg2rad(mPose.yaw), deg2rad(mPose.pitch), 0.0f);
		Eigen::Vector3f dir = q * Eigen::Vector3f::UnitX();

		Eigen::Vector3f pos(mPose.x, mPose.y, mPose.z);

		// TODO: transform according to optinal frame property




		mLight->setPosition(Ogre::Vector3(pos.x(), pos.y(), pos.z()));
		mLight->setDirection(Ogre::Vector3(dir.x(), dir.y(), dir.z()));
	}


protected:

	Ogre::SceneManager* mSceneManager;
	Ogre::Light* mLight;

	Ogre::ColourValue mLightColor;
	Pose mPose;
	Ogre::Light::LightTypes mType;

	SpotlightParams mSpotlightParams;



};

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::LightSource, mira::Visualization3D );
