#!/bin/bash

###############################################################################
#
# A bash script, which allows to create a source code distribution package
# of a specific MIRA package into as ZIP file.
#
# Usage:  createSrcPackage.sh PathToPackage PackageName
#
#     PathToPackage:  The path to package directory
#     PackageName:    The name of the package (without .package extension!)
#
# If the file $PathToPackage/source-distribution.exclude exists, the content of
# this file will be exclude from the ZIP file. All other files in $PathToPackage
# will be recursively package into the ZIP file. This ZIP file will also
# contain the name of the parent directory of the package file.
#
# Author: Christian Martin
#
###############################################################################
# Extract and check the command line arguments

if [ $# != 2 ]; then
	echo "Usage: $0 PathToPackage PackageName"
	exit
fi

SRCDIR=$1
PKGNAME=$2

if [ ! -e $SRCDIR ]; then
	echo "The directory $SRCDIR does not exist. Abort!"
	exit
fi

if [ ! -f $SRCDIR/$PKGNAME.package ]; then
	echo "The file $SRCDIR/$PKGNAME.package does not exist. Abort!"
	exit
fi

# Extract the version number from the package file
PKGVER=`cat $SRCDIR/$PKGNAME.package | grep MIRA_VERSION | \
	sed -r "s/MIRA_VERSION\([ \t]*([0-9]*)[ \t]*([0-9]*)[ \t]*([0-9]*)\)/\1.\2.\3/g"`

###############################################################################
# Check if the destination names already exits

DEST_DIR=`pwd`/$PKGNAME-$PKGVER
mkdir -p $DEST_DIR

DEST_ZIPNAME=$DEST_DIR/$PKGNAME.zip
DEST_PKGNAME=$DEST_DIR/$PKGNAME.package

if [ -f $DEST_ZIPNAME ]; then
	echo "The file $DEST_ZIPNAME already exists. Abort!"
	exit
fi

if [ -f $DEST_PKGNAME ]; then
	echo "The file $DEST_PKGNAME already exists. Abort!"
	exit
fi

###############################################################################
# zip all files together

THISDIR=`pwd`

ZIP_WRK_DIR=`dirname $SRCDIR`
ZIP_ROOT_NAME=`basename $SRCDIR`

cd $ZIP_WRK_DIR

if [ -f $SRCDIR/source-distribution.exclude ]; then
	# read ignore list and add base directory to it
	IGNORE_FILE=/tmp/.mira.ignore.lst
	awk -v n="$ZIP_ROOT_NAME" '{ print n"/"$1 }' $SRCDIR/source-distribution.exclude > $IGNORE_FILE

	# Add the ignore file itselt to the ignore list
	echo "*source-distribution.exclude*" >> $IGNORE_FILE

	# zip all files
	zip -r -9 -q $DEST_ZIPNAME $ZIP_ROOT_NAME -x@$IGNORE_FILE -x "*.svn*"

	# delete temporary ignore list
	rm -f $IGNORE_FILE
else
	zip -r -9 -q $DEST_ZIPNAME $ZIP_ROOT_NAME -x "*.svn*"
fi

cd $THISDIR

###############################################################################
# Finalize

cp $SRCDIR/$PKGNAME.package $DEST_DIR
cp $SRCDIR/$PKGNAME.changelog $DEST_DIR

echo "Generated $DEST_ZIPNAME"
echo "Generated $DEST_PKGNAME"
