/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationTextJSON.h
 *    Macros for easy creation of JSON based text visualizations.
 *
 * @author Tim Langner
 * @date   2011/02/09
 */

#ifndef _MIRA_VISUALIZATIONTEXTJSON_H_
#define _MIRA_VISUALIZATIONTEXTJSON_H_

#include <QInputDialog>

#include <serialization/Serialization.h>
#include <serialization/SetterNotify.h>

#include <utils/ToString.h>
#include <visualization/VisualizationTextBasic.h>

#include <fw/Framework.h>
#include <fw/ChannelManager.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template<typename T>
class VisualizationTextBase : public VisualizationTextBasic<T>
{
public:
	typedef VisualizationTextBasic<T> Base;

	VisualizationTextBase(const std::string& channelName) :
		Base(channelName)
	{}

	virtual void dataChanged(ChannelRead<T> data)
	{
		this->mItem->setText(toString(data->value(), this->mPrecision /*, true - to be implemented*/));
		this->mItem->template setStamped<T>(*data);
	}

	virtual std::string defaultText()
	{
		return toString(T(), this->mPrecision/*, true*/);
	}
};

/**
 * Use this macro to create a text visualization for a data type that
 * supports toString(DataType) (is a atomic/build in type or supports the 
 * ostream& operator<<(ostream& s) operator.
 * className specifies the class name
 * type is the data type of the channel
 * channel is the name of the channel property
 * name is the name of the visualization
 * description describes the visualization
 * category is the category of the visualization
 */
#define MIRA_TEXT_VISUALIZATION(className, type, channel, name, description, category) \
class className : public VisualizationTextBase<type>						\
{																			\
	MIRA_META_OBJECT(className,												\
			("Name", name)													\
			("Description", description)									\
			("Category", category))											\
public:																		\
	className() :															\
		VisualizationTextBase<type>(channel)								\
	{}																		\
};

///////////////////////////////////////////////////////////////////////////////

template<typename T>
class VisualizationTextJSONBase : public VisualizationTextBasic<T>
{
public:
	typedef VisualizationTextBasic<T> Base;

	VisualizationTextJSONBase(const std::string& channelName) :
		Base(channelName), mShowROProperties(true), mStringMapAsObject(true)
	{}

	template <typename Reflector>
	void reflect(Reflector& r) {
		MIRA_REFLECT_BASE(r, Base);
		r.property("ShowReadOnlyProperties", mShowROProperties,
		           setterNotify(mShowROProperties, &VisualizationTextJSONBase<T>::redraw, this),
		           "Show objects' read-only properties", true);
		r.property("StringMapAsObject", mStringMapAsObject,
		           setterNotify(mStringMapAsObject, &VisualizationTextJSONBase<T>::redraw, this),
		           "Serialize any map<string, T> to json object "
		           "(enables simple access to map items by key)", true);
	}

	virtual void setupScene(IVisualizationTextItem* item)
	{
		item->registerTextEditRequestCallback(boost::bind(&VisualizationTextJSONBase<T>::onEdit, this));
		item->enableEdit(true);
	}

	virtual void dataChanged(ChannelRead<T> data)
	{
		this->mItem->template setStamped<T>(*data);

		try
		{
			json::Value v;
			try {
				JSONSerializer js(mShowROProperties,
				                  this->mStringMapAsObject ? JSONSerializer::STRING_MAP_AS_OBJECT
				                                           : JSONSerializer::STANDARD);
				data.readJSON(v, js);
			}
			catch(XNotImplemented&)
			{
				data.readJSON(v);
			}
			this->mLastValue.reset(v);
			this->mItem->setText(json::write(v,true,this->mPrecision));
		}
		catch(Exception& ex)
		{
			int typeID = MIRA_FW.getChannelManager().getTypeId(Base::mChannel.getID());
			if (typeID < 0)
				this->mItem->setText(MakeString() << "Conversion to JSON format (from untyped channel's "
				                     "serialized data + metadata) failed:\n" << ex.message());
			else
				this->mItem->setText(MakeString() << "Serialization to JSON format not supported by "
				                     "channel type:\n" << ex.message());
		}
		catch(...)
		{
			int typeID = MIRA_FW.getChannelManager().getTypeId(Base::mChannel.getID());
			if (typeID < 0)
				this->mItem->setText("Conversion to JSON format (from untyped channel's "
				                     "serialized data + metadata) failed");
			else
				this->mItem->setText("Serialization to JSON format not supported by channel type");
		}
	}

	virtual std::string defaultText()
	{
		JSONSerializer js(mShowROProperties,
		                  this->mStringMapAsObject ? JSONSerializer::STRING_MAP_AS_OBJECT
		                                           : JSONSerializer::STANDARD);
		json::Value value = js.serialize(T());
		return json::write(value,true,this->mPrecision);
	}

	virtual void redraw()
	{
		try {
			if (Base::mChannel.isValid())
				dataChanged(this->mChannel.getChannel().read());
		} catch (...) {}
	}

protected:
	bool mShowROProperties;
	bool mStringMapAsObject;
};

/**
 * Use this macro to create a  text visualization for a data type that
 * supports serialization using JSON serializer.
 * className specifies the class name
 * type is the data type of the channel
 * channel is the name of the channel property
 * name is the name of the visualization
 * description describes the visualization
 * category is the category of the visualization
 */
#define MIRA_JSON_VISUALIZATION(className, type, channel, name, description, category) \
class className : public VisualizationTextJSONBase<type>					\
{																			\
	MIRA_META_OBJECT(className,												\
			("Name", name)													\
			("Description", description)									\
			("Category", category))											\
public:																		\
	className() :															\
		VisualizationTextJSONBase<type>(channel)							\
	{}																		\
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
