/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DynamicPoints.h
 *    Class for displaying dynamic points using vertex buffer.
 *    Adapted from http://www.ogre3d.org/tikiwiki/DynamicLineDrawing&structure=Cookbook
 *
 *    Original file: DynamicLines.h
 */

#ifndef _MIRA_DYNAMICPOINTS_H_
#define _MIRA_DYNAMICPOINTS_H_

#include <visualization/3d/DynamicRenderable.h>
#include <vector>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

class MIRA_GUI_VISUALIZATION_EXPORT DynamicPoints : public DynamicRenderable
{
	typedef Ogre::Vector3 Vector3;
	typedef Ogre::Quaternion Quaternion;
	typedef Ogre::Camera Camera;
	typedef Ogre::Real Real;
	typedef Ogre::RenderOperation::OperationType OperationType;

public:

	struct PointData
	{
		Vector3 point;
		Ogre::ColourValue color;
	};
	/// Constructor - see setOperationType() for description of argument.
	DynamicPoints(OperationType opType = Ogre::RenderOperation::OT_POINT_LIST);
	virtual ~DynamicPoints();

	/// Add a point to the point list
	void addPoint(const Ogre::Vector3 &p, const Ogre::ColourValue& c);
	/// Add a point to the point list
	void addPoint(Real x, Real y, Real z, const Ogre::ColourValue& c);

	/// Change the location of an existing point in the point list
	void setPoint(unsigned short index, const Vector3 &value);

	/// Return the location of an existing point in the point list
	const Vector3& getPoint(unsigned short index) const;

	/// Change the colour of existing points
	void setColor(const Ogre::ColourValue& c);

	/// Return the total number of points in the point list
	unsigned short getNumPoints(void) const;

	/// Remove all points from the point list
	void clear();

	/// Call this to update the hardware buffer after making changes.
	void update();

	/** Set the type of operation to draw with.
	 * @param opType Can be one of
	 *    - RenderOperation::OT_LINE_STRIP
	 *    - RenderOperation::OT_LINE_LIST
	 *    - RenderOperation::OT_POINT_LIST
	 *    - RenderOperation::OT_TRIANGLE_LIST
	 *    - RenderOperation::OT_TRIANGLE_STRIP
	 *    - RenderOperation::OT_TRIANGLE_FAN
	 *    The default is OT_POINT_LIST.
	 */
	void setOperationType(OperationType opType);
	OperationType getOperationType() const;

private:
	virtual void fillHardwareBuffers();

private:
	std::vector<PointData> mPoints;
	bool mDirty;
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
