/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Visualization3DView.h
 *    Declaration of the Visualization3DView.
 *
 * @author Erik Einhorn
 * @date   2010/12/01
 */

#ifndef _MIRA_VISUALIZATION3DVIEW_H_
#define _MIRA_VISUALIZATION3DVIEW_H_

#include <geometry/Point.h>

#include <views/VisualizationView.h>
#include <visualization/Visualization3D.h>
#include <visualization/InteractionListener3D.h>

#include <serialization/adapters/OGRE/OgreColourValue.h>

#include <serialization/SetterNotify.h>

class QGLWidget;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * 3D view of a scene that may contain different visualization objects which
 * show the actual content of the scene.
 */
class MIRA_GUI_VIEWS_EXPORT Visualization3DView : public VisualizationViewTransformable, public IVisualization3DSite
{
MIRA_META_OBJECT(Visualization3DView,
				("Name", "3D")
				("Description", "3D view for channel data")
				("Category"   , "Visualization")
				("Visualization", "mira::Visualization3D"))
Q_OBJECT

public:
	Visualization3DView();

	virtual ~Visualization3DView()
	{
		// we must make sure that we destroy our tools and visualizations
		// before we destroy ourself to give them the change to cleanup
		// (during cleanup they may need a fully operational Visualization3DView)
		destroyVisualizations();
		destroyTools();
	}

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, VisualizationViewTransformable);
		r.property("Camera Mode", mCameraMode,
		           setter(&Visualization3DView::setCameraMode, this),
		           "The view mode of the camera", PERSPECTIVE,
		           PropertyHints::enumeration("Perspective Orbit;Orthographic Orbit;Bird's-Eye"));

		r.property("Camera", mCameraSettings, "Camera view", Camera());

		r.property("Background", mBackgroundMode, setter(&Visualization3DView::setBackgroundMode, this), "Background mode", BACKGROUND_GRADIENT,
		           PropertyHints::enumeration("Solid;Gradient;White;Black"));
		r.property("Background Color", mBackgroundColor1, setter(&Visualization3DView::setBackgroundColor1, this), "Solid background color / Top gradient color", Ogre::ColourValue(0.78f,0.86f,1.0f) );
		r.property("Background Gradient Color", mBackgroundColor2, setter(&Visualization3DView::setBackgroundColor2, this), "Bottom gradient color", Ogre::ColourValue(0.90f,0.94f,1.0f) );

		r.property("Render Mode", mRenderMode,
		           setter(&Visualization3DView::setRenderMode, this),
		           "The view mode of the camera", NORMAL,
		           PropertyHints::enumeration("Normal;Wireframe;Wireframe (hidden line)"));

		r.property("Stereo Mode", mStereoMode,
		           setter(&Visualization3DView::setStereoMode, this),
		           "The view mode of the camera", NONE,
		           PropertyHints::enumeration("None;Cross-eyed / parallel"));
		r.property("Stereo Eye Distance", mStereoEyeDistance, "in meter, >0 for cross view, <0 for parallel view", 0.2f);

		r.property("Show Bounding Boxes", mShowBoundingBoxes, setter(&Visualization3DView::showBoundingBoxes, this), "Show bounding boxes of all objects (for debugging)", false);

		r.property("Light Color", mLightColor, setterNotify(mLightColor, &Visualization3DView::changeLightColor, this), "The color of the default light", Ogre::ColourValue::White);
		r.property("Light Azimuth", mLightAzimuth, setterNotify(mLightAzimuth, &Visualization3DView::changeLightDirection, this), "The azimuth of the light source [deg]", -135.0f, PropertyHints::limits(-180.0f,180.0f) | PropertyHints::step(5.0f));
		r.property("Light Height", mLightHeight, setterNotify(mLightHeight, &Visualization3DView::changeLightDirection, this), "The height of the light source [deg]", 45.0f, PropertyHints::limits(-90.0f,90.0f) | PropertyHints::step(5.0f));
		r.property("Ambient Light", mLightAmbient, setterNotify(mLightAmbient, &Visualization3DView::changeAmbientLight, this), "The strength of the ambient light", 0.3f, PropertyHints::minimum(0.0f));

		r.property("SaveContentMinPictureSize", mSaveContentMinPictureSize,
				   "Minimum width or height of the image saved by rendering the viewport at an appropriate resolution.", 2048);
	}

public:

	/// Returns the GLWidget where this view renders its content (may be NULL if the view is uninitialized)
	QGLWidget* getGLRenderWidget();

	virtual QImage captureContent();

public: // implementation of VisualizationView

	virtual void addVisualization(Visualization* vis);
	virtual void moveUpVisualization(Visualization* vis);
	virtual void moveDownVisualization(Visualization* vis);

	virtual void addTool(VisualizationTool* tool);

	virtual const Class& supportedVisualizationClass() const;
	virtual const Class& supportedVisualizationToolClass() const;

protected:

	virtual void update(Duration dt);

	void changeLightColor();
	virtual void changeLightDirection();
	void changeAmbientLight();

protected:

	virtual QWidget* createVisualizationPart();

	virtual void resetView();
	virtual void saveContentToFile();

public: // implementation of IVisualization3DSite

	virtual Ogre::SceneManager* getSceneManager();

	virtual void registerInteractionListener(InteractionListener3D* listener);
	virtual void removeInteractionListener(InteractionListener3D* listener);
	virtual std::set<InteractionListener3D*> getInteractionListeners();

	virtual const std::string& getFixedFrame() const {
		return mFixedFrame.getID();
	}

	virtual const std::string& getCameraFrame() const {
		return mCameraFrame.getID();
	}

public:

	virtual const Camera& getCamera() const;
	virtual void setCamera(const Camera& camera);

	virtual Ogre::Camera* acquireCamera(boost::function<void()> lostCameraCallback = boost::function<void()>());
	virtual void releaseCamera();

private:

	class RenderModeTechniqueMod;
	class UI;
	UI* ui;

	class RenderQueueSorter;

private:
	friend class UI;
	friend class RenderModeTechniqueMod;
	friend class RenderQueueSorter;

	void setupVisualizations();
	void setupTools();

	void updateCamera();

private:
	void populateDefaultVisualizations();
	void createDefaultVisualization(const std::string& className, const std::string& displayName);

private:
	enum CameraMode
	{
		PERSPECTIVE=0,
		ORTHOGRAPHIC,
		BIRDS_EYE
	};

	enum StereoMode
	{
		NONE,
		CROSSED_EYE
	};

	enum BackgroundMode
	{
		BACKGROUND_SOLID=0,
		BACKGROUND_GRADIENT,
		BACKGROUND_WHITE,
		BACKGROUND_BLACK
	};

	enum RenderMode
	{
		NORMAL = 0,
		WIREFRAME,
		HIDDEN_LINE
	};

	void setCameraMode(CameraMode mode);

	void setBackgroundMode(BackgroundMode mode);
	void setBackgroundColor1(Ogre::ColourValue color);
	void setBackgroundColor2(Ogre::ColourValue color);
	void showBoundingBoxes(bool show=true);

	void setRenderMode(RenderMode mode);
	void setStereoMode(StereoMode mode);

private:

	ChannelProperty<float> mIntrinsicParamsChannel;

	CameraMode mCameraMode;
	Camera mCameraSettings;

	bool mCameraAcquired;
	boost::function<void()> mLostCameraCallback;

	BackgroundMode mBackgroundMode;
	Ogre::ColourValue mBackgroundColor1;
	Ogre::ColourValue mBackgroundColor2;

	RenderMode mRenderMode;
	StereoMode mStereoMode;
	float mStereoEyeDistance;

	bool mShowBoundingBoxes;

	Ogre::ColourValue mLightColor;
	float mLightAzimuth;
	float mLightHeight;
	float mLightAmbient;
	boost::mutex mListenerMutex;
	std::set<InteractionListener3D*> mListeners;

	uint mSaveContentMinPictureSize;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
