/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

#include <pluginlib/class_list_macros.h>
#include <nodelet/nodelet.h>
#include <ros/ros.h>

#include <boost/thread.hpp>
#include <boost/bind.hpp>

#include <Benchmark/Data.h>

#include "../../PerfStats.h"

namespace benchmark {
using namespace Benchmark;

class BenchmarkPublisher : public nodelet::Nodelet
{
public:
	BenchmarkPublisher() : count(0), total(0)
	{
		// skip first 10 measurements
		first=10;
	}

	~BenchmarkPublisher()
	{
		std::cout << "stopping" << std::endl;
		std::cout << "Publishing Average: " << total/count << std::endl;
		thread.interrupt();
		thread.join();
	}
private:
	virtual void onInit()
	{
		ros::NodeHandle& private_nh = getPrivateNodeHandle();
		private_nh.getParam("N", N);

		ros::NodeHandle& nh = getNodeHandle();
		pub = nh.advertise< Data>("Data", 10);
		thread = boost::thread(boost::bind(&BenchmarkPublisher::process, this));
	}

	void process()
	{
		ros::Duration(2).sleep(); // make sure all libs are loaded and mem stats are stable

		memStart = getMemoryUsage();
		std::cout << "Publisher Start: " << memStart << std::endl;

		ros::Rate r(10);
		while (!boost::this_thread::interruption_requested())
		{
			uint64_t memDiff1 = getMemoryUsage() - memStart;
			std::cout << "Publisher Memory1: " << memDiff1 << std::endl;

			uint64_t start = getCycleCount();

			// we need to create a new message to handle concurrent access
			// (note that the subscriber could still read the data in the
			//  shared pointer)
			Data::Ptr msg(new Data);
			msg->values.resize(N*1024);


			msg->rdtsc = start;
			pub.publish(msg);

			// skip first measurements
			if(first>0) {
				--first;
			} else {
				uint64_t d = getCycleCount() - start;
				std::cout << "Publishing took: " << d << std::endl;
				total += d;
				++count;
			}


			uint64_t memDiff2 = getMemoryUsage() - memStart;
			std::cout << "Publisher Memory2: " << memDiff2 << std::endl;

			r.sleep();
		}

	}

	int N;
	ros::Publisher pub;
	boost::thread thread;

	int count;
	uint64_t total;
	int first;

	uint64_t memStart;
};

PLUGINLIB_DECLARE_CLASS(benchmark, BenchmarkPublisher, benchmark::BenchmarkPublisher, nodelet::Nodelet);



class BenchmarkSubscriber : public nodelet::Nodelet
{
public:
	BenchmarkSubscriber() : count(0), total(0)
	{
		// skip first 10 measurements
		first=10;
	}

	~BenchmarkSubscriber()
	{
		std::cout << "Latency Average: " << total/count << std::endl;
	}
private:
	virtual void onInit()
	{
		std::cout << "onInit" << std::endl;
		ros::NodeHandle& nh = getNodeHandle();
		sub = nh.subscribe("Data", 10, &BenchmarkSubscriber::callback, this);
	}

	void callback(const Data::Ptr& msg)
	{
		// skip first measurements
		if(first>0) {
			--first;
			return;
		}

		uint64_t d = getCycleCount() - msg->rdtsc;
		std::cout << "Transport took: " << d << std::endl;
		total += d;
		++count;
	}

	int count;
	uint64_t total;
	int first;
	ros::Subscriber sub;
};

PLUGINLIB_DECLARE_CLASS(benchmark, BenchmarkSubscriber, benchmark::BenchmarkSubscriber, nodelet::Nodelet);
}

