/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapePlayer.C
 *    Implementation of TapePlayer.h.
 *
 * @author Tim Langner
 * @date   2011/01/12
 */

#include <fw/TapePlayer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

TapePlayer::TapePlayer() :
	mAuthority("/", "TapePlayer", Authority::ANONYMOUS | Authority::INTERNAL),
	mVisitor(NULL)
{
	mLoopCount = -1;
	mLoop = false;
	mPlayImmediately = false;
	mTimeScaler = 1.0f;
	mIsPlaying = false;
	mPause = true;
}

void TapePlayer::stepTo(Duration time)
{
	boost::mutex::scoped_lock lock(mConditionMutex);
	mStepTo = time;
	mStepToChannel.reset();
	mPlayImmediately = true;
	mPause = false;
	mCondition.notify_one();
}

void TapePlayer::stepTo(const std::string& channelID)
{
	TapeVisitor::iterator i = mVisitor->findNextOf(getCurrentMessageTimeOffset(), channelID);

	boost::mutex::scoped_lock lock(mConditionMutex);
	if (i!=mVisitor->end())
		mStepToChannel = i;
	else
		// no next message found. play until end.
		mStepToChannel.reset();

	mStepTo.reset();
	mPlayImmediately = true;
	mPause = false;
	mCondition.notify_one();
}

void TapePlayer::load(TapeVisitor* visitor, Time startTime, const std::string& prefix)
{
	stop();
	mPause = true;
	mVisitor = visitor;
	mStartTime = startTime;
	mNamespacePrefix = prefix;
	const TapeVisitor::ChannelMap& channels = mVisitor->getChannels();
	foreach(auto c, channels)
	{
		Channel<void> channel = mAuthority.publish<void>(mNamespacePrefix + c.first, c.second->type);
		if (c.second->meta) {
			// if the channel is typed, the correct meta will be (or has already been) created when writing to it!
			if (!channel.isTyped()) {
				// warn when changing meta of untyped channel
				if (channel.getTypeMeta()) {
					std::string oldMeta = channel.getTypeMeta()->toString();
					std::string newMeta = c.second->meta->toString();
					if (oldMeta != newMeta) {
						MIRA_LOG(WARNING) << "TapePlayer: Changing type meta for untyped channel " << mNamespacePrefix + c.first
						                  << " from " << oldMeta << " to " << newMeta << std::endl;
					}
				}
				channel.setTypeMeta(c.second->meta);
			}
		}
		MIRA_FW.getMetaDatabase()->merge(c.second->metaDB);
	}
	mIsPlaying = true;
	mMessage = mVisitor->begin();
	mThread = boost::thread(boost::bind(&TapePlayer::doPlay, this));
}

void TapePlayer::play()
{
	boost::mutex::scoped_lock lock(mConditionMutex);
	mStepTo.reset();
	mStepToChannel.reset();
	mJumpTo.reset();
	mErrorMessage.reset();
	mPlayImmediately = false;
	mPause = false;
	mCondition.notify_one();
}

void TapePlayer::jumpTo(Duration time)
{
	if (!mIsPlaying)
		return;
	boost::mutex::scoped_lock lock(mConditionMutex);
	mJumpTo = time;
	mCondition.notify_one();
}

void TapePlayer::doPlay()
{
	try
	{
		// main loop for repeating the whole visited tapes in loop mode
		while(!boost::this_thread::interruption_requested())
		{
			Duration lastMessageTime = Duration::nanoseconds(0);
			boost::system_time nextMessageTime = boost::get_system_time();

			// loop over the whole visited tapes
			while (!boost::this_thread::interruption_requested() &&
				   mMessage != mVisitor->end())
			{
				boost::mutex::scoped_lock lock(mConditionMutex);
				// we fall through the pause loop if the user requests a single
				// step or step to or if we want to jump to a specific time
				while(mPause && !mStepTo && !mStepToChannel && !mJumpTo &&
					  !boost::this_thread::interruption_requested())
				{
					mCondition.wait(lock);
					nextMessageTime = boost::get_system_time();
				}

				// play all messages that have negative time offset immediately
				// (start at time offset 0)
				if (mMessage.getTimeOffset().is_negative())
				{
					mPlayImmediately = true;
					nextMessageTime = boost::get_system_time();
				}

				// if last played message had negative time offset set it to 0
				// so that we only wait for a timespan from 0 to the first message
				// with positive time offset
				if (lastMessageTime.is_negative())
				{
					lastMessageTime = Duration::nanoseconds(0);
					nextMessageTime = boost::get_system_time();
				}

				// only adapt next message time for messages that are not played immediately
				if (!mPlayImmediately)
					nextMessageTime += (mMessage.getTimeOffset() - lastMessageTime) * mTimeScaler;

				// sleep until time of next message is reached
				// only if we don't want to jump to a specific message
				// and we don't want to do a single step
				while (!mPlayImmediately && !mJumpTo && boost::get_system_time() < nextMessageTime)
					mCondition.timed_wait(lock, nextMessageTime);

				mPlayImmediately = false;
				if (mJumpTo)
				{
					mMessage = mVisitor->at(*mJumpTo);
					if (mMessage == mVisitor->end() )
					{
						mJumpTo.reset();
						break;
					}
					// we jumped back in time
					if (*mJumpTo < lastMessageTime)
						mStartTime = mStartTime + lastMessageTime-mMessage.getTimeOffset() +
									 Duration::microseconds(1);
					mJumpTo.reset();
				}

				// get the message and write it
				const TapeVisitor::MessageInstance& message = *mMessage;
				Channel<void> channel = mAuthority.getChannel<void>(mNamespacePrefix + mMessage.getChannelInfo()->name);
				ChannelWrite<void> cw = channel.write();
				cw->timestamp = mStartTime + mMessage.getTimeOffset();
				cw->frameID = message.frameID.c_str();
				cw->sequenceID = message.sequenceID;

				// try to restore data (skip it on error)
				try {
					cw.writeSerializedValue(message.data);
					lastMessageTime = mMessage.getTimeOffset();
				} catch(Exception& ex) {
					ex.addInfo(MakeString() << "Failed to read data of channel '" <<
					                           (mNamespacePrefix + mMessage.getChannelInfo()->name) << "' from tape. Skipping.",
					                           __FILE__, __LINE__);
					MIRA_LOG_EXCEPTION(ERROR, ex);
					cw.discard();
				} catch(const std::exception& ex) {
					MIRA_LOG(ERROR) << ex.what() << std::endl <<
					                   "Failed to read data of channel '" <<
					                   (mNamespacePrefix + mMessage.getChannelInfo()->name) << "' from tape. Skipping.";
					cw.discard();
				}

				// if stepto was requested check if the current message equals the end message
				// if we reached the next step set stepping mode to false
				// so we will pause on next message again
				if (mStepTo && mMessage.getTimeOffset() >= *mStepTo)
				{
					mStepTo.reset();
					mPause = true;
				}
				if (mStepToChannel && mMessage == *mStepToChannel)
				{
					mStepToChannel.reset();
					mPause = true;
				}
				++mMessage;
			}
			if (mLoop && mLoopCount > 0)
				--mLoopCount;
			if ( !mLoop || mLoopCount == 0)
			{
				mLoop = false;
				mIsPlaying = false;
				return;
			}
			// increase start time by time of last message and
			// add one microsecond to be sure to play first message of a channel in next
			// loop even if it has time offset 0 (needed for firstMessageTime != lastMessageTime)
			mStartTime += lastMessageTime + Duration::microseconds(1);
			// reset stepto to avoid endless loops
			mStepTo.reset();
			mStepToChannel.reset();
			mMessage = mVisitor->begin();
		}
	}
	catch(Exception& ex)
	{
		MIRA_LOG_EXCEPTION(ERROR, ex);
		mIsPlaying = false;
		mErrorMessage.reset(ex.message());
	}
}

///////////////////////////////////////////////////////////////////////////////

}
