/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteAuthority.C
 *    Implementation of RemoteAuthority.h.
 *
 * @author Tim Langner, Erik Einhorn
 * @date   2011/09/08
 */

#include <fw/RemoteAuthority.h>

#include <algorithm>

#include <serialization/PropertyTree.h>
#include <thread/ThreadMonitor.h>
#include <fw/Framework.h>

namespace mira {

const Duration cRPCTimeout = Duration::seconds(3);

///////////////////////////////////////////////////////////////////////////////

class RemoteAuthorityPropertyNode;

class RemoteAuthorityPropertyUpdater;
typedef boost::shared_ptr<class RemoteAuthorityPropertyUpdater> RemoteAuthorityPropertyUpdaterPtr;


class RemoteAuthorityPropertyUpdater
{
public:

	RemoteAuthorityPropertyUpdater();
	~RemoteAuthorityPropertyUpdater();

	/**
	 * Schedule an update of a remote property.
	 */
	void scheduleUpdate(const std::string& authorityID, RemoteAuthorityPropertyNode* property);

	void removeScheduledUpdates(RemoteAuthorityPropertyNode* property);

	static RemoteAuthorityPropertyUpdaterPtr instance() {
		static RemoteAuthorityPropertyUpdaterPtr inst(new RemoteAuthorityPropertyUpdater);
		return inst;
	}


private:
	void process();

private:
	boost::thread mThread;
	boost::mutex mMutex;
	boost::condition_variable mSignal;
	// queue with pairs of (Authority ID,PropertyNode)
	std::deque<std::pair<std::string,RemoteAuthorityPropertyNode*>> mPendingProperties;
};

/**
 * Implements AbstractRemotePropertyNode.
 * Each RemoteAuthorityPropertyNode instance mirrors a property of the
 * remote Authority. It delegates set/get to the remote Authority via
 * RPC calls to setPropertyJSON and getPropertyJSON.
 */
class RemoteAuthorityPropertyNode : public AbstractRemotePropertyNode
{
public:

	RemoteAuthorityPropertyNode(const PropertyNodeInfo& info) :
		AbstractRemotePropertyNode(info), authority(NULL), root(NULL)
	{
		updater = RemoteAuthorityPropertyUpdater::instance();
	}

	~RemoteAuthorityPropertyNode()
	{
		// remove all pending updates for us
		updater->removeScheduledUpdates(this);
	}

public:
	// implements AbstractRemotePropertyNode
	virtual void setFromJSON(const json::Value& node)
	{
		assert(root!=NULL);
		if (authority == NULL)
			MIRA_THROW(XIO, "Failed to set property, since remote authority is not valid.");

		std::string id = fullID(root);
		RPCFuture<void> res = MIRA_FW.getRPCManager().call<void>(authority->getGlobalID()+"#builtin",
		                                                         "setPropertyJSON", id,
		                                                         node);
		if(!res.timedWait(cRPCTimeout))
			MIRA_THROW(XIO, "Failed to set property '" << id <<
			           "' from remote authority '" << authority->getGlobalID() << "'");
	}

	virtual json::Value getAsJSON() const
	{
		assert(root!=NULL);
		if (authority == NULL)
			MIRA_THROW(XIO, "Failed to get property, since remote authority is not valid.");

		// In RemoteAuthority::getProperties() also the root node of the properties
		// will be added. But the root node is not a property. Instead it is the
		// authority itself.
		// We have to avoid doing updates on the root node. Otherwise we're getting
		// errors, since the name of the authority is used as a property name here.
		if (authority->getGlobalID() == id())
			MIRA_THROW(XIO, "Can't get property for the authority itself.")

		// Trigger an update of the property value
		RemoteAuthorityPropertyNode* This = const_cast<RemoteAuthorityPropertyNode*>(this);
		updater->scheduleUpdate(authority->getGlobalID(), This);

		// Throw an exception as long we don't have a property value.
		if (lastGetValue.is_null())
			MIRA_THROW(XIO, "Property is not available yet.")

		// Return the last known value
		return lastGetValue;
	}

	void addChild(PropertyNode* child)
	{
		RemoteAuthorityPropertyNode* c = dynamic_cast<RemoteAuthorityPropertyNode*>(child);
		assert(c!=NULL);
		c->authority = authority;
		c->root = root;
		AbstractRemotePropertyNode::addChild(child);
	}

	// Reset the authority pointer in this node and all children.
	void invalidateAuthority() {
		authority = NULL;
		foreach(auto node, this->children()) {
			RemoteAuthorityPropertyNode* p = dynamic_cast<RemoteAuthorityPropertyNode*>(node);
			if (p)
				p->invalidateAuthority();
		}
	}

public:
	RemoteAuthority* authority;
	PropertyNode* root;
	RemoteAuthorityPropertyUpdaterPtr updater;

	json::Value lastGetValue;
};
///////////////////////////////////////////////////////////////////////////////

RemoteAuthority::RemoteAuthority(const AuthorityDescription& desc)
{
	mDescription = desc;
#if	MIRA_MAJOR_VERSION(MIRA_PROTOCOL_VERSION)<=3
	mHasServiceInterfaceQueryMethod = true;
#endif
}

RemoteAuthority::~RemoteAuthority()
{
	// Since the PropertyEditor still may have a (shared) pointer to the
	// RemoteAuthorityPropertyNode(s), which were created by this RemoteAuthority,
	// and these nodes contain a pointer to this RemoteAuthority, we have to
	// reset this pointer to avoid segfaults when the RemoteAuthority gets
	// disconnected.
	if (mRemoteProperties) {
		RemoteAuthorityPropertyNode* p =
				dynamic_cast<RemoteAuthorityPropertyNode*>(mRemoteProperties.get());
		if (p)
			p->invalidateAuthority();
	}
}

StatusManager::StatusMap RemoteAuthority::getStatusMap()
{
	auto future = MIRA_FW.getRPCManager().call<StatusManager::StatusMap>(getGlobalID()+"#builtin",
	                                                                     "getStatusMap");
	if(!future.timedWait(cRPCTimeout))
		return StatusManager::StatusMap();
	return future.get();
}

boost::shared_ptr<PropertyNode> RemoteAuthority::getProperties()
{
	if(mRemoteProperties)
		return mRemoteProperties;

	try
	{
		// query properties from remote side
		RPCFuture<PropertyTree> res = MIRA_FW.getRPCManager().call<PropertyTree>(getGlobalID()+"#builtin",
		                                                                         "getProperties");
		if(!res.timedWait(cRPCTimeout))
			return boost::shared_ptr<PropertyNode>();
		PropertyTree tree = res.get();

		RemoteAuthorityPropertyNode* root = new RemoteAuthorityPropertyNode(tree.getRootNodeInfo());
		root->root = root;
		root->authority = this;
		tree.generatePropertyNodes<RemoteAuthorityPropertyNode>(root);
		mRemoteProperties.reset(root);

		return mRemoteProperties;
	}
	catch(std::exception& ex)
	{
		MIRA_LOG(ERROR) << "Exception while getting remote properties: " << ex.what() << std::endl;
	}
	return boost::shared_ptr<PropertyNode>();
}

std::set<std::string> RemoteAuthority::getPublishedChannels()
{
	auto future = MIRA_FW.getRPCManager().call<std::set<std::string>>(getGlobalID()+"#builtin",
	                                                                  "getPublishedChannels");
	if(!future.timedWait(cRPCTimeout))
		return std::set<std::string>();
	return future.get();
}

std::set<std::string> RemoteAuthority::getSubscribedChannels()
{
	auto future = MIRA_FW.getRPCManager().call<std::set<std::string>>(getGlobalID()+"#builtin",
	                                                                  "getSubscribedChannels");
	if(!future.timedWait(cRPCTimeout))
		return std::set<std::string>();
	return future.get();
}

std::set<std::string> RemoteAuthority::getServiceInterfaces()
{
#if	MIRA_MAJOR_VERSION(MIRA_PROTOCOL_VERSION)>3
	auto future = MIRA_FW.getRPCManager().call<std::set<std::string>>(getGlobalID()+"#builtin",
		                                                              "getServiceInterfaces");
	if(!future.timedWait(cRPCTimeout))
		return std::set<std::string>();
	return future.get();
#else
	if (!mHasServiceInterfaceQueryMethod)
			return std::set<std::string>();

	auto future = MIRA_FW.getRPCManager().call<std::set<std::string>>(getGlobalID()+"#builtin",
                                                                      "getServiceInterfaces");
	try {
		if(!future.timedWait(cRPCTimeout))
			return std::set<std::string>();
		return future.get();
	}
	catch(std::exception& ex) {
		MIRA_LOG(WARNING) << "Cannot obtain service interfaces from remote authority "
				<< getGlobalID() << "! Probably the version of the remote framework is too old!";
		mHasServiceInterfaceQueryMethod = false;
		return std::set<std::string>();
	}
#endif
}

AbstractAuthority::ChannelNameMapping RemoteAuthority::getPublishedChannelNames()
{
	auto future = MIRA_FW.getRPCManager().call<AbstractAuthority::ChannelNameMapping>(getGlobalID()+"#builtin",
	                                                                  "getPublishedChannelNames");
	if(!future.timedWait(cRPCTimeout))
		return AbstractAuthority::ChannelNameMapping();
	return future.get();
}

AbstractAuthority::ChannelNameMapping RemoteAuthority::getSubscribedChannelNames()
{
	auto future = MIRA_FW.getRPCManager().call<AbstractAuthority::ChannelNameMapping>(getGlobalID()+"#builtin",
	                                                                  "getSubscribedChannelNames");
	if(!future.timedWait(cRPCTimeout))
		return AbstractAuthority::ChannelNameMapping();
	return future.get();
}

void RemoteAuthority::start()
{
	auto future = MIRA_FW.getRPCManager().call<void>(getGlobalID()+"#builtin","start");
	if(!future.timedWait(cRPCTimeout))
		MIRA_THROW(XIO, "Failed to start remote authority '" << getGlobalID() << "'");
}

void RemoteAuthority::stop()
{
	auto future = MIRA_FW.getRPCManager().call<void>(getGlobalID()+"#builtin", "stop");
	if(!future.timedWait(cRPCTimeout))
		MIRA_THROW(XIO, "Failed to stop remote authority '" << getGlobalID() << "'");
}

bool RemoteAuthority::hasUnrecoverableFailure() const
{
	auto future = MIRA_FW.getRPCManager().call<bool>(getGlobalID()+"#builtin",
	                                                 "hasUnrecoverableFailure");
	if(!future.timedWait(cRPCTimeout))
		return false;
	return future.get();
}

bool RemoteAuthority::isStarted() const
{
	auto future = MIRA_FW.getRPCManager().call<bool>(getGlobalID()+"#builtin","isStarted");
	if(!future.timedWait(cRPCTimeout))
		return false;
	return future.get();
}

bool RemoteAuthority::isRunning() const
{
	auto future = MIRA_FW.getRPCManager().call<bool>(getGlobalID()+"#builtin","isRunning");
	if(!future.timedWait(cRPCTimeout))
		return false;
	return future.get();
}

///////////////////////////////////////////////////////////////////////////////

RemoteAuthorityPropertyUpdater::RemoteAuthorityPropertyUpdater()
{
	mThread = boost::thread(boost::bind(&RemoteAuthorityPropertyUpdater::process, this));
}

RemoteAuthorityPropertyUpdater::~RemoteAuthorityPropertyUpdater()
{
	mThread.interrupt();
	mThread.join();
}

void RemoteAuthorityPropertyUpdater::scheduleUpdate(const std::string& authorityID, RemoteAuthorityPropertyNode* property)
{
	// Add the property to the queue, if it is not included already
	boost::mutex::scoped_lock lock(mMutex);
	for(auto it = mPendingProperties.begin(); it!=mPendingProperties.end(); ++it)
	{
		if(it->second == property)
			return; // property already in queue
	}
	mPendingProperties.push_back(std::make_pair(authorityID, property));
	mSignal.notify_one();
}

void RemoteAuthorityPropertyUpdater::removeScheduledUpdates(RemoteAuthorityPropertyNode* property)
{
	boost::mutex::scoped_lock lock(mMutex);
	auto next = mPendingProperties.begin();
	for(auto it = mPendingProperties.begin(); it!=mPendingProperties.end(); it=next)
	{
		if(it->second == property)
			next = mPendingProperties.erase(it);
		else
			next = it+1;
	}
}

void RemoteAuthorityPropertyUpdater::process()
{
	ThreadMonitor::instance().addThisThread("#RemoteAuthorityPropertyUpdater");

	while (!boost::this_thread::interruption_requested())
	{
		try {
			// Get first property from the queue and collect all properties
			// from the same authority
			std::string authorityID;
			std::list<RemoteAuthorityPropertyNode*> properties;
			std::list<std::string> propertiesIDs;

			{
				boost::mutex::scoped_lock lock(mMutex);
				// If there nothing to do, wait
				if (mPendingProperties.empty()) {
					mSignal.wait(lock);
				}
				if (boost::this_thread::interruption_requested())
					break;

				// still empty? then continue and wait for next signal.
				if (mPendingProperties.empty())
					continue;

				authorityID = mPendingProperties.front().first;

				// collect all pending properties from that authority
				auto next = mPendingProperties.begin();
				for(auto it = mPendingProperties.begin(); it!=mPendingProperties.end(); it=next)
				{
					if(it->first == authorityID) {
						properties.push_back(it->second);
						propertiesIDs.push_back(it->second->fullID(it->second->root));
						next = mPendingProperties.erase(it);
					} else
						next = it+1;
				}
			}

			assert(!propertiesIDs.empty());
			assert(propertiesIDs.size()==properties.size());

			// Get the properties by using a RPC
			typedef std::list<json::Value> ValueList;
			RPCFuture<ValueList> future = MIRA_FW.getRPCManager().call<ValueList>(authorityID+"#builtin",
			                                            "getPropertyJSONBatch", propertiesIDs);
			if(!future.timedWait(cRPCTimeout))
				MIRA_THROW(XIO, "Failed to get properties from remote authority '" << authorityID << "'");

			// check result
			ValueList res = future.get();
			if(res.size()!=properties.size()) // should never happen, but it's better to be safe
				MIRA_THROW(XLogical, "The size of the returned value list does not match the requested list");

			// Store the return values in the properties
			auto it=properties.begin();
			foreach(json::Value& v, res)
			{
				(*it)->lastGetValue = v;
				++it;
			}

		} catch(std::exception& ex) {
			MIRA_LOG(ERROR) << "Exception while obtaining remote properties: " << ex.what() << std::endl;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}
