/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Framework.C
 *    Implementation of Framework.h.
 *
 * @author Tim Langner, Erik Einhorn
 * @date   2010/09/09
 */

#include <fw/Framework.h>

#include <boost/algorithm/string.hpp>

#include <error/Logging.h>
#include <error/LogConsoleSink.h>
#include <error/LogFileSink.h>
#include <error/SignalHandler.h>

#include <utils/PathFinder.h>
#include <utils/Profiler.h>
#include <factory/ManifestAgent.h>

#include <platform/Console.h>

#include <fw/FrameworkDefines.h>
#include <fw/RemoteModule.h>
#include <fw/FrameworkTransformer.h>
#include <fw/UnitManager.h>

#include <fw/ErrorService.h>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

Path getDefaultErrorDBPath()
{
	try {
		return getAppDataDirectory() / "errordb.sql";
	} catch(...) {}
	return "errordb.sql";
}

///////////////////////////////////////////////////////////////////////////////

// early initialization

Private::FrameworkStartup::FrameworkStartup(int argc, char** argv) :
	mOptions(argc, argv)
{
	initialize();
}

Private::FrameworkStartup::FrameworkStartup(const std::vector<std::string>& args) :
	mOptions(args)
{
	initialize();
}

void Private::FrameworkStartup::initialize()
{
	installSignalHandler(SIGSEGV, boost::bind(&Framework::errorHandler, _1));
	installSignalHandler(SIGFPE, boost::bind(&Framework::errorHandler, _1));
	installSignalHandler(SIGILL, boost::bind(&Framework::errorHandler, _1));
	installSignalHandler(SIGABRT, boost::bind(&Framework::errorHandler, _1));

	mOptions.getDescriptions().add_options()
		("config-file,c",
			boost::program_options::value<std::vector<std::string>>(),
			"The configuration files.")
		("config-string",
			boost::program_options::value<std::string>(),
			"The configuration to load specified directly as string.")
		("debug-level,d",
			boost::program_options::value<int>()->default_value((int)WARNING),
			"The log level from 0=CRITICAL to 5=TRACE.")
		("log-file,l",
			boost::program_options::value<std::string>(),
			"Name of an optional log file name. (Can't be mixed with --daily-log-file.)")
		("daily-log-file",
			boost::program_options::value<std::string>(),
			"Name of an optional log daily log file. (Can't be mixed with --log-file.)")
		("daily-log-file-start",
			boost::program_options::value<std::string>()->default_value("00:00:00"),
			"Optional daily log file start time.")
		("log-file-rotation-depth",
			boost::program_options::value<int>()->default_value(0),
			"Optional log file rotation depth.")
		("disable-stdlog",
			"Disable the standard cout log output.")
		("fw-port,p",
			boost::program_options::value<uint16>(),
			"Port of the remote framework module.")
		("fw-name,n",
			boost::program_options::value<std::string>(),
			"Name of the framework.")
		("error-db,e",
			boost::program_options::value<std::string>()->default_value(getDefaultErrorDBPath().string()),
			"The filename of the error database.")
		("auth-group",
			boost::program_options::value<std::string>(),
			"The working group this framework is allowed to connect.")
		("auth-passwd",
			boost::program_options::value<std::string>(),
			"Password for weak authentication.")
		("auth-keyfile",
			boost::program_options::value<std::string>(),
			"Path to file with RSA keys for strong authentication.")
		("known-fw,k",
			boost::program_options::value<CommandLineStrings>(&mCommandLineKnownFrameworks)->composing(),
			"Known frameworks [ip:]port[@option] separated by ',' or ';' (without ip assumes localhost). "
			"Recognized options: forceptp(=force PTP time sync),v0(=remote fw uses 'legacy' serialization format)")
		("disable-ptpsync",
			"Disables the PTP clock synchronization of remote frameworks.")
		("enable-pingtimeout",
			"Enables the ping timeout, that disconnects us from a remote framework that does not respond to pings.")
		("no-remote",
			"Disables any remote support (no remote server, no distributed frameworks).")
		("autodiscover",
			"Enables automatic discovery of other remote frameworks, manually specified frameworks are used.")
		("exit-on-disconnect",
			"Enables exit whenever a remote connection is closed or cannot be established.")
		("no-colors",
			"Disables colored text output in the console.")
		("using", boost::program_options::value<std::string>(),
			"Defines aliases for names in namespaces separated by , or ; It follows the syntax for the 'using' tag"
			" in config files (--using A=B can be read as 'using name A as B')(e.g. /robot/RobotFrame=/map/MapFrame "
			"would make the name/channel /robot/RobotFrame available in namespace /map as MapFrame).")
		("variables", boost::program_options::value<CommandLineStrings>(&mCommandLineVariablesLegacy)->composing(),
			"Deprecated. Please use --var or -v option instead.")
		("var,v", boost::program_options::value<CommandLineStrings>(&mCommandLineVariables)->composing(),
			"Defines variables in configuration files separated by , or ; (myName=\"Peter\",yourName=\"Paul\".")
		("enable-profiler", boost::program_options::value<std::string>()->implicit_value("profile"),
			"Use this option to write the profiler output. Optionally specify the directory.")
		("save-processed-config",
			boost::program_options::value<std::string>(),
			"Filename where to save final parsed and processed config (for debugging).")
		("no-autopromote",
			"Disables automatic promotion of channels that are published with type void (but valid typename).")
		("check-channel-registrations",
			"Enables check if channel type is registered for automatic promotion whenever "
			"publishing a non-void channel, and warn if missing.")
		("default-output-precision",
			boost::program_options::value<unsigned int>(),
			"Set precision for floating-point output from JSON (commonly used for generic MIRA object output).");

	mOptions.getPositionalDescriptions().add("config-file", -1);

	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();

	if ( vmap.count("default-output-precision") )
		json::JSONDefaultPrecision::set(vmap["default-output-precision"].as<unsigned int>());

	if ( vmap.count("disable-stdlog") == 0 ) {
		// set cout as default logging sink
		LogConsoleSink sink;
		sink.enableColors(vmap.count("no-colors")==0);
		MIRA_LOGGER.registerSink(sink);
	}

	if ( vmap.count("log-file") && vmap.count("daily-log-file") ) {
		MIRA_THROW(XInvalidParameter,
		           "Options --log-file and --daily-log-file can be used at the same time.");
	}

	// enable logging to file if desired
	if ( vmap.count("log-file") || vmap.count("daily-log-file") ) {
		std::string logFileName;
		bool dailyLogFile = false;
		Duration dailyLogFileStart = Duration(0, 0, 0);

		if ( vmap.count("log-file") ) {
			logFileName = vmap["log-file"].as<std::string>();
			dailyLogFile = false;
		}
		if ( vmap.count("daily-log-file") ) {
			logFileName = vmap["daily-log-file"].as<std::string>();
			dailyLogFile = true;

			if ( vmap.count("daily-log-file-start") ) {
				std::string startTimeStr = vmap["daily-log-file-start"].as<std::string>();
				std::vector<std::string> v;
				boost::split( v, startTimeStr, boost::is_any_of(":") );
				if (v.size() != 3)
					MIRA_THROW(XInvalidParameter,
					           "Bad format of argument of --daily-log-file-start. "
					           "Format must be HH:MM:SS");
				int hour = 0, minute = 0, second = 0;
				try {
					hour   = fromString<int>(v[0]);
					minute = fromString<int>(v[1]);
					second = fromString<int>(v[2]);
				} catch (Exception&) {
					MIRA_THROW(XInvalidParameter,
					           "Can't parse argument of --daily-log-file-start. "
					           "Format must be HH:MM:SS");
				}
				if ((hour < 0) || (hour > 23) ||
					(minute < 0) || (minute > 59) ||
					(second < 0) || (second > 59))
				{
					MIRA_THROW(XInvalidParameter,
					           "Invalid time argument for --daily-log-file-start. "
					           "Format must be HH:MM:SS");
				}
				dailyLogFileStart = Duration(hour, minute, second);
			}
		}

		int logFileRotationDepth = 0;
		if ( vmap.count("log-file-rotation-depth") )
			logFileRotationDepth = vmap["log-file-rotation-depth"].as<int>();

		// create the log file sink and register it
		LogFileSink fileSink(logFileName, logFileRotationDepth,
		                     dailyLogFile, dailyLogFileStart);
		MIRA_LOGGER.registerSink(fileSink);
	}

	MIRA_LOGGER.setSeverityLevel((SeverityLevel)vmap["debug-level"].as<int>());

	// load all manifests
	//try {
		loadManifests();
	//} catch(Exception& ex) {
	//	MIRA_LOG(WARNING) << ex.what();
	//}
}

///////////////////////////////////////////////////////////////////////////////

Framework::Framework(int argc, char** argv, bool startImmediately) :
	Private::FrameworkStartup(argc,argv),
	mTerminationRequested(false),
	mTerminationExitCode(0),
	mIsStarted(false),
	mInExec(true),
	mRemoteDisabled(false),
	mUnitManager(new UnitManager())
{
	try {
		initialize();
		if (startImmediately)
		{
			load();
			start();
		}
	}
	catch (std::exception& ex) {
		finalize();
		throw; /* rethrow */
	}
}

Framework::Framework(const std::vector<std::string>& args, bool startImmediately) :
	Private::FrameworkStartup(args),
	mTerminationRequested(false),
	mIsStarted(false),
	mInExec(true),
	mRemoteDisabled(false),
	mUnitManager(new UnitManager())
{
	try {
		initialize();
		if (startImmediately)
		{
			load();
			start();
		}
	}
	catch (std::exception& ex) {
		finalize();
		throw; /* rethrow */
	}
}

void parseVariables(const std::string& varStr, XMLVariablesMap& vMap)
{
	std::vector<std::string> assigns;
	boost::split( assigns, varStr, boost::is_any_of(",;"));
	foreach(const std::string& a, assigns)
	{
		std::vector<std::string> elems;
		boost::split( elems, a, boost::is_from_range('=','=') );
		if ( elems.size() != 2 )
			MIRA_THROW(XInvalidParameter,
			           a << " is not a valid variable definition (use X=Y as syntax)");
		vMap[elems[0]] = XMLVariableValue(elems[1],
		                                  MakeString() << "Command line: '--var " << varStr << "'");
	}
}

void Framework::initialize()
{
	installSignalHandler(SIGINT, boost::bind(&Framework::ctrlCHandler, this, _1));
#ifdef MIRA_LINUX
	installSignalHandler(SIGHUP, boost::bind(&Framework::ctrlCHandler, this, _1));
#endif

	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();

#ifdef MIRA_LINUX
	mXMLPreprocessor.variables["os"] = XMLVariableValue("linux", "Framework internal");
#else
	mXMLPreprocessor.variables["os"] = XMLVariableValue("windows", "Framework internal");
#endif
	mXMLPreprocessor.variables["severity"] = XMLVariableValue(toString(MIRA_LOGGER.getSeverityLevel()),
	                                                          "Framework internal");
	// TODO remove in a future version..only for backward compatibility
	if (!mCommandLineVariablesLegacy.empty())
	{
		MIRA_LOG(WARNING) << "The command line parameter --variables is deprecated. Please use --var or -v instead";
	}
	foreach(const std::string& v, mCommandLineVariablesLegacy)
		parseVariables(v, mXMLPreprocessor.variables);
	foreach(const std::string& v, mCommandLineVariables)
		parseVariables(v, mXMLPreprocessor.variables);
	std::string aliases;
	if ( vmap.count("using") )
		aliases = vmap["using"].as<std::string>();
	if (!aliases.empty())
	{
		std::vector<std::string> assigns;
		boost::split( assigns, aliases, boost::is_any_of(",;"));
		foreach(const std::string& a, assigns)
		{
			std::vector<std::string> elems;
			boost::split( elems, a, boost::is_from_range('=','=') );
			if ( elems.size() != 2 )
				MIRA_THROW(XInvalidParameter,
				           a << " is not a valid alias (use X=Y as syntax)");
			mNames.addAlias(elems[0], elems[1], "/",
				            MakeString() << "Command line: '--using " << aliases << "'");
		}
	}

	if (vmap.count("fw-name"))
		mName = vmap["fw-name"].as<std::string>();

	// create named authority or anonymous if no name specified
	if (!mName.empty())
		mAuthority.reset(new Authority("/", mName));
	else
		mAuthority.reset(new Authority("/", "Framework", Authority::ANONYMOUS |
		                               Authority::HIDDEN));

	if ( vmap.count("no-remote") )
		mRemoteDisabled = true;

	if ( !mRemoteDisabled )
	{
		mRemoteFramework.reset(new RemoteModule());
		mAuthority->publishService(*mRemoteFramework.get());
	}

	mTransformer.reset(new FrameworkTransformer(*mAuthority));

	if (vmap.count("no-autopromote"))
		mChannelManager.enableAutoPromoteChannels(false);

	if (vmap.count("check-channel-registrations"))
		mChannelManager.enableCheckChannelRegistrations(true);

	mAuthority->publishService(mRPCManager);
	mAuthority->publishService(mChannelManager);
	mAuthority->publishService(mAuthorityManager);
	mAuthority->publishService(mNames);

	// register all available preparer and loader plugins
	typedef std::map<std::string, ClassProxy> ClassMap;
	ClassMap preparers = ConfigurationPreparePlugin::CLASS().getDerivedClasses();
	foreach(ClassMap::value_type l, preparers)
	{
		if(l.second.isAbstract())
			continue;
		try{
			mConfigurationLoader.registerPreparePlugin(ConfigurationPreparePluginPtr(l.second.newInstance<ConfigurationPreparePlugin>()));
		}
		catch(std::exception& ex) {
			MIRA_LOG_EXCEPTION(ERROR, ex) << "Failed registering plugin '" << l.first << "':\n";
		}
	}

	typedef std::map<std::string, ClassProxy> ClassMap;
	ClassMap loaders = ConfigurationLoaderPlugin::CLASS().getDerivedClasses( );
	foreach(ClassMap::value_type l, loaders)
	{
		if(l.second.isAbstract())
			continue;

		try{
			mConfigurationLoader.registerLoaderPlugin(ConfigurationLoaderPluginPtr(l.second.newInstance<ConfigurationLoaderPlugin>()));
		}
		catch(std::exception& ex) {
			MIRA_LOG_EXCEPTION(ERROR, ex) << "Failed registering plugin '" << l.first << "':\n";
		}
	}

	mStartTime = Time::invalid();
	mAuthority->publishService(*this);
}

Framework::~Framework()
{
	finalize();
}

void Framework::finalize()
{
	MIRA_LOG(DEBUG) << "Destructing Framework";

	try {
		if (mRemoteFramework) {
			mRemoteFramework->stop();
			mRemoteFramework.reset();
		}
		mErrorServiceModule.reset();
		// make sure to stop and destroy all units before we destroy our framework.
		mUnitManager.reset();
		mTransformer.reset();

		mAuthority.reset();

		// write profiler output
		ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();
		if(vmap.count("enable-profiler")){
			boost::filesystem::path directory(vmap["enable-profiler"].as<string>());
			if (!boost::filesystem::exists(directory) && !boost::filesystem::create_directory(directory)) {
				MIRA_LOG(ERROR) << "Could not create profiler directory: '" << directory.string() << "'";
			} else
				Profiler::writeReport(directory.string(), "dot");
		};
	} catch(std::exception& ex) {
		MIRA_LOG_EXCEPTION(ERROR, ex) << "An exception was thrown in Framework::finalize(). "
		                                 "This indicates a critical issue in a component of the framework:\n";
	}
}

void Framework::load(XMLDom& xml)
{
	// if not yet started add the xml documents content to the global config document
	if (!mIsStarted)
	{
		XMLDom::sibling_iterator root = mConfigDom.root();
		for(auto i=xml.root().begin(); i!=xml.root().end(); ++i)
			root.add_child(i);
	}
	// otherwise load and parse the document and initialize and start all new units
	// now. Otherwise, they will be started in Framework::start()
	else
	{
		mConfigurationLoader.prepare(xml);
		mConfigurationLoader.load(xml);
		mUnitManager->initializeUnits();
	}
}

void Framework::load(const std::string& configFile)
{
	// first try to just resolve the path in the current directory
	Path path = resolvePath(configFile);
	// in case the file is not found give the user a more helpful error message
	if (!boost::filesystem::exists(path))
	{
		auto suggestedFiles = findProjectFiles(configFile, true);
		if (suggestedFiles.size() > 0)
		{
			MIRA_THROW(XFileNotFound, "The file " << path.string() << " does not exist! Do you mean any of the following: " << print(suggestedFiles));
		}
		else
		{
			MIRA_THROW(XFileNotFound, "The file " << path.string() << " does not exist!");
		}
	}

	if (!mIsStarted)
	{
		mConfigDom.root().add_child("include").add_attribute("file", configFile);
	}
	else
	{
		// load the config file
		XMLDom xml;
		xml.loadFromFile(path);
		load(xml);
	}
}

void Framework::load()
{
	// load all config files that were specified in the command line
	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();
	if ( vmap.count("config-file") ) {
		std::vector<std::string> configs = vmap["config-file"].as< std::vector<std::string>>();
		foreach(const std::string& c, configs)
			load(c);
	}
	if ( vmap.count("config-string") ) {
		XMLDom xml;
		xml.loadFromString(vmap["config-string"].as<std::string>());

		if (*(xml.croot()) != "root")
			MIRA_THROW(XIO, "Please make sure parameter value used with '--config-string' "
			                "is enclosed by '<root></root>'.");

		xml.setUri("Command line: '--config-string ...'");
		load(xml);
	}
}

bool compareCaseIns(const std::string& s, const std::string& ref)
{
	return boost::algorithm::to_lower_copy(s) == ref;
}

void Framework::start()
{
	if(mIsStarted)
	{
		MIRA_LOG(NOTICE)<<"Framework already started";
		return;
	}

	// load global config now
	mConfigurationLoader.prepare(mConfigDom);
	// save preprocessed config if filename is specified
	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();
	if (vmap.count("save-processed-config"))
		mConfigDom.saveToFile(vmap["save-processed-config"].as<std::string>());
	mConfigurationLoader.load(mConfigDom);

	// create the local error service
	mErrorServiceModule.reset(new ErrorService(*mAuthority, vmap["error-db"].as<std::string>()));

	// initialize and start all of our units here
	mUnitManager->initializeUnits();

	// is remote functionality used -> parse command line options
	if ( !mRemoteDisabled && mRemoteFramework )
	{
		if ( vmap.count("disable-ptpsync"))
			mRemoteFramework->enablePTPSync(false);

		if ( vmap.count("enable-pingtimeout"))
			mRemoteFramework->enablePingTimeout();

		if ( vmap.count("exit-on-disconnect"))
			mRemoteFramework->enableExitOnDisconnect(true);
		// If we have specified a port on the command line we will use this
		if ( vmap.count("fw-port") )
			mRemoteFramework->setPort(vmap["fw-port"].as<uint16>());
		foreach(const std::string& knownFW, mCommandLineKnownFrameworks)
		{
			std::vector<std::string> known;
			boost::split( known, knownFW, boost::is_any_of(",;") );
			foreach(std::string a, known)
			{
				std::vector<std::string> params;
				boost::split( params, a, boost::is_any_of("@") );

				bool ptp = false;
				if (std::find_if(params.begin(), params.end(),
				                 boost::bind(&compareCaseIns, _1, "forceptp")) != params.end()) {
					ptp = true;
				}

				bool legacy = false;
				if (std::find_if(params.begin(), params.end(),
				                 boost::bind(&compareCaseIns, _1, "v0")) != params.end()) {
					legacy = true;
				}

				// pure port number is directed to localhost
				if (params[0].find_first_not_of( "0123456789" ) == std::string::npos)
					params[0] = "127.0.0.1:"+params[0];

				MIRA_LOG(DEBUG) << "Adding known " << (legacy ? "legacy " : "") << "framework " << params[0]
				                << (ptp ? " (Forcing PTP sync)" : "");
				mRemoteFramework->addKnownFramework(params[0], ptp, legacy);
			}
		}

		// authentication settings
		if(vmap.count("auth-group"))
			mRemoteFramework->setAuthGroup(vmap["auth-group"].as<std::string>());

		if(vmap.count("auth-passwd"))
			mRemoteFramework->setAuthPassword(vmap["auth-passwd"].as<std::string>());

		if(vmap.count("auth-keyfile"))
			mRemoteFramework->setAuthKeyFile(vmap["auth-keyfile"].as<std::string>());

		mRemoteFramework->start();
	}

	mStartTime = Time::now();
	mIsStarted = true;
}

int Framework::exec()
{
	ThreadMonitor::instance().addThisThread("Main thread");
	mInExec = true;
	while(!mTerminationRequested)
		MIRA_SLEEP(1000)
	mInExec = false;
	return getTerminationExitCode();
}

void Framework::requestTermination(int exitcode)
{
	mTerminationRequested = true;
	mTerminationExitCode = exitcode;
}

bool Framework::isTerminationRequested() const
{
	return mTerminationRequested;
}

int Framework::getTerminationExitCode() const
{
	return mTerminationExitCode;
}

int Framework::run()
{
	try
	{
		ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();
		if ( vmap.count("help") ) {
			std::cout << std::endl << "Usage:" << std::endl << std::endl;
			std::cout << MIRA_CMDLINE.getDescriptions();
			return -1;
		}

		load();
		start();

		MIRA_LOG(NOTICE) << "Framework is up and running.";

		return this->exec();
	}
	catch(std::exception& ex)
	{
		MIRA_LOG_EXCEPTION(ERROR, ex) << "Exception:\n";
		return -1;
	}

	// we never reach here
	return 0;
}

///////////////////////////////////////////////////////////////////////////////

void Framework::ctrlCHandler(const IntSignal& sig)
{
	MIRA_LOG(NOTICE) << "Received signal " << sig.sig << ". Terminating...";
	requestTermination();
}

void Framework::errorHandler(const IntSignal& sig)
{
	if(!enterLeaveErrorHandler(true))
		return; // return, if already are inside a signal handler
	MIRA_LOG_SIGNAL(ERROR, sig);
	abort();
	enterLeaveErrorHandler(false);
}

bool Framework::enterLeaveErrorHandler(bool enter)
{
	static bool inHandler=false;
	if(enter && inHandler)
		return false;

	inHandler = enter;
	return true;
}

///////////////////////////////////////////////////////////////////////////////

}
