/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TransformProperty.h
 *    Property that can be used whenever a unit has a transform frame as property.
 *
 * @author Erik Einhorn
 * @date   2011/10/19
 */

#ifndef _MIRA_TRANSFORMPROPERTY_H_
#define _MIRA_TRANSFORMPROPERTY_H_

#include <serialization/ReflectorInterface.h>
#include <serialization/IsTransparentSerializable.h>
#include <serialization/IsObjectTrackable.h>

#include <fw/FrameworkExports.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Use this class instead of a string property whenever your module has 
 * a transform frame as property. When the user clicks on that property
 * a dialog will open where he can select a frame out of all existing ones.
 * \code
 * template <typename Reflector>
 * void reflect(Reflector& r) {
 *     r.property("FrameID", myFrame, "The frame id");
 * }
 * TransformProperty myFrame;
 * \endcode
 */
class MIRA_FRAMEWORK_EXPORT TransformProperty
{
public:

	template <typename Reflector>
	void reflect(Reflector& r) {
		r.delegate(mID);
	}

	bool operator==(const TransformProperty& other) const {
		return mID == other.mID;
	}

	bool operator!=(const TransformProperty& other) const {
		return !operator==(other);
	}

	TransformProperty& operator=(const std::string& s) {
		mID = s;
		return *this;
	}

	/// Returns the frame ID of this property
	const std::string& getID() const {
		return mID;
	}

	operator std::string() const { return mID; }

	bool isValid() const;

	bool isEmpty() const {
		return mID.empty();
	}

	bool empty() const {
		return mID.empty();
	}

private:

	std::string mID; ///< The selected frame id or empty if none
};

///////////////////////////////////////////////////////////////////////////////

template<typename SerializerTag>
class IsTransparentSerializable<TransformProperty,SerializerTag> : public std::true_type {};

template<>
class IsObjectTrackable<TransformProperty> : public std::false_type {};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
