/*
 * Copyright (C) 2018 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VersioningTestClasses.C
 *    Test classes for serialization framework versioning.
 *
 * @author Christof Schröter
 */

#include "CommonTest.h"

#ifndef Q_MOC_RUN
#include <boost/test/unit_test.hpp>
#endif

#include <serialization/ReflectorMacros.h>

#include <serialization/adapters/std/set>
#include <serialization/adapters/std/map>
#include <serialization/adapters/std/pair>
#include <serialization/adapters/std/vector>

///////////////////////////////////////////////////////////////////////////////

class BaseBaseClass
{
public:
	BaseBaseClass() : n(0), callVersion(true) {}
	BaseBaseClass(bool) : n(10), callVersion(true) {}

	void check() {
		BOOST_CHECK_EQUAL(n, 10);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		serialization::VersionType version = 1;
		if (callVersion) version = r.version(1, this);	// make sure this does not fail
//		std::cout << "BaseBase version " << (int)version << std::endl;
		if (version <= 1)
			r.member("n", n, "");               	 	// make sure we read back the member correctly for version 0 or 1
	}

	void enableCallVersion(bool enable) { callVersion = enable; }

protected:
	int n;
	bool callVersion;
};

///////////////////////////////////////////////////////////////////////////////

class BaseClass : public BaseBaseClass
{
public:
	BaseClass() : i(0), properBaseReflect(true) {}
	BaseClass(bool) : BaseBaseClass(true), i(1), properBaseReflect(true) {}

	void check() {
		BaseBaseClass::check();
		BOOST_CHECK_EQUAL(i, 1);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		if (properBaseReflect)
			MIRA_REFLECT_BASE(r, BaseBaseClass);			// this allows the reflector to know we are entering a base class
		else
			BaseBaseClass::reflect(r);						// this does not, it should be illegal

		serialization::VersionType version = r.version(2, this);

		if(version==2) // make sure that we get the correct version
			r.member("i", i, "");
		else
			i=0;
	}

	void enableProperBaseReflect(bool enable) { properBaseReflect = enable; }

protected:
	int i;
	bool properBaseReflect;
};

///////////////////////////////////////////////////////////////////////////////

namespace ns {
class SubClass : public BaseClass
{
public:
	SubClass() : j(0) {}
	SubClass(bool) : BaseClass(true), j(1) {}

	void check() {
		BaseClass::check();
		BOOST_CHECK_EQUAL(j, 1);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, BaseClass);

		serialization::VersionType version = r.version(3, this);

		if(version==3) // make sure that we get the correct version
			r.member("j", j, "");
		else
			j=0;
	}

protected:
	int j;
};
} // namespace

///////////////////////////////////////////////////////////////////////////////
// Test that binary compatibility is preserved when reflection transparency of a
// class changes

// First the class just had j, so it acted as transparent type and delegated
// to j. Then it got another member i, now it is not transparent anymore and has 2
// members instead of a delegate. It had to get a version of course.

class ClassWithDelegate
{
public:
	ClassWithDelegate() : j(2) {}

	void check() {
		BOOST_CHECK_EQUAL(j, 4);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.delegate(j);
	}

protected:
	int j;
};

namespace mira {

template <typename SerializerTag>
class IsTransparentSerializable<ClassWithDelegate,SerializerTag> : public std::true_type {};

}

///////////////////////////////////////////////////////////////////////////////

class ClassWithoutDelegate
{
public:
	ClassWithoutDelegate() : i(3), j(4) {}

	void check() {
		BOOST_CHECK_EQUAL(i, 3);
		BOOST_CHECK_EQUAL(j, 2);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		serialization::VersionType v = r.version(1, this);
		if (v > 0)
			r.member("i", i, "");

		r.member("j", j, "");
	}

protected:
	int i, j;
};

///////////////////////////////////////////////////////////////////////////////
// Test binary serialize+deserialize for different combinations of getters
// and setters, both for primitive types and objects with own reflect() methods

class ClassWithSetterSimple
{
public:
	ClassWithSetterSimple() : i(0){}
	ClassWithSetterSimple(bool) : i(1) {}

	void check() {
		BOOST_CHECK_EQUAL(i, 1);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("i", i,
		         setter(&ClassWithSetterSimple::set, this), "");
	}

	void set(int v) { i = v; }

private:
	int i;
};

///////////////////////////////////////////////////////////////////////////////

class ClassWithGetterSetterSimple
{
public:
	ClassWithGetterSetterSimple() : i(0){}
	ClassWithGetterSetterSimple(bool) : i(1) {}

	void check() {
		BOOST_CHECK_EQUAL(i, 1);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("i",
		         getter(&ClassWithGetterSetterSimple::get, this),
		         setter(&ClassWithGetterSetterSimple::set, this), "");
	}

	void set(int v) { i = v; }
	int get() const { return i; }

private:
	int i;
};

///////////////////////////////////////////////////////////////////////////////

class ClassWithSetterComplex
{
public:
	ClassWithSetterComplex() {}
	ClassWithSetterComplex(bool) : s(true) {}

	void check() {
		s.check();
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("s", s,
		         setter(&ClassWithSetterComplex::set, this), "");
	}

	void set(const ns::SubClass& v) { s = v; }

private:
	ns::SubClass s;
};

///////////////////////////////////////////////////////////////////////////////

class ClassWithGetterSetterComplex
{
public:
	ClassWithGetterSetterComplex() {}
	ClassWithGetterSetterComplex(bool) : s(true) {}

	void check() {
		s.check();
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("s",
		         getter(&ClassWithGetterSetterComplex::get, this),
		         setter(&ClassWithGetterSetterComplex::set, this), "");
	}

	void set(const ns::SubClass& v) { s = v; }

	const ns::SubClass& get() const { return s; }

private:
	ns::SubClass s;
};

///////////////////////////////////////////////////////////////////////////////
// Test MIRA_REFLECT_CALL functionality:
// an externally called reflect method can define an own reflection version
// when MIRA_REFLECT_CALL is used

struct TestStruct {
	TestStruct() : n(5) {}
	TestStruct(bool) : n(6) {}

	int n;
};

struct TestStructReflectABCTag {};

template<typename Reflector>
void reflectABC(Reflector& r, TestStruct & t)
{
	// locally declared tag would be prefered actually,
	// but not allowed as template param on gcc 4.4
	//struct TestStructReflectTag {};

	r.template version<TestStructReflectABCTag>(2);
	r.member("t", t.n, "");
}

template<typename Reflector>
void reflectRead(Reflector& r, TestStruct & t)
{
	MIRA_REFLECT_CALL(Reflector, r, "TestStruct reflectABC", (reflectABC(r, t)));
}

template<typename Reflector>
void reflectWrite(Reflector& r, TestStruct & t)
{
	MIRA_REFLECT_CALL(Reflector, r, "TestStruct reflectABC", (reflectABC(r, t)));
}

struct TestStructExternalReflectTag {};

template<typename Reflector>
void externalReflect(Reflector& r, TestStruct & t)
{
	// locally declared tag would be prefered actually,
	// but not allowed as template param on gcc < 4.4
	//struct TestStructReflectTag {};

	r.template version<TestStructExternalReflectTag>(3);
	splitReflect(r, t);
}

///////////////////////////////////////////////////////////////////////////////

class ClassWithExternalReflect
{
public:
	ClassWithExternalReflect() : properExternalReflect(true), i(0){}
	ClassWithExternalReflect(bool) : properExternalReflect(true), i(1), t(true) {}

	void check() {
		BOOST_CHECK_EQUAL(i, 1);
		BOOST_CHECK_EQUAL(t.n, 6);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		serialization::VersionType v = r.version(4, this);
		if (properExternalReflect) {
			MIRA_REFLECT_CALL(Reflector, r,
			                  "ClassWithExternalReflect externalReflect",
			                  (externalReflect(r, t)));
		}
		else
			externalReflect(r, t);

		if (v == 4)		
			r.member("i", i, "");

		r.member("l1", l1, "");
		r.property("l2", l2, "");
		r.property("s1", s1, "");
		r.property("s2", s2, "");
		r.property("m1", m1, "");
		r.property("m2", m2, "");
		r.property("v1", v1, "");
		r.property("v2", v2, "");
		r.property("b", b, "");
		r.property("a1", a1, "");
		r.property("a2", a2, "");
	}

	void enableProperExternalReflect(bool enable) { properExternalReflect = enable; }

private:
	bool properExternalReflect;
	int i;
	TestStruct t;
	std::list<int> l1;
	std::list<std::list<float>*> l2;
	std::set<std::list<int>*> s1;
	std::set<ns::SubClass*> s2;
	std::map<int, float> m1;
	std::map<std::list<int>, std::list<float>*> m2;
	std::vector<int> v1;
	std::vector<std::vector<ns::SubClass>> v2;
	std::vector<bool> b;
	int a1[5];
	BaseClass a2[1];
};

///////////////////////////////////////////////////////////////////////////////

class ClassWithSplitReflect
{
public:
	ClassWithSplitReflect() {}
	ClassWithSplitReflect(bool) : x(true) {}

	void check() {
		x.check();
	}

	template<typename Reflector>
	void reflectRead(Reflector& r)
	{
		r.delegate(x);
	}

	template<typename Reflector>
	void reflectWrite(Reflector& r)
	{
		r.delegate(x);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		serialization::VersionType v = r.version(4, this);
		splitReflectMember(r, this);
	}	

private:
	ClassWithExternalReflect x;
};

namespace mira {

template <typename SerializerTag>
class IsTransparentSerializable<ClassWithSplitReflect,SerializerTag> : public std::true_type {};

}
