/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BinaryJSONConverterTest.C
 *
 * @author Erik Einhorn
 */

#include <iostream>
#include <boost/test/unit_test.hpp>

#include <serialization/BinarySerializer.h>
#include <serialization/BinaryJSONConverter.h>
#include <serialization/MetaSerializer.h>
#include <serialization/Print.h>
#include <serialization/adapters/std/list>
#include <serialization/Print.h>
#include <serialization/adapters/std/map>

#include <boost/make_shared.hpp>

#include <transform/Pose.h>
#include <utils/Stamped.h>

#include "VersioningTestClasses.h"

using namespace mira;

struct ClassWithSharedPtr
{
	ClassWithSharedPtr()     : a(1), b(), s(boost::make_shared<BaseBaseClass>()) {}
	ClassWithSharedPtr(bool) : a(5), b(true), s(boost::make_shared<BaseBaseClass>(true)) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("a", a, "");
		r.member("b", b, "");
		r.member("s", s, "");
	}

	int a;
	BaseBaseClass b;
	boost::shared_ptr<BaseBaseClass> s;
};

BOOST_AUTO_TEST_CASE( BinaryJSONTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	Stamped<std::list<std::vector<Pose2*>*>> v;
	auto typemeta = ms.addMeta(v);

	// print out database for manual debugging purposes
//	foreach(auto p, db) {
//		std::cout << p.first <<  p.second->toString() << std::endl;
//	}

	v.timestamp = Time::now();
	v.frameID = "myframe";

	v.push_back(new std::vector<Pose2*>);
	v.push_back(NULL);

	v.front()->push_back(new Pose2(1.0f,2.0f,3.0f));
	v.front()->push_back(NULL);
	v.front()->push_back(new Pose2(6.0f,2.0f,7.0f));
	//v.front()->push_back(v.front()->back()); // pointer tracking not implemented yet

//	std::cout << print(v) << std::endl;
	Buffer<uint8> buf;
	BinaryBufferSerializer bs(&buf);
	bs.serialize(v);

//	std::cout << print(buf) << std::endl;

	JSONValue val;
	BinaryJSONConverter::binaryToJSON(buf, true, *typemeta, db, val);

//	std::cout << json::write(val) << std::endl;

	JSONSerializer js;
	json::Value val2 = js.serialize(v);

//	std::cout << json::write(val2) << std::endl;

	//      object     =        object
	//        |                   |
	// BinarySerializer     JSONSerializer
	//        |                   |
	// BinaryJSONConverter        |
	//        |                   |
	//      json       ==        json  ?

	BOOST_CHECK_EQUAL(json::write(val, true), json::write(val2, true));
	BOOST_CHECK(val == val2);
}

BOOST_AUTO_TEST_CASE( BinaryJSONPlainArrayTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	std::vector<int> v;
	auto typemeta = ms.addMeta(v);

	// print out database for manual debugging purposes
//	foreach(auto p, db) {
//		std::cout << p.first <<  p.second->toString() << std::endl;
//	}

	v.push_back(1);
	v.push_back(2);
	v.push_back(3);

//	std::cout << print(v) << std::endl;
	Buffer<uint8> buf;
	BinaryBufferSerializer bs(&buf);
	bs.serialize(v);

//	std::cout << print(buf) << std::endl;

	JSONValue val;
	BinaryJSONConverter::binaryToJSON(buf, true, *typemeta, db, val);

//	std::cout << json::write(val) << std::endl;

	JSONSerializer js;
	json::Value val2 = js.serialize(v);

//	std::cout << json::write(val2) << std::endl;

	//      object     =        object
	//        |                   |
	// BinarySerializer     JSONSerializer
	//        |                   |
	// BinaryJSONConverter        |
	//        |                   |
	//      json       ==        json  ?

	BOOST_CHECK_EQUAL(json::write(val, true), json::write(val2, true));
	BOOST_CHECK(val == val2);
}

BOOST_AUTO_TEST_CASE( BinaryJSONVersionsTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	std::list<ns::SubClass> v;

	auto typemeta = ms.addMeta(v);

	// print out database for manual debugging purposes
//	foreach(auto p, db) {
//		std::cout << p.first <<  p.second->toString() << std::endl;
//	}

	v.push_back(ns::SubClass());
	v.push_back(ns::SubClass(true));	

//	std::cout << print(v) << std::endl;
	Buffer<uint8> buf;
	BinaryBufferSerializer bs(&buf);
	bs.serialize(v);

//	std::cout << print(buf) << std::endl;

	JSONValue val;
	BinaryJSONConverter::binaryToJSON(buf, true, *typemeta, db, val);

//	std::cout << json::write(val, true) << std::endl;

	BOOST_CHECK_EQUAL(val.type(), json_spirit::array_type);
	json::Array& a = val.get_array();
	BOOST_CHECK_EQUAL(a.size(), 2);

	std::stringstream ss;
	ss << print(v);
	
	BOOST_CHECK_EQUAL(ss.str(), json::write(val, true));
}

BOOST_AUTO_TEST_CASE( BinaryJSONTransparentSerializableTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	ClassWithSharedPtr v;
	auto typemeta = ms.addMeta(v);

	// print out database for manual debugging purposes
//	foreach(auto p, db) {
//		std::cout << p.first <<  p.second->toString() << std::endl;
//	}

//	std::cout << print(v) << std::endl;
	Buffer<uint8> buf;
	BinaryBufferSerializer bs(&buf);
	bs.serialize(v);

//	std::cout << print(buf) << std::endl;

	JSONValue val;
	BinaryJSONConverter::binaryToJSON(buf, true, *typemeta, db, val);

//	std::cout << json::write(val, true) << std::endl;

	std::stringstream ss;
	ss << print(v);
	
	BOOST_CHECK_EQUAL(ss.str(), json::write(val, true));
}

BOOST_AUTO_TEST_CASE( BinaryJSONTransparentSerializableTopLevelTest )
{
	{
		MetaTypeDatabase db;
		MetaSerializer ms(db);

		boost::shared_ptr<BaseBaseClass> v;
		v = boost::make_shared<BaseBaseClass>(true);

		auto typemeta = ms.addMeta(v);

		// print out database for manual debugging purposes
//		foreach(auto p, db) {
//			std::cout << p.first <<  p.second->toString() << std::endl;
//		}

//		std::cout << print(v) << std::endl;
		Buffer<uint8> buf;
		BinaryBufferSerializer bs(&buf);
		bs.serialize(v);

//		std::cout << print(buf) << std::endl;

		JSONValue val;
		BinaryJSONConverter::binaryToJSON(buf, true, *typemeta, db, val);

//		std::cout << json::write(val, true) << std::endl;

		std::stringstream ss;
		ss << print(v);
		
		BOOST_CHECK_EQUAL(ss.str(), json::write(val, true));
	}
	{
		MetaTypeDatabase db;
		MetaSerializer ms(db);

		Degreef v(90.f);
		auto typemeta = ms.addMeta(v);

		// print out database for manual debugging purposes
//		foreach(auto p, db) {
//			std::cout << p.first <<  p.second->toString() << std::endl;
//		}

//		std::cout << print(v) << std::endl;
		Buffer<uint8> buf;
		BinaryBufferSerializer bs(&buf);
		bs.serialize(v);

//		std::cout << print(buf) << std::endl;

		JSONValue val;
		BinaryJSONConverter::binaryToJSON(buf, true, *typemeta, db, val);

//		std::cout << json::write(val, true) << std::endl;

		std::stringstream ss;
		ss << print(v);
		
		// different number of floating point digits may occur, but the first few should match
		BOOST_CHECK_EQUAL(ss.str().substr(0, 6), json::write(val, true).substr(0, 6));
	}
}

BOOST_AUTO_TEST_CASE( BinaryJSONDeserializeTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	std::map<int, int> v;
	auto typemeta = ms.addMeta(v);

	v[0] = 1;
	v[1] = 2;
	v[2] = 3;

	Buffer<uint8> buf;
	BinaryBufferSerializer bs(&buf);
	bs.serialize(v);

	JSONValue val;
	BinaryJSONConverter::binaryToJSON(buf, true, *typemeta, db, val);

	JSONDeserializer jd(val);

	std::map<int, int> backV;
	jd.deserialize(backV);

//	std::cout << "in: " << print(v) << std::endl;
//	std::cout << "out: " << print(backV) << std::endl;

	BOOST_CHECK(v == backV);
}

BOOST_AUTO_TEST_CASE( JSONBinaryTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	Stamped<std::list<std::vector<Pose2*>*>> v;
	auto typemeta = ms.addMeta(v);

	v.timestamp = Time::now();
	v.frameID = "myframe";

	v.push_back(new std::vector<Pose2*>);
	v.push_back(NULL);

	v.front()->push_back(new Pose2(1.0f,2.0f,3.0f));
	v.front()->push_back(NULL);
	v.front()->push_back(new Pose2(6.0f,2.0f,7.0f));
	//v.front()->push_back(v.front()->back()); // pointer tracking not implemented yet

	JSONSerializer js;
	json::Value val = js.serialize(v);

//	std::cout << json::write(val, true) << std::endl;

	Buffer<uint8> backBuf;
	BinaryJSONConverter::JSONToBinary(val, *typemeta, db, backBuf, true);

	BinaryBufferDeserializer bd(&backBuf);
	Stamped<std::list<std::vector<Pose2*>*>> backV;
	bd.deserialize(backV);

	BOOST_CHECK_EQUAL(v.timestamp, backV.timestamp);
	BOOST_CHECK_EQUAL(v.frameID, backV.frameID);
	BOOST_REQUIRE_EQUAL(v.size(), backV.size());
	BOOST_CHECK_EQUAL(v.front()->size(), backV.front()->size());
	BOOST_CHECK(v.front()->at(0)->isApprox(*backV.front()->at(0)));
	BOOST_CHECK_EQUAL(v.front()->at(1), backV.front()->at(1));
	BOOST_CHECK(v.front()->at(2)->isApprox(*backV.front()->at(2)));
	BOOST_CHECK_EQUAL(v.back(), backV.back());
}

BOOST_AUTO_TEST_CASE( JSONBinaryPlainArrayTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	std::vector<int> v;
	auto typemeta = ms.addMeta(v);

	v.push_back(1);
	v.push_back(2);
	v.push_back(3);

	JSONSerializer js;
	json::Value val = js.serialize(v);

//	std::cout << json::write(val, true) << std::endl;

	Buffer<uint8> backBuf;
	BinaryJSONConverter::JSONToBinary(val, *typemeta, db, backBuf, true);

	BinaryBufferDeserializer bd(&backBuf);
	vector<int> backV;
	bd.deserialize(backV);

	BOOST_CHECK(v == backV);
}

BOOST_AUTO_TEST_CASE( JSONBinaryTransparentSerializableTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	ClassWithSharedPtr v(true);
	BOOST_CHECK(v.a == 5);
	v.b.check();
	v.s->check();

	auto typemeta = ms.addMeta(v);

	JSONSerializer js;
	json::Value val = js.serialize(v);

//	std::cout << json::write(val, true) << std::endl;

	Buffer<uint8> backBuf;
	BinaryJSONConverter::JSONToBinary(val, *typemeta, db, backBuf, true);

	BinaryBufferDeserializer bd(&backBuf);
	ClassWithSharedPtr backV;
	BOOST_CHECK(backV.a == 1);
	bd.deserialize(backV);

	BOOST_CHECK(v.a == backV.a);
	backV.b.check();
	backV.s->check();
}

BOOST_AUTO_TEST_CASE( JSONBinaryTransparentSerializableTopLevelTest )
{
	{
		MetaTypeDatabase db;
		MetaSerializer ms(db);

		boost::shared_ptr<BaseBaseClass> v;
		v = boost::make_shared<BaseBaseClass>(true);

		auto typemeta = ms.addMeta(v);

		JSONSerializer js;
		json::Value val = js.serialize(v);

		std::cout << json::write(val, true) << std::endl;

		Buffer<uint8> backBuf;
		BinaryJSONConverter::JSONToBinary(val, *typemeta, db, backBuf, true);

		BinaryBufferDeserializer bd(&backBuf);
		boost::shared_ptr<BaseBaseClass> backV;
		bd.deserialize(backV);

		backV->check();
	}
	{
		MetaTypeDatabase db;
		MetaSerializer ms(db);

		Degreef v(90.f);
		auto typemeta = ms.addMeta(v);

		JSONSerializer js;
		json::Value val = js.serialize(v);

		std::cout << json::write(val, true) << std::endl;

		Buffer<uint8> backBuf;
		BinaryJSONConverter::JSONToBinary(val, *typemeta, db, backBuf, true);

		BinaryBufferDeserializer bd(&backBuf);
		Degreef backV;
		bd.deserialize(backV);

		BOOST_CHECK_EQUAL(v, backV);
	}
}

BOOST_AUTO_TEST_CASE( JSONBinarySpecializedReflectTest )
{
	// Test datatype that does have a specialized reflect function
	// for the binary serializer
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	Stamped<RigidTransformCov3f> v;
	v.value().x() = 1;
	v.value().y() = 2;
	v.value().z() = 3;
	auto typemeta = ms.addMeta(v);

	// print out database for manual debugging purposes
//	foreach(auto p, db)
//	{
//		std::cout << p.first <<  p.second->toString() << std::endl;
//	}

	v.timestamp = Time::now();
	v.frameID = "myframe";

	// cannot use JSONSerializer here to create initial JSON,
	// must use BinarySerializer+binaryToJSON instead!
	// reason: RigidTransformCov3f reflection is different in JSONSerializer and
	// BinarySerializer (-> MetaSerializer), therefore the JSONSerializer does not
	// create JSON that the MetaSerializer understands/describes -> JSONToBinary would fail

	JSONValue val;
	Buffer<uint8> buf;
	BinaryBufferSerializer bs(&buf);
	bs.serialize(v);

	BinaryJSONConverter::binaryToJSON(buf, true, *typemeta, db, val);

	Buffer<uint8> backBuf;
	BinaryJSONConverter::JSONToBinary(val, *typemeta, db, backBuf, true);

	BinaryBufferDeserializer bd(&backBuf);
	Stamped<RigidTransformCov3f> backV;
	bd.deserialize(backV);

	BOOST_CHECK(v.timestamp == backV.timestamp);
	BOOST_CHECK(v.frameID == backV.frameID);
	BOOST_CHECK(v.value().x() == backV.value().x());
}


