/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file XMLDomPreprocessor.C
 *    Implementation of XMLDomPreprocessor.h
 *
 * @author Tim Langner
 * @date   2010/10/02
 */

#include <xml/XMLDomPreprocessor.h>
#include <error/Logging.h>
#include <factory/Factory.h>
#include <platform/Environment.h>
#include <utils/ResolveVariable.h>
#include <utils/PackageFinder.h>
#include <utils/PathFinder.h>
#include <utils/MakeString.h>
namespace mira {

///////////////////////////////////////////////////////////////////////////////

std::ostream& operator<<(std::ostream& s, const XMLVariableValue& x) { return s << x.value; }

///////////////////////////////////////////////////////////////////////////////

std::string resolveDefineVariable(const std::string& var,
                                  const XMLVariablesMap& variables)
{
	auto i = variables.find(var);
	if (i == variables.end())
		MIRA_THROW(XInvalidConfig, "No defined variable '" << var << "' found");
	return i->second;
}


void XMLDomPreprocessor::preprocessAll(XMLDom::iterator& iNode,
                                       IncludePathStack& ioPathStack)
{
		preprocessAllLowLevel(iNode, ioPathStack);
}

std::string XMLDomPreprocessor::resolveContent(std::string content)
{
	content = resolveVariables(content, boost::bind(resolveDefineVariable, _1, variables));
	content = resolveEnvironmentVariables(content, false);
	content = resolveSpecialVariables(content, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2), false);
	return content;
}

XMLDom::iterator XMLDomPreprocessor::parseInclude(XMLDom::iterator& ioNode,
                                                  IncludePathStack& ioPathStack)
{
	try {
		Path filename(resolvePath(ioNode.get_attribute<std::string>("file"), false));
		std::string parse = ioNode.get_attribute<std::string>("parse", "xml");

		Path currentPath = Path(ioNode.uri()).branch_path();
		if (currentPath.empty() && !ioPathStack.empty())
			currentPath = ioPathStack.back().branch_path();

		if (!filename.has_root_directory())
			filename = currentPath / filename;
		filename = resolvePath(filename);

		if (parse=="xml")
		{
			if (std::find(ioPathStack.begin(),
			              ioPathStack.end(),
			              filename) != ioPathStack.end())
				MIRA_THROW(XInvalidConfig, "Error file '"
				           << filename.string() << "' is included recursively");
			MIRA_LOG(DEBUG) << "Including file " << filename.string();

			ioPathStack.push_back(filename);
			std::string select = ioNode.get_attribute<std::string>("select", "");
			std::vector<std::string> selectedNodeNames;
			boost::split(selectedNodeNames, select, boost::is_from_range(':',':'));

			XMLDom xml;
			xml.loadFromFile(filename.string());
			XMLDom::iterator root = xml.root();

			std::list<IteratorPair> includeNodes;
			size_t includeCounter = 0;

			// if we have a select attribute add all paths of it to the include list
			if (!select.empty())
			{
				std::string as = ioNode.get_attribute<std::string>("as", "");

				foreach(std::string& s, selectedNodeNames)
				{
					// check for slash * (slash means we add all childs of the node instead of the node itself
					if (s.length() >= 2 && s[s.length()-2] == '/' && s[s.length()-1] == '*')
					{
						s.erase(s.length()-2, 2);
						XMLDom::iterator n = root.find(s);
						includeCounter += std::distance(n.begin(), n.end());
						preprocessAll(n, ioPathStack);
						includeNodes.push_back(IteratorPair({n.begin(), n.end(), as}));
					}
					else // add node directly
					{
						XMLDom::iterator start = root.find(s);
						if (start != root.end())
							++includeCounter;
						preprocessAll(start, ioPathStack);
						XMLDom::iterator end = start;
						++end;
						includeNodes.push_back(IteratorPair({start, end, as}));
					}
				}
			}
			else // no select attribute -> add all childs of root
			{
				includeCounter += std::distance(root.begin(), root.end());
				preprocessAll(root, ioPathStack);
				includeNodes.push_back(IteratorPair({root.begin(), root.end(), ""}));
			}

			// iterate over all pairs of iterators and include all nodes
			foreach(IteratorPair& p, includeNodes)
			{
				XMLDom::iterator it = p.first;
				for(; it != p.second; ++it)
				{
					preprocessAll(it, ioPathStack);
					if (it != p.second) {
						XMLDom::iterator inserted = ioNode.insert_before(it);
						if (!p.name.empty())
							inserted.setName(p.name);
					}
				}
			}
			if (includeCounter == 0)
			{
				MIRA_LOG(WARNING) << "Including empty content from " << filename.string()
					<< " to " << ioNode.uri() <<  " (line " << ioNode.line() << ")";
			}
			ioNode = ioNode.remove();
			ioPathStack.pop_back();
		}
		else if (parse=="text")
		{
			if (!boost::filesystem::exists(filename))
				MIRA_THROW(XFileNotFound, "The file " << filename.string() << " does not exist!");
			MIRA_LOG(DEBUG) << "Including content of file " << filename.string();
			std::ifstream ifs(filename.string().c_str());
			std::string content((std::istreambuf_iterator<char>(ifs)),
			                    (std::istreambuf_iterator<char>()));
			// remove end of line (end of file)
			if (!content.empty() && content[content.length()-1] == '\n')
				content.erase(content.length()-1);
			ioNode = ioNode.replace_by_content(resolveContent(content));
		}
		else
			MIRA_THROW(XInvalidConfig, "Unknown content for parse attribute. Must be 'xml' or 'text'")
		return ioNode;
	} catch(Exception& ex) {
		if (!ioNode.uri().empty()) {
			MIRA_RETHROW(ex, "in file included at: " <<  ioNode.uri() <<
			             " (line " << ioNode.line() << ")");
		}
		else
			throw;
	}
}

void XMLDomPreprocessor::processSpecialVariables(XMLDom::iterator& node)
{
	// Replace all occurrences of ${XXX} in attributes and content
	// with either defines or environment variables
	// Replace all occurrences of ${pattern variable} in attributes and content
	// by calling resolveXMLVariables
	XMLDom::attribute_iterator attr = node.attribute_begin();
	for( ; attr != node.attribute_end(); ++attr)
		attr = resolveContent(attr.value());
	XMLDom::content_iterator content = node.content_begin();
	for( ; content != node.content_end(); ++content)
		content = resolveContent(*content);
}

void checkCondition(XMLDom::iterator& ioNode,
                    XMLDom::iterator& ifNode,
                    XMLVariablesMap& ioVariables, bool condition)
{
	XMLDom::iterator possibleElse = ifNode; ++possibleElse;
	// if condition is met use content of <if> tag
	if (condition)
	{
		// remove if node if content is empty
		if (ifNode.begin() != ifNode.end())
		{
			XMLDom::iterator start = ifNode;
			XMLDom::iterator i = ifNode.begin();
			for (; i!=ifNode.end();++i) {
				start = start.insert_after(i);
			}
		}
		// remove else node if exists
		if (possibleElse != ioNode.end() && *possibleElse == "else")
			possibleElse.remove();
		// remove <if> node and continue
		ifNode = ifNode.remove();
	}
	else
	{
		// remove <if> node
		ifNode = ifNode.remove();
		if (possibleElse != ioNode.end() && *possibleElse == "else")
		{
			// remove else node if content is empty
			if (ifNode.begin() != ifNode.end())
			{
				XMLDom::iterator start = ifNode;
				XMLDom::iterator i = ifNode.begin();
				for (; i!=ifNode.end();++i)
					start = start.insert_after(i);
			}
			ifNode = possibleElse.remove();
		}
	}
}

bool XMLDomPreprocessor::parseCondition(XMLDom::iterator& node)
{
	auto it = node.attribute_begin();

	bool condition = true;
	processSpecialVariables(node);

	auto v = variables.find(it.name());
	if ((v == variables.end()) ||
	    (v->second != it.value()))
	{
		condition = false;
	}

	++it;
	if (it != node.attribute_end())
		MIRA_THROW(XInvalidConfig,
		           "Error replacing <" << *node << "> node "
		           "- more than one attribute specified "
		           "(use <" << *node << "_all/any> instead)");

	return condition;
}

bool XMLDomPreprocessor::parseConditionsRequireAll(XMLDom::iterator& node)
{
	auto it = node.attribute_begin();

	bool condition = true;
	processSpecialVariables(node);

	for(; it != node.attribute_end(); ++it)
	{
		auto v = variables.find(it.name());
		if ((v == variables.end()) ||
		    (v->second != it.value()))
		{
			condition = false;
			break;
		}
	}

	return condition;
}

bool XMLDomPreprocessor::parseConditionsRequireAny(XMLDom::iterator& node)
{
	auto it = node.attribute_begin();

	bool condition = false;
	processSpecialVariables(node);

	for(; it != node.attribute_end(); ++it)
	{
		auto v = variables.find(it.name());
		if ((v != variables.end()) &&
		    (v->second == it.value()))
		{
			condition = true;
			break;
		}
	}

	return condition;
}

bool XMLDomPreprocessor::parseExists(XMLDom::iterator& node)
{
	auto it = node.attribute_begin();
	bool condition = true;

	if (it.name() == "class")
	{
		if (!ClassFactory::isClassRegistered(it.value()))
		{
			condition = false;
		}
	}
	else if (it.name() == "file")
	{
		try
		{
			std::string value = resolveVariables(it.value(),
												 boost::bind(resolveDefineVariable, _1, variables));
			value = resolveSpecialVariables(value, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2));
			it = value;
			Path p = resolvePath(it.value());
			if (!boost::filesystem::exists(p))
			{
				condition = false;
			}
		} catch(...)
		{
			condition = false;
		}
	}
	else if (it.name() == "var")
	{
		if (variables.count(it.value()) == 0)
		{
			condition = false;
		}
	}
	else if (it.name() == "env")
	{
		try
		{
			resolveEnvironmentVariable(it.value());
		}
		catch(...)
		{
			condition = false;
		}
	}
	else if (it.name() == "package")
	{
		try
		{
			std::string value = resolveVariables(it.value(),
												 boost::bind(resolveDefineVariable, _1, variables));
			value = resolveSpecialVariables(value, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2));
			it = value;
			findPackage(it.value());
		} catch(...)
		{
			condition = false;
		}
	}
	else
		MIRA_THROW(XInvalidConfig,
		           "Error replacing <" << *node << "> node "
		           "- unknown attribute '" << it.name() << "'");

	++it;
	if (it != node.attribute_end())
		MIRA_THROW(XInvalidConfig,
		           "Error replacing <" << *node << "> node "
		           "- more than one attribute specified "
		           " (use <" << *node << "_all/any> instead)");

	return condition;
}

bool XMLDomPreprocessor::parseExistsRequireAll(XMLDom::iterator& node)
{
	auto it = node.attribute_begin();
	bool condition = true;
	for(; it != node.attribute_end(); ++it)
	{
		if (it.name().rfind("class", 0) != std::string::npos)
		{
			if (!ClassFactory::isClassRegistered(it.value()))
			{
				condition = false;
				break;
			}
		}
		else if (it.name().rfind("file", 0) != std::string::npos)
		{
			try
			{
				std::string value = resolveVariables(it.value(),
				                                     boost::bind(resolveDefineVariable, _1, variables));
				value = resolveSpecialVariables(value, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2));
				it = value;
				Path p = resolvePath(it.value());
				if (!boost::filesystem::exists(p))
				{
					condition = false;
					break;
				}
			} catch(...)
			{
				condition = false;
				break;
			}
		}
		else if (it.name().rfind("var", 0) != std::string::npos)
		{
			if (variables.count(it.value()) == 0)
			{
				condition = false;
				break;
			}
		}
		else if (it.name().rfind("env", 0) != std::string::npos)
		{
			try
			{
				resolveEnvironmentVariable(it.value());
			}
			catch(...)
			{
				condition = false;
				break;
			}
		}
		else if (it.name().rfind("package", 0) != std::string::npos)
		{
			try
			{
				std::string value = resolveVariables(it.value(),
				                                     boost::bind(resolveDefineVariable, _1, variables));
				value = resolveSpecialVariables(value, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2));
				it = value;
				findPackage(it.value());
			} catch(...)
			{
				condition = false;
				break;
			}
		}
		else
			MIRA_THROW(XInvalidConfig,
			           "Error replacing <" << *node << "> node "
			           "- unknown attribute '" << it.name() << "'");
	}
	return condition;
}

bool XMLDomPreprocessor::parseExistsRequireAny(XMLDom::iterator& node)
{
	auto it = node.attribute_begin();
	bool condition = false;
	for(; it != node.attribute_end(); ++it)
	{
		if (it.name().rfind("class", 0) != std::string::npos)
		{
			if (ClassFactory::isClassRegistered(it.value()))
			{
				condition = true;
				break;
			}
		}
		else if (it.name().rfind("file", 0) != std::string::npos)
		{
			try
			{
				std::string value = resolveVariables(it.value(),
				                                     boost::bind(resolveDefineVariable, _1, variables));
				value = resolveSpecialVariables(value, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2));
				it = value;
				Path p = resolvePath(it.value());
				if (boost::filesystem::exists(p))
				{
					condition = true;
					break;
				}
			} catch(...) {}
		}
		else if (it.name().rfind("var", 0) != std::string::npos)
		{
			if (variables.count(it.value()) > 0)
			{
				condition = true;
				break;
			}
		}
		else if (it.name().rfind("env", 0) != std::string::npos)
		{
			try
			{
				resolveEnvironmentVariable(it.value());
				condition = true;
				break;
			}
			catch(...) {}
		}
		else if (it.name().rfind("package", 0) != std::string::npos)
		{
			try
			{
				std::string value = resolveVariables(it.value(),
				                                     boost::bind(resolveDefineVariable, _1, variables));
				value = resolveSpecialVariables(value, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2));
				it = value;
				findPackage(it.value());
				condition = true;
				break;
			} catch(...) {}
		}
		else
			MIRA_THROW(XInvalidConfig,
			           "Error replacing <" << *node << "> node "
			           "- unknown attribute '" << it.name() << "'");
	}
	return condition;
}

void XMLDomPreprocessor::preprocessAllLowLevel(XMLDom::iterator& ioNode,
                                               IncludePathStack& ioPathStack)
{
	mAddedInfoToException = false;

	XMLVariableValue filePathBacktrack = variables["filePath"];
	variables["filePath"] = XMLVariableValue(Path(ioNode.uri()).parent_path().string(),
	                                         MakeString() << ioNode.uri() << " (line " << ioNode.line() << ")");

	XMLDom::iterator node = ioNode.begin();
	for ( ; node != ioNode.end(); )
	{
		try {
			if (*node == "define" || *node == "var")
			{
				processSpecialVariables(node);
				XMLDom::attribute_iterator ai = node.attribute_begin();
				if (ai == node.attribute_end())
					MIRA_THROW(XInvalidConfig, "Error in define/var. "
					           "No attribute specified. "
					           "Usage <define X=\"Y\"/> or <var X=\"Y\"/>");
				XMLDom::Attribute attribute = *ai;
				++ai;
				bool overwrite = false;
				for ( ; ai != node.attribute_end(); ++ai)
				{
					if (ai.name() == "overwrite" &&
						(ai.value() == "true" || ai.value() == "1"))
						overwrite = true;
				}
				if (variables.count(attribute.first) == 0 || overwrite)
					variables[attribute.first] = XMLVariableValue(attribute.second,
					                                              MakeString() << node.uri() << " (line " << node.line() << ")");
				// remove node
				node = node.remove();
			}
			else
			if (*node == "include")
			{
				processSpecialVariables(node);
				node = parseInclude(node, ioPathStack);
			}
			else
			if (*node == "if" || *node == "if_not")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseCondition(node);
				if (*node == "if_not")
					condition = !condition;
				checkCondition(ioNode, node, variables, condition);
			}
			else
			if (*node == "if_all" || *node == "if_not_all")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseConditionsRequireAll(node);
				if (*node == "if_not_all")
					condition = !condition;
				checkCondition(ioNode, node, variables, condition);
			}
			else
			if (*node == "if_any" || *node == "if_not_any")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseConditionsRequireAny(node);
				if (*node == "if_not_any")
					condition = !condition;
				checkCondition(ioNode, node, variables, condition);
			}
			else
			if (*node == "if_exists" || *node == "if_not_exists")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseExists(node);
				if (*node == "if_not_exists")
					condition = !condition;
				checkCondition(ioNode, node, variables, condition);
			}
			else
			if (*node == "if_exists_all" || *node == "if_not_exists_all")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseExistsRequireAll(node);
				if (*node == "if_not_exists_all")
					condition = !condition;
				checkCondition(ioNode, node, variables, condition);
			}
			else
			if (*node == "if_exists_any" || *node == "if_not_exists_any")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseExistsRequireAny(node);
				if (*node == "if_not_exists_any")
					condition = !condition;
				checkCondition(ioNode, node, variables, condition);
			}
			else
			if(*node=="notice" || *node=="warning" || *node=="error" || *node=="abort")
			{
				processSpecialVariables(node);
				auto itMessage = node.find_attribute("message");
				if (itMessage == node.attribute_end())
					MIRA_THROW(XInvalidConfig, "Error in <notice>, <warning>, <error> or <abort> tag: "
					          "No message specified. Usage: <warning message=\"My message\"/>");
				if (*node =="abort")
				{
					MIRA_THROW(XInvalidConfig, "Aborting! " << itMessage.value() << " ("<<node.uri()<<":"<<node.line()<<")")
				}
				SeverityLevel lvl = WARNING;
				if(*node=="notice")
					lvl = NOTICE;
				else if(*node=="error")
					lvl = ERROR;
				MIRA_LOG(lvl) << itMessage.value() << " ("<<node.uri()<<":"<<node.line()<<")";
				node = node.remove();
			}
			else
			if (*node=="assert" || *node == "assert_not")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseCondition(node);
				if (*node == "assert_not")
					condition = !condition;
				if (!condition)
				{
					std::string message;
					if (node.content_begin() != node.content_end())
						message = *node.content_begin();
					MIRA_THROW(XInvalidConfig, "Assertion! Conditions are not met: " << message)
				}
				else // remove assert node
					node = node.remove();
			}
			else
			if (*node=="assert_all" || *node == "assert_not_all")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseConditionsRequireAll(node);
				if (*node == "assert_not_all")
					condition = !condition;
				if (!condition)
				{
					std::string message;
					if (node.content_begin() != node.content_end())
						message = *node.content_begin();
					MIRA_THROW(XInvalidConfig, "Assertion! Conditions are not met: " << message)
				}
				else // remove assert node
					node = node.remove();
			}
			else
			if (*node=="assert_any" || *node == "assert_not_any")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseConditionsRequireAny(node);
				if (*node == "assert_not_any")
					condition = !condition;
				if (!condition)
				{
					std::string message;
					if (node.content_begin() != node.content_end())
						message = *node.content_begin();
					MIRA_THROW(XInvalidConfig, "Assertion! Conditions are not met: " << message)
				}
				else // remove assert node
					node = node.remove();
			}
			else
			if (*node=="assert_exists" || *node == "assert_not_exists")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseExists(node);
				if (*node == "assert_not_exists")
					condition = !condition;
				if (!condition)
				{
					std::string message;
					if (node.content_begin() != node.content_end())
						message = *node.content_begin();
					MIRA_THROW(XInvalidConfig, "Assertion! Conditions are not met: " << message)
				}
				else // remove assert node
					node = node.remove();
			}
			else
			if (*node=="assert_exists_all" || *node == "assert_not_exists_all")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseExistsRequireAll(node);
				if (*node == "assert_not_exists_all")
					condition = !condition;
				if (!condition)
				{
					std::string message;
					if (node.content_begin() != node.content_end())
						message = *node.content_begin();
					MIRA_THROW(XInvalidConfig, "Assertion! Conditions are not met: " << message)
				}
				else // remove assert node
					node = node.remove();
			}
			else
			if (*node=="assert_exists_any" || *node == "assert_not_exists_any")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
							  "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseExistsRequireAny(node);
				if (*node == "assert_not_exists_any")
					condition = !condition;
				if (!condition)
				{
					std::string message;
					if (node.content_begin() != node.content_end())
						message = *node.content_begin();
					MIRA_THROW(XInvalidConfig, "Assertion! Conditions are not met: " << message)
				}
				else // remove assert node
					node = node.remove();
			}
			else
			if (*node == "else")
			{
				MIRA_THROW(XInvalidConfig, "Error stray 'else' tag found. "
						   "(Missing 'if', 'if_not', 'if_exists' or 'if_not_exists' tag)");
			}
			else
			{
				processSpecialVariables(node);
				preprocessAllLowLevel(node, ioPathStack);
				++node;
			}
		} catch(Exception& ex) {
			if (!mAddedInfoToException && !node.uri().empty())
			{
				mAddedInfoToException = true;
				MIRA_RETHROW(ex, "while parsing tag starting at: " <<
				             node.uri() << " (line " << node.line() << ")");
			}
			else
				throw;
		}
	}

	variables["filePath"] = filePathBacktrack;
}

void XMLDomPreprocessor::preprocessXML(XMLDom& ioXml)
{
	XMLDom::iterator root = ioXml.root();
	IncludePathStack pathStack;
	pathStack.push_back(Path(ioXml.uri()));
	preprocessAll(root, pathStack);
	variables.erase("filePath"); // not meaningful after processing, remove to avoid confusion
}

void XMLDomPreprocessor::registerXMLVariableCallback(const std::string& pattern, Callback&& callback)
{
	mCallbacks[pattern] = std::move(callback);
}

std::string XMLDomPreprocessor::resolveXMLVariables(const std::string& pattern, const std::string& var)
{
	std::string p = boost::to_lower_copy(pattern);
	if (p == "find")
		return findProjectPath(var);
	if (p=="findpkg")
		return findPackage(var).string();
	if (p == "env")
		return resolveEnvironmentVariable(var);
	foreach(auto cb, mCallbacks)
		if (cb.first == p)
			return cb.second(var);
	MIRA_THROW(XInvalidConfig, "Could not resolve pattern, "
			   "variable combination " << pattern << ":" << var);
	return "";
}

void preprocessXML(XMLDom& ioXml, XMLVariablesMap& variables)
{
	XMLDomPreprocessor pp(variables);
	pp.preprocessXML(ioXml);
	variables = pp.variables;
}

///////////////////////////////////////////////////////////////////////////////

}
