/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyNode.C
 *    Parts of implementation of PropertyNode.h.
 *
 * @author Erik Einhorn
 * @date   2011/12/20
 */

#include <boost/algorithm/string.hpp>

#include <serialization/PropertyNode.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

PropertyNode::~PropertyNode()
{
	if(mParent!=NULL) {
		auto it = std::find(mParent->mChildren.begin(), mParent->mChildren.end(),this);
		if(it!=mParent->mChildren.end())
			mParent->mChildren.erase(it);
	}
	while(!mChildren.empty())
		delete *(mChildren.begin()); // will also remove child from mChildren set
}

const PropertyNode* PropertyNode::findChildNode(const std::vector<std::string>& ids,
                                                std::size_t level) const
{
	assert(level < ids.size());

	const PropertyNode* child = NULL;
	const std::vector<PropertyNode*>& v = children();
	foreach(const PropertyNode* c, v)
		if(c->id()==ids[level]) {
			child = c;
			break;
		}

	if(child==NULL) // not found
		return NULL;

	// continue on next level
	++level;
	if(level == ids.size()) // reached end of id list
		return child;

	return child->findChildNode(ids,level);
}

const PropertyNode* PropertyNode::findChildNode(const std::string& id) const
{
	std::vector<std::string> ids;
	boost::split(ids, id, boost::is_from_range('.','.'));
	foreach (auto& id, ids)
		boost::algorithm::replace_all(id, "\\", ".");
	return findChildNode(ids,0);
}


PropertyNode* PropertyNode::findChildNode(const std::string& id)
{
	const PropertyNode* This = this;
	return const_cast<PropertyNode*>(This->findChildNode(id));
}

void PropertyNode::setFromString(const std::string& value) {
	if(isReadOnly())
		MIRA_THROW(XLogical, "Cannot set value of read-only property.");
	json::Value jsonOld = getAsJSON();
	if (jsonOld.type() == json_spirit::str_type)
		setFromJSON(json::Value(value));
	else
	{
		json::Value json;
		json::read(value, json);
		setFromJSON(json);
	}
}

std::string PropertyNode::getAsString() const {
	json::Value json = getAsJSON();
	if (json.type() == json_spirit::str_type)
		return json.get_str();
	return json::write(json, false);
}


void PropertyNode::addChild(PropertyNode* child) {
	mChildren.push_back(child);
	child->mParent = this;
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

