/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SymbolTable.C
 *    Implementation of SymbolTable.h
 *
 * @author Erik Einhorn, Christian Martin
 * @date   2010/12/27
 */

#include <error/private/SymbolTableLinux.h>

#ifndef bfd_get_section_flags
// The macros have been renamed and restructured. See the following link for further details
// https://sourceware.org/git/gitweb.cgi?p=binutils-gdb.git;h=fd3619828e94a24a92cddec42cbc0ab33352eeb4
#define bfd_get_section_flags(NOT_USED, ptr) bfd_section_flags(ptr)
#define bfd_get_section_vma(NOT_USED, ptr) bfd_section_vma(ptr)
#define bfd_get_section_size bfd_section_size
#endif

#include <stdlib.h> // for malloc
#include <string.h> // strcmp
#include <assert.h>

#include <error/Logging.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////
// Implementation of SymbolTable::SourceLookup

class SymbolTable::SourceLookup
{
public:
	asymbol** syms;
	bfd_vma pc;
	const char* filename;
	const char* functionname;
	unsigned int line;
	bool found;

public:

	SourceLookup() : filename(NULL), functionname(NULL), line(0), found(false) {}

	void find(bfd* abfd, asymbol** iSyms, bfd_vma iPC)
	{
		syms = iSyms;
		pc = iPC;
		bfd_map_over_sections(abfd, staticFindAddressInSection, this);
	}

private:

	void findAddressInSection(bfd* abfd, asection* section)
	{
		using namespace std;

		if (found)
			return;

		if ((bfd_get_section_flags(abfd, section) & SEC_ALLOC) == 0)
			return;

		bfd_vma vma;
		vma = bfd_get_section_vma(abfd, section);
		if (pc < vma)
			return;

		bfd_size_type size;
		size = bfd_get_section_size(section);
		if (pc >= vma + size)
			return;

		found = bfd_find_nearest_line(abfd, section, syms, pc - vma,
									&filename, &functionname, &line);
	}

	// Look for an address in a section.  This is called via bfd_map_over_sections.
	static void staticFindAddressInSection(bfd* abfd, asection* section, void* data)
	{
		SourceLookup* This = (SourceLookup*) data;
		This->findAddressInSection(abfd, section);
	}

};

///////////////////////////////////////////////////////////////////////////////

SymbolTable::Tables SymbolTable::sTableCache;

///////////////////////////////////////////////////////////////////////////////

const SymbolTable* SymbolTable::forBinary(const std::string& file)
{
	SymbolTable* table = NULL;
	Tables::iterator i = sTableCache.find(file);
	if(i==sTableCache.end()) {
		table = new SymbolTable(file);
		sTableCache[file] = table;
	} else
		table = i->second;

	assert(table!=NULL);
	return table;
}

///////////////////////////////////////////////////////////////////////////////

SymbolTable::SymbolTable(const std::string& fileName) :
	mAbfd(NULL), mSyms(NULL)
{
	// open the file
	mAbfd = bfd_openr(fileName.c_str(), NULL);
	if(mAbfd==NULL) // failed, so the symbol table remains empty
		return;

	if (bfd_check_format(mAbfd, bfd_archive)) {
		MIRA_LOG(DEBUG) << "Cannot get addresses from archive: '" << fileName << "'";
		return;
	}

	char **matching;
	if (! bfd_check_format_matches(mAbfd, bfd_object, &matching)) {
		MIRA_LOG(DEBUG) << "bfd_check_format_matches failed: '" << fileName << "'";
	}

	// get the symbol table
	mIsDynamic = false;

	if ((bfd_get_file_flags(mAbfd) & HAS_SYMS) == 0) {
		MIRA_LOG(DEBUG) << "No syms: '" << fileName << "'";
		return;
	}

	mNrSymbols = bfd_read_minisymbols(mAbfd, false /* not dynamic */, (void **)&mSyms, &mSymbolSize);
	if (mNrSymbols == 0)  {
		// No static symbols, trying dynamic...
		mIsDynamic = true;
		mNrSymbols = bfd_read_minisymbols(mAbfd, true /* dynamic */, (void **)&mSyms, &mSymbolSize);
	}

	if (mNrSymbols < 0) {
		MIRA_LOG(DEBUG) << "Symboltable error: " << bfd_get_filename(mAbfd) << std::endl;
		free(mSyms);
		mSyms=NULL;
	}


	// create a symbol we can use for different purposes
	mDummySymbol = bfd_make_empty_symbol(mAbfd);
	assert(mDummySymbol != NULL);
}

SymbolTable::~SymbolTable()
{
	if(mSyms!=NULL)
		free(mSyms);
	if(mAbfd!=NULL)
		bfd_close (mAbfd);
}

//////////////////////////////////////////////////////////////////////////////

SymbolTable::SourceLocation SymbolTable::sourceLocation(const char* hexAddr, const char* symbol, int64 symbolOffset) const
{
	return sourceLocation(bfd_scan_vma(hexAddr, NULL, 16), symbol, symbolOffset);
}

SymbolTable::SourceLocation SymbolTable::sourceLocation(uint64 addr, const char* symbol, int64 symbolOffset) const
{
	using namespace std;

	if(mSyms==NULL) // make sure we have a symbol table
		return SourceLocation();

	// get offset of code relocation
	int64 relocOffset = 0;
	if(symbol!=NULL) { // if we don't have determined it yet, get the reloc offset
		uint64 relocatedSymbolAddr = addr - symbolOffset;
		uint64 unrelocatedAddr = getSymbolAddr(symbol);
		if(unrelocatedAddr!=0) {
			relocOffset = relocatedSymbolAddr - unrelocatedAddr + 1;
			//cout << "offset: " << hex << relocOffset << endl;
		}
	}

	//cout << "symboffset: " << symbolOffset << endl;

	// relocate the address using the precomputed offset
	addr -= relocOffset;

	SourceLookup lookup;
	lookup.find(mAbfd, mSyms, addr);
	if(lookup.filename!=NULL)
		return SourceLocation(lookup.filename, lookup.line);
	else
		return SourceLocation();
}

//////////////////////////////////////////////////////////////////////////////

uint64 SymbolTable::getSymbolAddr(const char* symbol) const
{
	bfd_byte *from = (bfd_byte *)mSyms;
	bfd_byte *fromend = from + mNrSymbols * mSymbolSize;
	int index = 0;
	for (; from < fromend; from += mSymbolSize, index++)
	{
		asymbol *sym = bfd_minisymbol_to_symbol(mAbfd, mIsDynamic, from, mDummySymbol);
		assert(sym != NULL);

		symbol_info syminfo;
		bfd_get_symbol_info(mAbfd, sym, &syminfo);

		if(strcmp(syminfo.name,symbol)==0)
			return syminfo.value;
	}
	return 0;
}

///////////////////////////////////////////////////////////////////////////////

}
