/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ProgramOptions.h
 *    Class for accessing command line parameters.
 *
 * @author Tim Langner
 * @date   2010/09/22
 */

#ifndef _MIRA_PROGRAMOPTIONS_H_
#define _MIRA_PROGRAMOPTIONS_H_

#ifndef Q_MOC_RUN
#include <boost/program_options.hpp>
#endif
#include <utils/Singleton.h>

/// For easier access to the program options instance
#define MIRA_CMDLINE mira::ProgramOptions::instance()

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * A central singleton class for accessing command line parameters
 * Underlying class is boost::program::options see 
 * http://www.boost.org/doc/libs/1_44_0/doc/html/program_options.html
 * Usage:
 * \code
 * // First add some options
 * MIRA_CMDLINE.getDescriptions().add_options()
 *     ("version,v", "Displays version");
 *     ("file,f", boost::program_options::value<std::string>()->default_value("File.xml"), "My file")
 *     ("count,c", boost::program_options::value<int>(), "Count of something");
 *
 * // Then ask if the parameters exist
 * ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();
 * if ( vmap.count("version") )
 * {
 *     std::cout << "Version 1.0.0" << std::endl;
 *	   ::exit(-1);
 * }
 * //  and get the value
 * if ( vmap.count("count") )
 *     int c = vmap["count"].as<int>();
 * \endcode
 */
class MIRA_BASE_EXPORT ProgramOptions :
	public ExplicitSingleton<ProgramOptions>
{
public:
	typedef boost::program_options::options_description Description;
	typedef boost::program_options::positional_options_description PositionalDescription;
	typedef boost::program_options::variables_map VariableMap;

	/**
	 * Constructor that takes command line arguments
	 */
	ProgramOptions(int argc, char** argv);

	/**
	 * Constructor that takes command line arguments as vector
	 */
	ProgramOptions(const std::vector<std::string>& args);

	/**
	 * Return the option description map. Provided to add more options.
	 */
	Description& getDescriptions()
	{
		return mOptionDescriptions;
	}

	/**
	 * Return the option description map. Provided to add more options.
	 * Calling this method also enables positional arguments.
	 * Positional arguments are identified by their position rather than their
	 * name. For positional options only the relative order of the arguments
	 * matters.
	 */
	PositionalDescription& getPositionalDescriptions()
	{
		mEnablePositionalArguments = true;
		return mPositionalDescriptions;
	}

	/**
	 * Return the option variable map. Command line arguments are parsed 
	 * according to the given description map and stored in this variable map.
	 */
	void getOptions(VariableMap& oMap) const;

	/**
	 * Return the option variable map. Command line arguments are parsed
	 * according to the given description map and stored in this variable map.
	 */
	VariableMap getOptions() const
	{
		VariableMap vmap;
		getOptions(vmap);
		return vmap;
	}

	/**
	 * Return the command line arguments as passed in the constructor
	 * @return Command line options as a pair of number of arguments and pointer
	 * to arguments.
	 */
	const std::vector<std::string>& getArgs() const
	{
		return mArgs;
	}

private:
	Description mOptionDescriptions;    ///< The description map
	PositionalDescription mPositionalDescriptions;
	bool mEnablePositionalArguments;
	std::vector<std::string> mArgs;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
