/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file StlCollections.h
 *    Helpers for serialization and deserialization of STL containers
 *
 * @author Erik Einhorn
 * @date   2010/07/16
 */

#ifndef _MIRA_STLCOLLECTIONS_H_
#define _MIRA_STLCOLLECTIONS_H_

#include <utils/Foreach.h>

#include <serialization/ReflectCollection.h>
#include <serialization/RecursiveMemberReflector.h> // for exception types

#include <serialization/IsPointerOrSharedPointer.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

namespace serialization { // our private namespace

//////////////////////////////////////////////////////////////////////////////

// SEQUENCES: list, vector, deque

/**
 * Reflects generic sequential containers like vectors, list, deque
 * (Read Only / Serialization)
 *
 * The resulting format is like:
 * \code
 *   item ...
 *   item ...
 *   item ...
 *   ...
 * \endcode
 */
template<typename Reflector, typename Container>
struct ReflectReadSeq
{
	typedef typename Container::iterator iterator;
	typedef typename Container::value_type type;

	static void reflect(Reflector& r, Container& c)
	{
		// store the size first
		uint32 count = c.size();
		MIRA_REFLECT_CALL(Reflector, r, (typeName<Container>() + " ReflectCollectionCount").c_str(),
		                  (ReflectCollectionCount<Reflector,Container>::reflect(r, count)));

		MIRA_REFLECT_CALL(Reflector, r, (typeName<Container>() + " ReflectCollectionItems").c_str(),
		                  (ReflectCollectionItems<Reflector,Container>::reflect(r, c)));
	}
};

/**
 * Reflects generic sequencial containers like vectors, list, deque
 * (Write Only / Deserialization)
 *
 * The resulting format is like:
 * \code
 *   item ...
 *   item ...
 *   item ...
 *   ...
 * \endcode
 *
 */
template<typename Reflector, typename Container>
struct ReflectWriteSeq
{
	typedef typename Container::iterator iterator;
	typedef typename Container::value_type type;

	static void reflect(Reflector& r, Container& c)
	{
		c.clear(); // clear previous content

		// restore the size first
		uint32 count;
		MIRA_REFLECT_CALL(Reflector, r, (typeName<Container>() + " ReflectCollectionCount"),
			              (ReflectCollectionCount<Reflector,Container>::reflect(r, count)));

		// reserve the space and create the elements
		c.resize(count);

		MIRA_REFLECT_CALL(Reflector, r, (typeName<Container>() + " ReflectCollectionItems"),
		                  (ReflectCollectionItems<Reflector,Container>::reflect(r, c)));
	}
};

template<typename Reflector, typename Container>
struct ReflectReadSetItems
{
	typedef typename Container::value_type type;

	static void reflect(Reflector& r, Container& c)
	{
		// store each item
		int id=0;
		foreach(const type& cv, c)
		{
			// deserializing references to non pointer-type keys will not work
			// (because these cannot be deserialized 'in place',
			// the serialization framework is unable to track the correct memory address),
			// so we prevent creating references during serialization already
			// by use of the REFLECT_CTRLFLAG_TEMP_TRACKING
			type& nonconstv = const_cast<type&>(cv);
			if(IsPointerOrSharedPointer<type>::value) {
				MIRA_MEMBER_WITH_ID(r, "item", "item["+toString(id)+"]", nonconstv, "");
			} else {
				MIRA_MEMBER_WITH_ID(r, "item", "item["+toString(id)+"]", nonconstv,
				                    "", REFLECT_CTRLFLAG_TEMP_TRACKING);
			}

			++id;
		}
	}
};

// SET: set, multiset
/**
 * Reflects generic set containers like set, multiset
 *
 * The resulting format is like:
 * \code
 *   item ...
 *   item ...
 *   item ...
 *   ...
 * \endcode

 */
template<typename Reflector, typename Container>
struct ReflectReadSet
{
	static void reflect(Reflector& r, Container& c)
	{
		// store the size first
		uint32 count = c.size();
		MIRA_REFLECT_CALL(Reflector, r, (typeName<Container>() + " ReflectCollectionCount"),
		                  (ReflectCollectionCount<Reflector,Container>::reflect(r, count)));

		MIRA_REFLECT_CALL(Reflector, r, (typeName<Container>() + " ReflectReadSetItems"),
		                  (ReflectReadSetItems<Reflector, Container>::reflect(r, c)));
	}
};

template<typename Reflector, typename Container>
struct ReflectWriteSetItems
{
	typedef typename Container::value_type type;
	typedef typename Container::iterator iterator;

	static void reflect(Reflector& r, Container& c, uint32 count)
	{
		iterator hint = c.begin();

		// restore each item and insert into the set
		for(uint32 id=0; id<count; ++id)
		{
			// reflect the value
			type v;

			if(IsPointerOrSharedPointer<type>::value) {
				MIRA_MEMBER_WITH_ID(r, "item", "item["+toString(id)+"]", v, "");
			} else {
				// we have to deserialize the key to a temp object, cannot restore references to it !!!
				MIRA_MEMBER_WITH_ID(r, "item", "item["+toString(id)+"]", v, "", REFLECT_CTRLFLAG_TEMP_TRACKING);
			}
			// add the value to the container
			hint = c.insert(hint, v);
		}
	}
};

/**
 * Deserializes generic set containers like set, multiset
 *
 * The resulting format is like:
 * \code
 *   item ...
 *   item ...
 *   item ...
 *   ...
 * \endcode
 */
template<typename Reflector, typename Container>
struct ReflectWriteSet
{
	static void reflect(Reflector& r, Container& c)
	{
		c.clear(); // clear previous content

		// restore the size first
		uint32 count;
		MIRA_REFLECT_CALL(Reflector, r, (typeName<Container>() + " ReflectCollectionCount"),
		                  (ReflectCollectionCount<Reflector,Container>::reflect(r, count)));

		MIRA_REFLECT_CALL(Reflector, r, (typeName<Container>() + " ReflectWriteSetItems"),
		                  (ReflectWriteSetItems<Reflector, Container>::reflect(r, c, count)));
	}
};

// Helper classes and functions for map pair reflection
// In contrast to normal pairs, the 'first' element in a map pair has const-type
// and is not object trackable (with few exceptions)
template<typename Derived, typename key_type, typename value_type>
struct ReadMapPairHelper {

	typedef typename value_type::second_type mapped_type;

	ReadMapPairHelper(const value_type& p)	: mP(p) {}

	void reflect(Derived& r) {
		// deserializing references to non pointer-type keys will not work
		// (because these cannot be deserialized 'in place',
		// the serialization framework is unable to track the correct memory address),
		// so we prevent creating references during serialization already
		// by use of the REFLECT_CTRLFLAG_TEMP_TRACKING
		key_type& nonconstkey = const_cast<key_type&>(mP.first);
		if(IsPointerOrSharedPointer<key_type>::value)
			r.member("First", nonconstkey, "");
		else
			r.member("First", nonconstkey, "", REFLECT_CTRLFLAG_TEMP_TRACKING);

		mapped_type& nonconstv = const_cast<mapped_type&>(mP.second);
		r.property("Second", nonconstv, "");
	}

	const value_type& mP;
};


template<typename Derived, typename Container>
void reflectReadMapPair(Derived& r, const char* itemName, uint32 id,
                        const typename Container::value_type& p)
{
	typedef typename Container::key_type   key_type;
	typedef typename Container::value_type   value_type;

	ReadMapPairHelper<Derived, key_type, value_type> rp(p);
	MIRA_MEMBER_WITH_ID(r, itemName, std::string(itemName)+"["+toString(id)+"]", rp, "");
}

template<typename Derived, typename Container>
struct WriteMapPairHelper {

	typedef typename Container::key_type key_type;
	typedef typename Container::mapped_type mapped_type;
	typedef typename Container::value_type value_type;
	typedef typename Container::iterator iterator;

	WriteMapPairHelper(Container& c, iterator& hint)
		: mC(c), mHint(hint) {}

	void reflect(Derived& r) {
		key_type key;
		if(IsPointerOrSharedPointer<key_type>::value)
			r.member("First", key, "");
		else
			r.member("First", key, "", REFLECT_CTRLFLAG_TEMP_TRACKING);

		// insert a dummy value with the key
		iterator it = mC.insert(mHint, value_type(key, mapped_type()));
		mHint = it; // the iterator will be insertion hint for next element
		r.property("Second", it->second, "");
	}

	Container& mC;
	iterator& mHint;
};

template<typename Derived, typename Container>
void reflectWriteMapPair(Derived& r, const char* itemName, const char* keyName, uint32 id,
                         Container& c, typename Container::iterator& ioHint)
{
	typedef typename Container::iterator iterator;
	typedef typename Container::value_type   value_type;
	typedef typename Container::key_type     key_type;
	typedef typename Container::mapped_type  mapped_type;

	std::string nameId = "["+toString(id)+"]";
	std::string itemNameWithId = std::string(itemName)+nameId;
	WriteMapPairHelper<Derived, Container> wp(c, ioHint);
	MIRA_MEMBER_WITH_ID(r, itemName, itemNameWithId, wp, "");
}

template<typename Reflector, typename Container>
struct ReflectReadMapItems
{
	typedef typename Container::value_type  value_type;

	static void reflect(Reflector& r, Container& c)
	{
		// store each item
		int id=0;
		foreach(const value_type& p, c)
		{
			reflectReadMapPair<Reflector, Container>(r, "item", id, p);
			++id;
		}
	}
};

/**
 * reflects generic associative containers like map, multimap, hash_map
 *
 * The resulting format is like:
 * \code
 *   item
 *       First (key)
 *       Second (mapped)
 *   item
 *       First (key)
 *       Second (mapped)
 *   item
 *       First (key)
 *       Second (mapped)
 *   ...
 * \endcode
 */
template<typename Reflector, typename Container>
struct ReflectReadMap
{
	static void reflect(Reflector& r, Container& c)
	{
		// store the size first
		uint32 count = c.size();
		MIRA_REFLECT_CALL(Reflector, r, (typeName<Container>() + " ReflectCollectionCount").c_str(),
		                  (ReflectCollectionCount<Reflector,Container>::reflect(r, count)));

		MIRA_REFLECT_CALL(Reflector, r, (typeName<Container>() + " ReflectReadMapItems").c_str(),
		                  (ReflectReadMapItems<Reflector, Container>::reflect(r, c)));
	}
};


template<typename Reflector, typename Container>
struct ReflectWriteMapItems
{
	typedef typename Container::iterator iterator;

	static void reflect(Reflector& r, Container& c, uint32 count)
	{
		iterator hint = c.begin();

		// restore each item and insert into the map
		for(uint32 id=0; id<count; ++id)
			reflectWriteMapPair(r, "item", "key", id, c, hint);
	}
};

/**
 * reflects generic associative containers like map, multimap, hash_map
 *
 * The resulting format is like:
 * \code
 *   item
 *       First (key)
 *       Second (mapped)
 *   item
 *       First (key)
 *       Second (mapped)
 *   item
 *       First (key)
 *       Second (mapped)
 *   ...
 * \endcode
 */
template<typename Reflector, typename Container>
struct ReflectWriteMap
{
	static void reflect(Reflector& r, Container& c)
	{
		c.clear(); // clear previous content

		// restore the size first
		uint32 count;
		MIRA_REFLECT_CALL(Reflector, r, (typeName<Container>() + " ReflectCollectionCount").c_str(),
		                  (ReflectCollectionCount<Reflector,Container>::reflect(r, count)));

		MIRA_REFLECT_CALL(Reflector, r, (typeName<Container>() + " ReflectWriteMapItems").c_str(),
		                  (ReflectWriteMapItems<Reflector, Container>::reflect(r, c, count)));
	}
};

//////////////////////////////////////////////////////////////////////////////

} //namespace

//////////////////////////////////////////////////////////////////////////////

} //namespace

#endif
