/*
 * Copyright (C) 2019 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ReflectorMacros.h
 *    Macros for use with reflectors
 *
 * @author Christof Schröter
 * @date   2019/01/11
 */

#ifndef _MIRA_REFLECTORMACROS_H_
#define _MIRA_REFLECTORMACROS_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

namespace serialization {

template <typename Reflector>
struct ReflectCallHelper
{
	typedef typename Reflector::ReflectState State;

	// this one calls Reflector::pre/postReflect()
	struct __ReflectBarrier {
		static State preCommand(Reflector& r, const std::string& context) {
			static const std::string ctxt = context;
			return r.template preReflect(ctxt.c_str());
		}
		static void postCommand(Reflector& r, const State& state) {
			r.template postReflect(state);
		}
	};

	// this one does NOT try to call Reflector::pre/postReflect()
	struct __NoReflectBarrier {
		static State preCommand(Reflector&, const std::string& context) { return State(); }
		static void postCommand(Reflector& r, const State& state) {}
	};

	// select whether pre/postReflect must be called on the reflector
	// (at compilation time!)
	typedef typename boost::mpl::eval_if<
				typename Reflector::requireReflectBarriers,
					boost::mpl::identity<__ReflectBarrier>,
					boost::mpl::identity<__NoReflectBarrier>
		>::type BarrierType;
};

} // namespace

///////////////////////////////////////////////////////////////////////////////

/**
 * Whenever a reflection function calls another function that is independently maintained,
 * the call should be marked to the reflector. The reflector will then regard these as
 * separate reflection blocks and keep a separate state for them. E.g. the
 * BinarySerializer can allow these blocks to have separate reflection versions and will
 * make sure these version numbers are correctly serialized (and deserialized later).
 * Without making the reflector aware, it might know only one state for these different blocks,
 * e.g. the BinarySerializer would potentially assume a version is reported twice for the same
 * object (possibly with different version numbers!), and throw an exception.
 * MIRA_REFLECT_CALL is used to insert the code required to
 * mark the start and end of a call to a separate reflection function to the reflector.
 * It can thus be seen as a generalization of the concept of using
 * reflectBase()/\ref MIRA_REFLECT_BASE instead of calling Base::reflect() directly.
 * 
 * \code
 * template<typename Reflector>
 * void reflectMyStruct(Reflector& r, MyStruct& c) {
 * {
 *     MIRA_REFLECT_CALL(Reflector, r, "MyStruct someReflectMethod", (someReflectMethod(r, c)));
 * }
 * \endcode
 */
#define MIRA_REFLECT_CALL(ReflectorType, reflector, context, COMMAND)					\
{																						\
	typedef typename serialization::ReflectCallHelper<ReflectorType>::BarrierType type;	\
																						\
	typename ReflectorType::ReflectState state = type::preCommand(reflector, context);	\
	COMMAND;																			\
	type::postCommand(reflector, state);												\
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
