/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertySerializer.h
 *    Serializer that handles properties and creates PropertyNodes.
 *
 * @author Erik Einhorn
 * @date   2010/12/31
 */

#ifndef _MIRA_PROPERTYSERIALIZER_H_
#define _MIRA_PROPERTYSERIALIZER_H_

#include <assert.h>

#include <serialization/PropertyReflector.h>
#include <serialization/PropertyNode.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 * A special PropertyReflector that creates a PropertyNode for each reflected
 * property. Moreover, it updates existing property nodes, if a property is
 * updated and reflected again.
 */
class PropertySerializer : public PropertyReflector<PropertySerializer>
{
	typedef PropertyReflector<PropertySerializer> Base;

public:

	PropertySerializer() : mCurrentNode(NULL)
	{}

public:

	/**
	 * Is called by invokePropertyMemberOverwrite() and will call
	 * the actual Base::invokePropertyMemberOverwrite() method that
	 * recursively examines all properties.
	 * This method is a helper which is overloaded below, to avoid
	 * recursive examination for properties with Getters and Setters which
	 * is not implemented yet.
	 */
	template <typename T>
	void continueWithBaseInvoke(T& member, const ReflectMemberMeta& meta,
	                            PropertyHint&& hint, bool isReadOnly) {
		Base::invokePropertyMemberOverwrite(member, meta,
		                                    std::move(hint),isReadOnly);
	}

	/**
	 * Specialization of the above method for properties with getters and
	 * setters. For those, a recursive reflection is not yet implemented.
	 */
	template <typename Getter, typename Setter>
	void continueWithBaseInvoke(Accessor<Getter,Setter>& accessor,
	                            const ReflectMemberMeta& meta, PropertyHint&& hint,
	                            bool isReadOnly) {
	}

	// implemented from base class

	/**
	 * Invokes the PropertyReflector on the specified member.
	 * If the corresponding property node for the member exists already, its
	 * content and sub-nodes will be updated. Otherwise, a new property node
	 * is created. All sub-properties of the member are invoked recursively.
	 */
	template<typename T>
	void invokePropertyMemberOverwrite(T& member, const ReflectMemberMeta& meta,
	                                   PropertyHint&& hint, bool isReadOnly)
	{
		// current node becomes parent node
		PropertyNode* parentNode = mCurrentNode;

		if ( meta.id != NULL )
		{
			PropertyNode* node = NULL;

			if(parentNode!=NULL)
				node = parentNode->findChildNode(meta.id);

			if(node!=NULL) {
				// if node already exists, use it (then we are updating)
				TypedPropertyNodeImpl<T>* typedNode =
								dynamic_cast<TypedPropertyNodeImpl<T>*>(node);
				assert(typedNode!=NULL);
				typedNode->update(member);
			} else { // otherwise create a new node

				if(parentNode!=NULL) // inherit read only flag
					isReadOnly |= parentNode->isReadOnly();

				node = new TypedPropertyNodeImpl<T>(meta.id, meta.name,
										meta.comment, member, isReadOnly);

				// set the given hint to the node (by swapping the contents)
#if defined(MIRA_GNUC_VERSION)
# if MIRA_GNUC_VERSION >= 40600
				std::swap(node->mHint, (PropertyHint&)hint);
# else
				std::swap(node->mHint, hint);
# endif
#else
				std::swap(node->mHint, hint); // << this should be the preferred version, that is supported by GCC < 4.6 and VC 2010
#endif
				if(parentNode!=NULL)
					parentNode->addChild(node);
			}

			assert(node!=NULL);

			// set the node as current node
			mCurrentNode=node;
		}

		// now do the real reflector invoke (but only if T is a type that
		// is worth to continue. If it is a GetterSetter, we stop here)
		continueWithBaseInvoke(member,meta, std::move(hint), isReadOnly);

		// go back to the parent node (but only if it is not NULL, otherwise we
		// did not have a root node, so keep the top most node that we have
		// created, it will be returned by reflectProperties)
		if(parentNode!=NULL)
			mCurrentNode=parentNode;
	}


	void itemName(const std::string& name) {
		if(mCurrentNode) {
			// replace items only
			if(mCurrentNode->mID.compare(0,4,"item")==0)
				mCurrentNode->setName(name);
		}
	}

public:

	// public entry points

	/**
	 * Reflects the properties of the specified 'object'. The properties will
	 * be stored as a sub-node of 'root'. The name of that sub-node will be
	 * 'name'.
	 */
	template <typename T>
	void reflectProperties(PropertyNode* root, const std::string& name, T& object) {
		mCurrentNode=root;
		this->property(name.c_str(), object, "");
	}

	/**
	 * Same as above, but additionally, a unique id can be specified for the
	 * object. In the above method, the id will be equal to the name.
	 */
	template <typename T>
	void reflectProperties(PropertyNode* root, const std::string& name,
	                       const std::string& id, T& object) {
		mCurrentNode=root;
		this->property(name.c_str(), id, object, "");
	}

	/**
	 * Reflects the properties of the specified 'object'. In contrast to the
	 * above methods, the created property sub-node is not added to any
	 * parent node. Instead it is returned by the method.
	 * The name of the created property node can be specified by 'name'.
	 */
	template <typename T>
	PropertyNode* reflectProperties(const std::string& name, T& object) {
		mCurrentNode=NULL;
		this->property(name.c_str(), object, "");
		assert(mCurrentNode!=NULL);
		return mCurrentNode;
	}

	/**
	 * Same as above, but additionally, a unique id can be specified for the
	 * object. In the above method, the id will be equal to the name.
	 */
	template <typename T>
	PropertyNode* reflectProperties(const std::string& name,
	                                const std::string& id, T& object) {
		mCurrentNode=NULL;
		this->property(name.c_str(), id, object, "");
		assert(mCurrentNode!=NULL);
		return mCurrentNode;
	}

public:

	PropertyNode* mCurrentNode;
};

///////////////////////////////////////////////////////////////////////////////

template <typename T>
inline void TypedPropertyNodeImpl<T>::synchronize()
{
	PropertySerializer s;
	s.reflectProperties(this->parent(), this->name().c_str(),
	                    this->id(), *this->mValue);
}


// specialization for properties of pointers
template <typename T>
inline void TypedPropertyNodeImpl<T*>::synchronize()
{
	PropertySerializer s;
	s.reflectProperties(this->parent(), this->name().c_str(),
	                    this->id(), this->mPointer);
}

// specialization for properties with setter
template <typename Getter, typename Setter>
inline void TypedPropertyNodeImpl<Accessor<Getter,Setter>>::synchronize()
{
	PropertySerializer s;
	s.reflectProperties(this->parent(), this->name().c_str(),
	                    this->id(), this->mAccessor);
}

///@cond INTERNAL

// specializations for STL containers

namespace serialization { // our private namespace

///////////////////////////////////////////////////////////////////////////////

template<typename Reflector, typename Container> struct ReflectReadSetItems;

/**
 * Specialization for Property Serializer which creates read-only properties
 * for set elements instead of members (which would be ignored by this reflector).
 */
template<typename Container>
struct ReflectReadSetItems<PropertySerializer, Container>
{
	typedef typename Container::value_type type;

	static void reflect(PropertySerializer& r, Container& c)
	{
		// store each item
		int id=0;
		foreach(const type& v, c)
		{
			r.roproperty("item", "item["+toString(id)+"]", v, "");
			++id;
		}
	}
};

template<typename Reflector, typename Container> struct ReflectReadMapItems;

/**
 * Specialization for Property Serializer which creates read-only 'key' properties
 * for map keys and 'item' properties for items, instead of pairs (which do not seem
 * to have a proper editor delegate?)
 */
template<typename Container>
struct ReflectReadMapItems<PropertySerializer, Container>
{
	typedef typename Container::value_type  value_type;

	static void reflect(PropertySerializer& r, Container& c)
	{
		// store each item
		int id=0;
		foreach(value_type& p, c)
		{
			r.roproperty("key", "key["+toString(id)+"]", p.first, "");

			// the values can be reflected/deserialized with tracking
			MIRA_PROPERTY_WITH_ID(r, "item", "item["+toString(id)+"]", p.second, "");

			++id;
		}
	}
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

///@endcond

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_PROPERTYSERIALIZER_H_ */
