/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Truncate.h
 *    Truncates a floating point value to a given number of decimals.
 *
 * @author Tim Langner
 * @date   2012/01/04
 */

#ifndef _MIRA_TRUNCATE_H_
#define _MIRA_TRUNCATE_H_

#include <cmath>
#include <type_traits>
#include <platform/Types.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Truncates a floating point value to a given number of decimals.
 * \code
 * truncate(3.141f,1) will give 3.1f
 * \endcode
 * @param value Value to truncate
 * @param decimals Decimals that should be kept
 * @return Truncated value
 * @ingroup MathModule
 */
template<typename T>
T truncate(T value, uint32 decimals)
{
	static_assert(std::is_floating_point<T>::value, "truncate must be used with floating point types");

	T factor = std::pow((T)10, (int)decimals);
	if (value > 0)
	{
		T v = factor * value;
		return std::floor(v) / factor;
	}
	else
	{
		T v = factor * value;
		return std::ceil(v) / factor;
	}
}

///////////////////////////////////////////////////////////////////////////////

}

#endif 
