/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Power.h
 *    Functions for computing powers with constant integral exponent or
 *    constant integral exponent and base.
 *
 * @author Erik Einhorn
 * @date   2012/09/15
 */

#ifndef _MIRA_POWER_H_
#define _MIRA_POWER_H_

#include <cmath>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

namespace detail {

template <typename T, int N>
struct PowerComputer
{
	inline static T eval(T base) {
		return PowerComputer<T,N-1>::eval(base)*base;
	}
};

template <typename T>
struct PowerComputer<T,1>
{
	inline static T eval(T base) {
		return base;
	}
};

template <typename T>
struct PowerComputer<T,0>
{
	inline static T eval(T base) {
		return 1;
	}
};

}
///////////////////////////////////////////////////////////////////////////////

/**
 * Computes the power of 'base' with a constant integral exponent.
 * Instead of:
 * \code
 * float result = 0.4f*0.4f*0.4f*0.4f*0.4f*0.4f*0.4f*0.4f*0.4f*0.4f;
 * \endcode
 *
 * you can now write:
 * \code
 * float result = pow<10>(0.4f);
 * \endcode
 *
 * which is much better to read and produces exactly the same code. Hence, the
 * second variant is exactly as fast as the first one.
 *
 * @ingroup MathModule
 */
template <int exponent, typename T>
inline T pow(T base)
{
	return detail::PowerComputer<T,exponent>::eval(base);
}

/**
 * Computes base^2.
 * @ingroup MathModule
 */
template <typename T>
inline T pow2(T base)
{
	return base*base;
}

/**
 * Computes base^3.
 * @ingroup MathModule
 */
template <typename T>
inline T pow3(T base)
{
	return base*base*base;
}

///////////////////////////////////////////////////////////////////////////////

/**
 * Computes the power base^exponent at compile time.
 * The computation at compile time allows to use the value as constant
 * in other expressions (e.g. array sizes, etc).
 *
 * \code
 * Usage:
 *      cout << TPower<2,8>::value << endl; // prints 256
 * \endcode
 * @ingroup MathModule
 */
template<int base, int exponent>
struct TPower {
	enum {
		value = TPower<base,exponent-1>::value * base
	};
};

template<int base>
struct TPower<base,0> {
	enum {
		value = 1
	};
};


///////////////////////////////////////////////////////////////////////////////

// two workarounds for those guys that use the old C pow(double,double) method -
// this now delegate those calls to the preferred std::pow() methods
template <typename T, typename U>
inline T pow(T base, U exponent)
{
	return std::pow(base,exponent);
}

template <typename T>
inline T pow(T base, T exponent)
{
	return std::pow(base,exponent);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
