/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Exceptions.h
 *    Commonly used exception classes
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/04/01
 */

#ifndef _MIRA_EXCEPTIONS_H_
#define _MIRA_EXCEPTIONS_H_

#include <error/Exception.h>
#include <factory/Factory.h>

 /**
  * @ingroup ExceptionModule
  * Macro for easily defining a new serializable exception class.
  * Beside the name of the new exception class, the base class of the exception
  * must be specified.
  *
  * Example:
  * The following example defines a new exception class XMyNewException that is
  * derived from the XLogical exception.
  * \code
  * MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XMyNewException, XLogical)
  * \endcode
  */
#define MIRA_DEFINE_SERIALIZABLE_EXCEPTION(Ex, Base)                             \
	class Ex : public Base                                                       \
	{                                                                            \
		MIRA_OBJECT(Ex)                                                          \
	protected:                                                                   \
		friend class ClassFactoryDefaultConstClassBuilder;                       \
		Ex() MIRA_NOEXCEPT_OR_NOTHROW {}                                         \
                                                                                 \
	public:                                                                      \
		Ex(const std::string& msg, const char* file=NULL, int line=0) MIRA_NOEXCEPT_OR_NOTHROW :  \
			Base(msg, file, line) {}                                             \
                                                                                 \
		virtual void raise(bool recursive = false) { throw *this; }              \
	};

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class SerializableException : public Object, public Exception
{
	MIRA_OBJECT(SerializableException)

protected:
	friend class ClassFactoryDefaultConstClassBuilder;
	SerializableException() {}

public:
	SerializableException(const std::string& msg, const char* file=NULL, int line=0) MIRA_NOEXCEPT_OR_NOTHROW :
		Exception(msg, file, line) {}

	virtual ~SerializableException() MIRA_NOEXCEPT_OR_NOTHROW {}

	MIRA_SPLIT_REFLECT_MEMBER

	// We do serialization of base class elements here and only use most basic serialization features,
	// as things that we would like to use like serialization/ReflectorInterface.h and
	// serialization/adapters/std/list depend (indirectly) on the exceptions we will define below. 
	template<typename Reflector>
	void reflectRead(Reflector& r) {
		uint32 count = mInfos.size();
		r.member("InfoCount", count, "Number of info fields");
		uint64 n = 0;
		for (std::list<Exception::Info>::const_iterator it = mInfos.begin(); it != mInfos.end(); ++it) {
			r.member(("Info"+std::to_string(n)+"Message").c_str(), it->message, "Message");
			r.member(("Info"+std::to_string(n)+"File").c_str(), it->file, "File");
			r.member(("Info"+std::to_string(n)+"Line").c_str(), it->line, "Line");
			++n;
		}
		r.member("Stack", mStack, "Call stack");
		r.member("ThreadID", mThreadID, "Thread ID");
	}

	template<typename Reflector>
	void reflectWrite(Reflector& r) {
		uint32 count;
		r.member("InfoCount", count, "Number of info fields");
		std::string message;
		std::string file;
		int line;
		for (uint64 n = 0; n < count; ++n) {
			r.member(("Info"+std::to_string(n)+"Message").c_str(), message, "Message");
			r.member(("Info"+std::to_string(n)+"File").c_str(), file, "File");
			r.member(("Info"+std::to_string(n)+"Line").c_str(), line, "Line");
			mInfos.emplace_back(message, file, line);
		}
		
		r.member("Stack", mStack, "Call stack");
		r.member("ThreadID", mThreadID, "Thread ID");
	}

	// C++ does not support polymorphic throw, i.e. throwing a base class reference that holds a subclass object
	// will invoke the base class catch clause. Therefore we need to do the throw in a virtual method.
	virtual void raise(bool recursive = false) = 0;
};

typedef boost::shared_ptr<SerializableException> SerializableExceptionPtr;

/**
 * @ingroup ExceptionModule
 * A class that signals an exception which can only be detected at
 * runtime, a so called runtime exception.
 * In contrast to logical exceptions (see XLogical) runtime exceptions
 * can not be detected or avoided at compile time.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XRuntime, SerializableException)

/**
 * @ingroup ExceptionModule
 * A exception that occurred due to an invalid configuration.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XInvalidConfig, XRuntime)

/**
 * @ingroup ExceptionModule
 * Generic IO exception.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XIO, XRuntime)

/**
 * @ingroup ExceptionModule
 * An exception that occurred in conjunction with network connections
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XConnection, XRuntime)

/**
 * @ingroup ExceptionModule
 * An exception that occurred after a failed system call.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XSystemCall, XRuntime)

/**
 * @ingroup ExceptionModule
 * An exception that indicates that a given file was not found.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XFileNotFound, XIO)

/**
 * @ingroup ExceptionModule
 * An exception that is thrown when a method is not supported or implemented
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XNotImplemented, XRuntime)

/**
 * @ingroup ExceptionModule
 * An exception that is thrown when a timeout occurs
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XTimeout, XRuntime)

/**
 * @ingroup ExceptionModule
 * An exception that occurred due to invalid program logic. This kind of exception
 * could be detected at design and compile time and should be fixed in the
 * final release. In bug free software only runtime exceptions should occur since
 * those cannot be avoided at compile time.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XLogical, SerializableException)

/**
 * @ingroup ExceptionModule
 * An exception that signals invalid parameters for method calls, etc.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XInvalidParameter, XLogical)

/**
 * @ingroup ExceptionModule
 * An exception that signals an assertion, i.e. a required condition that was
 * not fulfilled. This usually is the result of a violated pre- or post-
 * condition.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XAssertion, XLogical)

/**
 * @ingroup ExceptionModule
 * An exception that is thrown if a cast to a target type cannot be performed.
 * This usually is the result of a dynamic cast that is invoked at runtime.
 * It should be used to generate a more verbose exception than std::bad_cast
 * (which is thrown by the dynamic_cast operator as the result of a failed cast
 *  to a reference type).
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XBadCast, XRuntime)

/**
 * @ingroup ExceptionModule
 * An exception that occurs if an entity is accessed without  permission.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XAccessViolation, XRuntime)

/**
 * @ingroup ExceptionModule
 * An exception that occurs when allocation of requested memory fails.
 * Based on Exception and not on Runtime to match the STL exception categories.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XBadAlloc, SerializableException)

///////////////////////////////////////////////////////////////////////////////

}

#endif
