/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Exception.h
 *    Exception base class.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/04/01
 */

#ifndef _MIRA_EXCEPTION_H_
#define _MIRA_EXCEPTION_H_

#include <assert.h>

#include <string.h>

#include <exception>
#include <sstream>

#include <list>

#include <error/CallStack.h>
#include <thread/ThreadID.h>
#include <utils/NoExcept.h>

///////////////////////////////////////////////////////////////////////////////

/// Define for maximum call stack depth
#ifndef MIRA_MAX_CALLSTACK_SIZE
# define MIRA_MAX_CALLSTACK_SIZE 16
#endif

/// Start at this position in call stack
#define MIRA_CALLSTACK_START 2

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup ExceptionModule
 * Macro for throwing an exception.
 * The macro additionally stores the filename and the line of the source file
 * where the exception was thrown. Additional information for the user
 * can be specified as stream in the second macro parameter.
 *
 * Example:
 * \code
 *     void readValue(istream& ioStream)
 *     {
 *         if(...error while reading from io...)
 *             MIRA_THROW(XIO, "Invalid format, the tag '" << foo "' is missing");
 *     }
 * \endcode
 */
#define MIRA_THROW(ex, msg)                                         \
{                                                                   \
	std::ostringstream ex_str;                                      \
	ex_str << msg;                                                  \
	throw ex(ex_str.str(), __FILE__, __LINE__).addStackInfo<ex>();  \
}

/**
 * @ingroup ExceptionModule
 * Same as MIRA_THROW, except that no stack trace information is stored.
 * This macro can be used when many exceptions are thrown and catched and no
 * stack information is necessary, to avoid performance penalties.
 */
#define MIRA_THROW_NOSTACK(ex, msg)              \
{                                                \
	std::ostringstream ex_str;                   \
	ex_str << msg;                               \
	throw ex(ex_str.str(), __FILE__, __LINE__)   \
}

/**
 * @ingroup ExceptionModule
 * Same as MIRA_THROW, except that the stack trace information that
 * is stored is provided externally.
 * This macro can be used when an exception is thrown to signal
 * another original exception, e.g. in RPC handling.
 */
#define MIRA_THROW_EXTSTACK(ex, msg, stack, thread)  \
{                                                    \
	std::ostringstream ex_str;                       \
	ex_str << msg;                                   \
	throw ex(ex_str.str(), __FILE__, __LINE__)       \
		.addExternalStackInfo<ex>(stack, thread);    \
}

/**
 * @ingroup ExceptionModule
 * Macro for rethrowing an exception with file and line information and
 * for adding additional information.
 *
 * Example:
 *
 * In the example the readValue() function is called for
 * a file stream. readValue() will throw exceptions if the given
 * stream violates certain syntax rules. The readFile() function
 * below can now catch the exception in order to provide additional
 * information like the filename or line number to the user. This
 * information can be specified in the second parameter of
 * MIRA_RETHROW.
 *
 * \code
 *     void readFile(const char* filename)
 *     {
 *         ifstream s(filename);
 *         try {
 *             readValue(s);
 *         } catch(mira::Exception& e) {
 *             // append additional information and rethrow
 *             MIRA_RETHROW(e, "filename: " << filename);
 *         }
 *     }
 * \endcode
 */
#define MIRA_RETHROW(ex, msg)                    \
{                                                \
	std::ostringstream ex_str;                   \
	ex_str << msg;                               \
	ex.addInfo(ex_str.str(),__FILE__, __LINE__); \
	throw; /* rethrow */                         \
}

 /**
  * @ingroup ExceptionModule
  * Macro for easily defining a new compatible exception class.
  * Beside the name of the new exception class, the base class of the exception
  * must be specified.
  *
  * Example:
  * The following example defines a new exception class XMyNewException that is
  * derived from the XLogical exception.
  * \code
  * MIRA_DEFINE_EXCEPTION(XMyNewException, XLogical)
  * \endcode
  */
#define MIRA_DEFINE_EXCEPTION(Ex, Base)                                        \
	class Ex : public Base                                                     \
	{                                                                          \
	public:                                                                    \
		Ex(const std::string& msg, const char* file=NULL, int line=0) MIRA_NOEXCEPT_OR_NOTHROW : \
			Base(msg, file, line) {}                                           \
		                                                                       \
		virtual ~Ex() MIRA_NOEXCEPT_OR_NOTHROW {}                              \
	};

///////////////////////////////////////////////////////////////////////////////

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup ExceptionModule
 * Base class for exceptions. The what()-text contains file name and line
 * number where the exception was thrown as well as additional information
 * that was specified in MIRA_THROW and MIRA_RETHROW.
 *
 * This exception class also keeps track of the call stack, which can
 * be used to improve tracing of errors and exceptions.
 *
 * For detailed information see @ref ExceptionPage.
 *
 * @see MIRA_THROW, MIRA_RETHROW
 */
class MIRA_BASE_EXPORT Exception : public std::exception
{
protected:
	Exception() MIRA_NOEXCEPT_OR_NOTHROW {}

public:

	/**
	 * The constructor.
	 * @note Ususally this constructor is not called directly, instead
	 *       MIRA_THROW should be used to create and throw an exception.
	 *
	 * @see MIRA_THROW
	 */
	Exception(const std::string& message, const char* file=NULL, int line=0) MIRA_NOEXCEPT_OR_NOTHROW
	{
		addInfo(message, file, line);
	}

	/// Destructor
	virtual ~Exception() MIRA_NOEXCEPT_OR_NOTHROW {}

	/**
	 * Adds additional information to the exception.
	 * @note Ususally this method is not called directly, instead
	 *       MIRA_RETHROW should be used to add additional information and
	 *       to rethrow the exception.
	 *
	 * @see MIRA_THROW, MIRA_RETHROW
	 *
	 */
	void addInfo(const std::string& message, const char* file=NULL, int line=0)
	{
		mInfos.push_back(Info(message, file ? std::string(file) : "", line));
	}

	/**
	 * Returns the text of exception containing the information given in
	 * MIRA_THROW and MIRA_RETHROW as well as the file name and line number
	 * where it was thrown.
	 *
	 * @see MIRA_THROW, MIRA_RETHROW
	 */
	virtual const char* what() const MIRA_NOEXCEPT_OR_NOTHROW;


	/**
	 * Similar to what(). Returns the text of exception containing the
	 * information given in MIRA_THROW and MIRA_RETHROW without the file name
	 * and line number in a single line string.
	 *
	 * @see MIRA_THROW, MIRA_RETHROW
	 */
	std::string message() const MIRA_NOEXCEPT_OR_NOTHROW;

	/**
	 * Returns the state of the callstack at the moment when the exception was thrown.
	 */
	const CallStack& callStack() const { return mStack; }


	/// Returns the id of the thread where the exception was thrown.
	ThreadID getThreadID() const { return mThreadID; }

public:

	/**
	 * FOR INTERNAL USE ONLY.
	 * Stores the current callstack and thread id within the exception.
	 * This method is called by the MIRA_THROW macro and is for internal
	 * use. You never need to call this method manually.
	 */
	template <typename DerivedException>
	DerivedException& addStackInfo()
	{
		// save the call stack:
		mStack = CallStack::backtrace(MIRA_MAX_CALLSTACK_SIZE, MIRA_CALLSTACK_START);
		mThreadID = getCurrentThreadID();
		return (DerivedException&)*this;
	}

	/**
	 * Stores the provided callstack and thread id within the exception.
	 * This can be used to create an exception with a callstack from an external source
	 * (e.g. for an exception originally thrown in an RPC).
	 */
	template <typename DerivedException>
	DerivedException& addExternalStackInfo(const CallStack& stack, ThreadID thread)
	{
		mStack = stack;
		mThreadID = thread;
		return (DerivedException&)*this;
	}

public:

	/**
	 * The info packet that is added in MIRA_THROW and MIRA_RETHROW
	 */
	struct Info {
		std::string message;	/// The exception message
		std::string file;		/// The file the exception occurred
		int         line;		/// The line the exception occurred

	public:
		Info(const std::string& iMessage, const std::string& iFile, int iLine) :
			message(iMessage), file(iFile), line(iLine) {}

		std::string what(std::size_t messageWidth) const;
	};


	/// Returns the first info packet that describes the location where the exception has occured
	const Info& getInfo() const { assert(!mInfos.empty()); return mInfos.front(); }

protected:
	std::list<Info> mInfos;
	CallStack       mStack;
	ThreadID        mThreadID;
	mutable std::string mMessage; ///< as cache for what()
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
