/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeTimeView.C
 *    Implementation of TapeTimeView.h.
 *
 * @author Tim Langner
 * @date   2012/01/01
 */

#include <TapeTimeView.h>

#include <QMouseEvent>
#include <QToolTip>

#include <TapeEditor.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

TapeTimeView::TapeTimeView(QWidget* parent, TapeEditor* editor) :
	ScrollView(parent)
{
	mEditor = editor;
	setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	setFrameStyle(QFrame::Panel | QFrame::Raised);
	setFocusPolicy(Qt::NoFocus);
	setMouseTracking(true);
	const QFont& font = this->font();
	setFont(QFont(font.family(), font.pointSize() - 1));
}

void TapeTimeView::resizeEvent(QResizeEvent* resizeEvent)
{
	ScrollView::resizeEvent(resizeEvent);
	updateContents();
}

void TapeTimeView::mousePressEvent(QMouseEvent* mouseEvent)
{
	mDragState = DragNone;

	if (mouseEvent->button() == Qt::LeftButton)
	{
		QPoint pos = viewportToContents(mouseEvent->pos());
		checkDragging(pos);
	}
	ScrollView::mousePressEvent(mouseEvent);
}

void TapeTimeView::mouseMoveEvent(QMouseEvent* mouseEvent)
{
	QPoint pos = viewportToContents(mouseEvent->pos());
	if (mDragState != DragNone)
	{
		int64 t = mEditor->getTimeFromPixel(pos.x());
		ScrollView::ensureVisible(pos.x(), pos.y(), 16, 0);
		if (mDragState == DragSelectionStart)
			mEditor->setSelectionStart(t);
		else
		if (mDragState == DragSelectionEnd)
			mEditor->setSelectionEnd(t);
		QToolTip::showText(QCursor::pos(), mEditor->getStringFromTime(t), viewport());
		mEditor->getDataView()->viewport()->update();
		mEditor->getThumbView()->update();
		viewport()->update();
	}
	else
	{
		checkDragging(pos);
		mDragState = DragNone;
	}
}

void TapeTimeView::mouseReleaseEvent(QMouseEvent* mouseEvent)
{
	ScrollView::mouseReleaseEvent(mouseEvent);

	if (mDragState != DragNone)
	{
		mDragState = DragNone;
		mEditor->getDataView()->viewport()->update();
		mEditor->getThumbView()->update();
		viewport()->update();
		unsetCursor();
	}
}

void TapeTimeView::updatePixmap(int cx, int cy)
{
	QWidget *viewport = this->viewport();
	int w = viewport->width();
	int h = viewport->height();

	if (w < 1 || h < 1)
		return;

	const QPalette& pal = palette();

	mPixmap = QPixmap(w, h);
	mPixmap.fill(pal.window().color());

	QPainter painter(&mPixmap);
	painter.initFrom(this);
	const QFontMetrics& fm = painter.fontMetrics();
	int y1 = h - 7;
	int y2 = h - 1;

	int64 t = mEditor->getTimeFromPixel(cx);
	t = (t/mEditor->getTimeStep())*mEditor->getTimeStep()-mEditor->getTimeStep();
	int x = mEditor->getPixelFromTime(t)-cx;
	int steps = 0;
	int subStep = mEditor->getTimeStep() / mEditor->getSubTimeSteps();
	while (x < w)
	{
		if (x>=0)
		{
			if (steps == 0)
			{
				painter.setPen(pal.base().color().value() < 0x7f
									? pal.light().color() : pal.dark().color());

				QString txt = mEditor->getStringFromTime(t, true);
				painter.drawText(x, fm.ascent(), txt);
				painter.setPen(pal.mid().color());
				painter.drawLine(x, y1, x, y2);
				painter.setPen(pal.light().color());
				painter.drawLine(x + 1, y1, x + 1, y2);
			}
			else
			{
				painter.setPen(pal.mid().color());
				painter.drawLine(x, y1+4, x, y2);
				painter.setPen(pal.light().color());
				painter.drawLine(x + 1, y1+4, x + 1, y2);
			}
		}
		if (steps == 0)
			steps = mEditor->getSubTimeSteps();
		--steps;
		t += subStep;
		x = mEditor->getPixelFromTime(t)-cx;
	}
}

void TapeTimeView::updateContents(const QRect& rect)
{
	updatePixmap(contentsX(), contentsY());
	ScrollView::updateContents(rect);
}

void TapeTimeView::updateContents()
{
	updatePixmap(contentsX(), contentsY());
	ScrollView::updateContents();
}

void TapeTimeView::onContentsMoving(int cx, int cy)
{
	if (contentsX() != cx)
		setContentsPos(cx, contentsY());
}

void TapeTimeView::drawContents(QPainter* painter, const QRect& rect)
{
	painter->drawPixmap(rect, mPixmap, rect);

	int cx = contentsX();
	int h = height() - 4;
	int d = (h >> 2);
	int x = mEditor->getSelectionStartX() - cx;
	if (x >= rect.left() && x <= rect.right() + d)
	{
		QPolygon poly(3);
		poly.putPoints(0, 3,
			x + d, h - d,
			x, h,
			x, h - d);
		painter->setPen(Qt::green);
		painter->setBrush(Qt::green);
		painter->drawPolygon(poly);
	}
	x = mEditor->getSelectionEndX() - cx;
	if (x >= rect.left() - d && x <= rect.right())
	{
		QPolygon poly(3);
		poly.putPoints(0, 3,
			x, h - d,
			x, h,
			x - d, h - d);
		painter->setPen(Qt::red);
		painter->setBrush(Qt::red);
		painter->drawPolygon(poly);
	}
}

void TapeTimeView::checkDragging(const QPoint& p)
{
	int h = height(); // - 4;
	int d = (h >> 1);
	QRect toleranceRect(0, h - d, d << 1, d);

	// Check selection start
	toleranceRect.moveLeft(mEditor->getSelectionStartX() - d);
	if (toleranceRect.contains(p))
	{
		mDragState = DragSelectionStart;
		setCursor(QCursor(Qt::SizeHorCursor));
		return;
	}
	// Check selection end
	toleranceRect.moveLeft(mEditor->getSelectionEndX() - d);
	if (toleranceRect.contains(p))
	{
		mDragState = DragSelectionEnd;
		setCursor(QCursor(Qt::SizeHorCursor));
		return;
	}
	unsetCursor();
}

///////////////////////////////////////////////////////////////////////////////

}
