/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PackageFilterModel.h
 *    The filter model for the package list widget.
 *
 * @author Ronny Stricker, Christian Martin
 * @date   2011-08-16
 */

#ifndef _MIRA_PACKAGEFILTERMODEL_H_
#define _MIRA_PACKAGEFILTERMODEL_H_

#include <QSortFilterProxyModel>
#include <QRegExp>

#include <core/Database.h>
#include <core/Package.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief Filter for the package tree model.
 * Provide filtering of tags and expressions (applied to name and/or author
 * and/or description).
 * Furthermore, the class provides an internal cache for speed up.
 */
class PackageFilterModel : public QSortFilterProxyModel
{
	Q_OBJECT

public:
	/**
	 * @brief Flags to determine to active filter method.
	 */
	enum FilterType {
		FilterNone        = 0x0000,
		FilterName        = 0x0001,
		FilterAuthor      = 0x0002,
		FilterDescription = 0x0004,
		FilterTags        = 0x0008
	};

	PackageFilterModel(QObject *parent = 0) :
		QSortFilterProxyModel(parent)
	{
		mActiveFilter = FilterNone;
		mFilterExp = QRegExp("");
		mFilterCache = new std::map<qint64,bool>();
		mActiveTags = new std::vector<std::string>();
	}

	virtual ~PackageFilterModel()
	{
		delete mFilterCache;
		delete mActiveTags;
	}

	/**
	 * @brief Set the active filter type.
	 * FilterExpression and/or tag filter is applied according to the active
	 * filter type.
	 */
	void setActiveFilter(FilterType filter) {
		mActiveFilter = filter;
		mFilterCache->clear();
		mActiveTags->clear();
		invalidateFilter();
	}

	/**
	 * @brief Set the expression used for filtering of names and/or authors
	 * and/or descriptions.
	 * @See setActiveFilter()
	 */
	void setFilterExpression(QRegExp const& regExp) {
		mFilterExp = regExp;
		mFilterCache->clear();
		mActiveTags->clear();
		invalidateFilter();
	}

	/**
	 * @brief Set the tags used for filtering.
	 * @See setActiveFilter()
	 */
	void setTagFilter( std::vector<std::string> const& filter )
	{
		mTagFilter = filter;
		mFilterCache->clear();
		mActiveTags->clear();
		invalidateFilter();
	}

	/**
	 * @brief obtain the active tags.
	 * The function returns all the tags occurring in the packages which are NOT
	 * filtered by the current filter settings.
	 */
	std::vector<std::string> const& getActiveTags()
	{
		return *mActiveTags;
	}

	MIRA_ENUM_TO_FLAGS_INCLASS( FilterType )

protected:

	/**
	 * @brief decide if the item should be filtered or not.
	 */
	virtual bool filterAcceptsRow(int sourceRow,
	                              const QModelIndex &sourceParent) const
	{
		QModelIndex index = sourceModel()->index(sourceRow, 0, sourceParent);

		// ask the internal cache first of all for speed up
		if ( mFilterCache->find( index.internalId() ) != mFilterCache->end() ) {
			return (*mFilterCache)[ index.internalId() ];
		}

		PackageGroup* package = static_cast<PackageGroup*>(sourceModel()->index(
				sourceRow, 0, sourceParent).internalPointer() );
		Package* source = dynamic_cast<Package*>( package );

		// direct filtering is only applied to "real" packages
		// group are filtered according to the decisions taken for their children
		if ( source ) {

			/*
			// Filter "latest version only" only for real groups
			if (package->mName == package->mParentPackage->mName) {
				foreach( PackageGroup* pg, package->mParentPackage->mSubPackages ) {
					Package* p = dynamic_cast<Package*>(pg);
					if (p && (p->mVersion > source->mVersion)) {
						(*mFilterCache)[ index.internalId() ] = false;
						return false;
					}
				}
			}
			*/

			// Filter tags first of all
			foreach( std::string const& tag, mTagFilter ) {
				if ( find( source->mTags.begin(), source->mTags.end(), tag) == source->mTags.end() ) {
					(*mFilterCache)[ index.internalId() ] = false;
					return false;
				}
			}

			bool match = ( (mActiveFilter & FilterName)
					&& mFilterExp.indexIn( QString::fromStdString( package->mName ) ) != -1 )
				|| ( source && (mActiveFilter & FilterAuthor)
					&& mFilterExp.indexIn( QString::fromStdString( source->mAuthor ) ) != -1 )
				|| ( (mActiveFilter & FilterDescription)
					&& mFilterExp.indexIn( QString::fromStdString( package->mDescription ) ) != -1 );
			(*mFilterCache)[ index.internalId() ] = match;
			if ( match ) {
				foreach( std::string const& tag, source->mTags ) {
					if ( std::find( mActiveTags->begin(), mActiveTags->end(), tag ) == mActiveTags->end() ) {
						mActiveTags->push_back( tag );
					}
				}
			}
			(*mFilterCache)[ index.internalId() ] = match;
			return match;
		}

		// look at the children to decide if the group should be filtered or not
		for( int i=0; i< int( package->mSubPackages.size() ); ++i ) {
			if ( filterAcceptsRow( i, index ) ) {
				(*mFilterCache)[ index.internalId() ] = true;
				return true;
			}
		}

		(*mFilterCache)[ index.internalId() ] = false;
		return false;
	}

	virtual bool lessThan(const QModelIndex& left, const QModelIndex& right) const
	{
		if ((left.column() == 4) && (right.column() == 4)) {
			// Handle comparison of two versions
			QVariant dataL = sourceModel()->data(left);
			Package::Version versionL;
			versionL.fromString(dataL.toString().toStdString());

			QVariant dataR = sourceModel()->data(right);
			Package::Version versionR;
			versionR.fromString(dataR.toString().toStdString());

			return versionL < versionR;
		}
		return QSortFilterProxyModel::lessThan(left, right);
	}

private:
	FilterType mActiveFilter;                ///< active filter type
	QRegExp mFilterExp;                      ///< current filter expression
	std::vector<std::string> mTagFilter;     ///< current tag filter
	std::map<qint64,bool>* mFilterCache;     ///< the filter cache maps item index to filter decision

	std::vector< std::string >* mActiveTags; ///< list of tags occurring in the packages which are not filtered
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
