/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BatteryState.h
 *    A class for battery state data.
 *
 * @author Tim Langner
 * @date   2010/08/20
 */

#ifndef _MIRA_BATTERYSTATE_H_
#define _MIRA_BATTERYSTATE_H_

#include <utils/Stamped.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

/**
 * A class for holding the current battery state.
 *
 * The battery state is described by the current voltage, the battery current,
 * the charge state in percent, an estimated life time in minutes and a flag,
 * which indicates if the battery is currently charged or not.
 *
 * The battery life percent and battery life time are optional values.
 */
class BatteryState
{
public:
	/** @name Constructors and reflect */
	//@{

	/// The constructor.
	BatteryState() :
		voltage(0.0f), current(0.0f), lifePercent(255), lifeTime(-1),
		charging(false), powerSupplyPresent(false)
	{}

	/**
	 * Construct a new battery state object with specified parameters
	 * @param[in] iVoltage  The current voltage in [V].
	 * @param[in] iCurrent  The current current in [A].
	 * @param[in] iPercent  The current battery capacity in [%] (255 if unknown).
	 * @param[in] iTime     The estimated life time in minutes (-1 if unknown).
	 * @param[in] iCharging Charging flag.
	 * @param[in] iPowerSupplyPresent External power supply connected flag.
	 */
	BatteryState(float iVoltage, float iCurrent,
	             uint8 iPercent, int16 iTime, bool iCharging,
	             bool iPowerSupplyPresent) :
		voltage(iVoltage), current(iCurrent),
		lifePercent(iPercent), lifeTime(iTime),
		charging(iCharging),
		powerSupplyPresent(iPowerSupplyPresent)
	{}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		int v = r.version(2);

		r.member("Voltage", voltage, "The voltage in [V]");
		r.member("Current", current, "The current in [A]");
		r.member("LifePercent", lifePercent,
		         "The battery life percent [0..100] or 255 if unknown");
		r.member("LifeTime", lifeTime,
		         "The battery life time in minutes or -1 if unknown");
		r.member("Charging", charging, "Is the battery being charged");

		// new in version 2
		if (v == 2) {
			r.member("CellVoltage", cellVoltage, "The voltage of the single cells in [V]");
			r.member("PowerSupplyPresent", powerSupplyPresent,
			         "Is the robot powered by an external power supply");
		} else
			powerSupplyPresent = charging;
	}

	//@}

public:
	/// The battery voltage in [V]
	float voltage;

	/// The battery current in [A]
	float current;

	/// The battery life percent [0..100] or 255 if unknown
	uint8 lifePercent;

	/// The battery life time in minutes or -1 if unknown
	int16 lifeTime;

	/// Is the battery currently being charged
	bool charging;

	/// Is the robot powered by an external power supply
	bool powerSupplyPresent;

	/// The voltage of the single battery cells in [V]
	std::vector<float> cellVoltage;
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif
