/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DriveView.h
 *    A view plugin for steering a robot using arrow keys or button clicks
 *
 * @author Tim Langner
 * @date   2011/04/26
 */

#ifndef _MIRA_DRIVEVIEW_H_
#define _MIRA_DRIVEVIEW_H_

#include <QPushButton>
#include <QTimer>
#include <QKeyEvent>

#include <serialization/SetterNotify.h>
#include <rcp/ViewPart.h>
#include <views/PropertyViewPage.h>
#include <fw/Framework.h>
#include <fw/ServiceProperty.h>
#include <visualization/ChannelProperty.h>

#include <ui_DriveView.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

/**
 * A view plugin for steering a robot using arrow keys or button clicks.
 * Using the keyboard the robot can be steered with:
 * - Up: Increase forward speed / Decrease backward speed
 * - Down: Increase backward speed / Decrease forward speed
 * - Left: Increase counter clock wise rotation speed /
 *         Decrease clock wise rotation speed
 * - Right: Increase clock wise rotation speed /
 *          Decrease counter clock wise rotation speed
 * - Space: Full stop
 *
 * The keyboard control only works, if the drive view window has the current
 * keyboard input focus!
 *
 * If the robot has a current rotation velocity of non-zero and Up or Down is
 * pressed, the translation is not changed but the rotation velocity is set to
 * zero. So the robot stops turning and continues to move straight forward or
 * backward.
 *
 * The drive view uses RPCs of a service object, which can be configured by the
 * member @ref driveService. To be able to drive the robot, this service must
 * provide at least the method @ref IMotorController::setVelocity(const Velocity2&).
 * Furthermore, the service should provide the methods @ref IMotorController::resetMotorStop()
 * and @ref IMotorController::resetOdometry() to be able to reset the "stop"
 * flag and to reset the odometry to zero (see class @ref robot::IDrive for
 * more details).
 *
 * If no service is explicitly given by @ref driveService, the DriveView tries to
 * find a service called "IDrive".
 */
class DriveView : public ViewPart
{
	MIRA_META_OBJECT(mira::robot::DriveView,
	                 ("Category",    "Robot")
	                 ("Name",        "DriveView")
	                 ("Description", "View for driving a robot around"))

public:
	/// The constructor
	DriveView();

	/// The destructor
	virtual ~DriveView();

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("VelocityIncrement", velocityIncrement,
		           "Increment for velocity in [m/s]", 0.1f);
		r.property("WatchdogTimeout", watchdogTimeout,
		           "Watchdog for stopping when no commands are signaled for "
		           "a given time", Duration::seconds(5));
		r.property("DriveService", driveService,
		           setterNotify(driveService, &DriveView::driveServiceChanged, this),
		           "Service used for driving", ServiceProperty("IDrive"));
		ChannelProperty<bool>::channelProperty(r, "MotorStopChannel", motorstopChannel,
		                                       "The channel used to check motorstop status", ui);
	}

	/// Implementation of mira::IAdaptable
	virtual Object* getAdapter(const Class& adapter);

protected:
	/// Implementation of mira::ViewPart
	virtual QWidget* createPartControl();

	void driveServiceChanged();

protected:
	/// The velocity increment, default: 0.1f [m/s]
	float velocityIncrement;

	/// Watchdog interval for stopping when no commands are signaled. default: 5sec.
	Duration watchdogTimeout;

	/// The service name used for sending driving commands.
	ServiceProperty driveService;

	/// The channel used for listening to the motorstop status 
	ChannelProperty<bool> motorstopChannel;

private:
	PropertyViewPage* mControl;
	class UI;
	UI* ui;
};

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL

class DriveView::UI : public QWidget, public IAuthorityProvider, protected Ui::DriveViewWidget
{
	Q_OBJECT
public:
	UI(DriveView* parent);

public slots:
	void onForward();
	void onBackward();
	void onLeft();
	void onRight();
	void onStop();
	void onResetMotorStop();
	void onSuspendBumper();
	void onResetOdometry();
	void onEnableMotors();
	void onDisableMotors();
	void tick();
	void updateService();
	void onGrabKeyboard(bool activate);
	void onMute(bool mute);

public:
	virtual Authority& getAuthority() { return mAuthority; }
	void onMotorstop(ChannelRead<bool> status);

public:
	float mTransSpeed;
	float mRotSpeed;

	QTimer* mCommandTimer;
	QTimer* mServiceTimer;
	Authority mAuthority;
	std::string mService;
	Time mLastCommand;
	bool mMute;
	bool mSuspendBumperSupported;

protected:
	virtual void keyPressEvent(QKeyEvent *pEvent);

	void enableDriveButtons(bool enable);
	void enableMotorButtons(bool enable);
	void enableBumperButton(bool enable);

	void muteAllINavigation(bool mute);

private:
	QTabBar* getTabBar(QTabWidget* widget);

private:
	DriveView* mParent;
};

///@endcond

///////////////////////////////////////////////////////////////////////////////

}}

#endif
