/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CollisionTest.C
 *    Test cases for collision
 *
 * @author Tim Langner
 * @date   2012/04/03
 */

#include <boost/test/unit_test.hpp>

#include <model/CollisionTest.h>
#include <model/RigidModel.h>

#include <opencv2/imgproc/imgproc.hpp>

using namespace mira;

const char* argv[] = {"RigidModelCollisionTest", "-d2", "--no-colors"};
Framework fw(3,(char**)argv);

BOOST_AUTO_TEST_CASE(TestCollision)
{
	// generate a model with a single link and a single joint
	// the model is 6mx1m in size and shifted 1.5m to the side
	// so it rotates about a center that is 1.5m to the models right side
	model::RigidModel m;
	model::LinkPtr link(new model::Link());
	link->name = "Link1";
	link->collisions.push_back(model::CollisionPtr(new model::Collision()));
	link->collisions.front()->origin.y() = 1.5f;
	model::BoxPtr box(new model::Box());
	box->size = Size3f(6, 1, 1);
	link->collisions.front()->geometry = box;
	m.links.push_back(link);

	model::JointPtr joint(new model::Joint());
	joint->parent = "Root";
	joint->child = "Link1";
	m.joints["root"] = joint;

	m.resolveAndPublishLinks("/");

	// create and initialize a collision test object
	float cellSize = 0.01f;
	model::CollisionTest ct(cellSize);
	ct.initialize(m.getFootprint("/Root"));

	// create a 10mx10m collision map and draw a 5mx5m obstacle in the upper left corner 
	Point2i center((int)(5.0f / cellSize), (int)(5.0f / cellSize));
	Img8U1 map((int)(10.0f / cellSize), (int)(10.0f / cellSize));
	map = 0;
	cv::rectangle((cv::Mat&)map, Point2i(0, 0), center, 255, CV_FILLED);
	// calculate distance map for distance to obstacle computations
	Img32F1 distanceMap;
	ct.distanceTransform(map, distanceMap);

	// there should be no collision between the models footprint 
	// and the map for 270° <= d <= 360° 
	for (float d = 270.0f; d<=360.0f; d+=10.0f)
	{
		BOOST_CHECK(!ct.testCollision(map, center, deg2rad(d)));
		float distance = ct.distanceToObstacle(distanceMap, center, deg2rad(d)) * cellSize;
		std::cout << d << ":" << distance << std::endl;
		// TODO check why this is not always in the range of 1.0f +- 2*cellSize
		// BOOST_CHECK_SMALL(distance - 1.0f, cellSize*2.0f);
	}

	// there should be a collision between the models footprint 
	// and the map for 90° <= d <= 180° 
	for (float d = 90.0f; d<=180.0f; d+=10.0f)
	{
		BOOST_CHECK(ct.testCollision(map, center, deg2rad(d)));
		// distance to obstacle must be 0.0f since we are colliding
		float distance = ct.distanceToObstacle(distanceMap, center, deg2rad(d)) * cellSize;
		BOOST_CHECK(distance == 0.0f);
	}
}

