/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RigidModelVisualization.C
 *    Visualization for rigid models.
 *
 * @author Tim Langner
 * @date   2011/06/30
 */

#include <OGRE/OgreMaterialManager.h>
#include <OGRE/OgreSceneNode.h>
#include <OGRE/OgreSceneManager.h>

#include <transform/Pose.h>
#include <serialization/Serialization.h>
#include <visualization/Visualization3D.h>
#include <visualization/3d/MeshObject.h>
#include <visualization/3d/PolygonObject.h>
#include <widgets/OgreUtils.h>

#include <model/RigidModel.h>
#include <model/RigidModelObserver.h>

#include <fw/TransformProperty.h>
#include <fw/ServiceProperty.h>

namespace mira { namespace  model {

///////////////////////////////////////////////////////////////////////////////

Polygon2f circleToPolygon(float radius)
{
	Polygon2f polygon;
	const int sectors = 24;
	float dalpha = two_pi<float>() / sectors;
	float alpha = 0.0f;
	for(int i=0; i<=sectors; ++i, alpha+=dalpha)
	{
		Point2f d(sin(alpha)*radius, cos(alpha)*radius);
		polygon.push_back(d);
	}
	return polygon;
}

class RigidModelVisualization3D : public Visualization3D
{
	MIRA_META_OBJECT(RigidModelVisualization3D,
				("Name", "Rigid Model")
				("Description", "3D Visualization of rigid models")
				("Category", "Models"))

public:
	typedef PolygonObject<float> PolygonObjectType;

public:

	struct ObjectRepresentation
	{
		ObjectRepresentation() : node(NULL) {}
		std::string frame;
		Ogre::SceneNode* node;
		std::vector<boost::shared_ptr<VisualizationObject>> objects;
	};

	RigidModelVisualization3D() :
		mModelProvider("IRigidModelProvider"),
		mModelConsumer("IRigidModelConsumer"),
		mNode(NULL)
	{
		mShowCollision = false;
		mShowVisual = true;
		mShowBoundingBoxes = false;
		mShowFootprint = false;
		mShowFootprintCircles = false;
	}

	~RigidModelVisualization3D()
	{
		if (getSite() != NULL && getSite()->getSceneManager() != NULL)
			clearObjects();
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		Visualization3D::reflect(r);
		r.property("ModelProvider", mModelProvider,
		           "The service that provides us with a rigid model");
		r.property("FollowModelConsumer", mModelConsumer,
		           "Optionally, we can follow a ModelConsumer (a service that has a 'ModelProvider' property)\n"
		           "Setting this will mirror the selected service's 'ModelProvider' property.",
		           serialization::IgnoreMissing());
		r.property("Show Collision", mShowCollision,
		           setter<bool>(&RigidModelVisualization3D::setShowCollision, this),
		           "Enable/disable visualization of collision links", false);
		r.property("Show Visual", mShowVisual,
		           setter<bool>(&RigidModelVisualization3D::setShowVisual, this),
		           "Enable/disable visualization of visual links", true);
		r.property("Show Footprint", mShowFootprint,
		           setter<bool>(&RigidModelVisualization3D::setShowFootprint, this),
		           "Enable/disable visualization of footprint", false);
		r.property("Show Footprint Circles", mShowFootprintCircles,
		           setter<bool>(&RigidModelVisualization3D::setShowFootprintCircles, this),
		           "Enable/disable visualization of the outer and inner circles "
		           "(encircle and incircle)", false);
		r.property("Show Bounding Boxes", mShowBoundingBoxes,
		           setter<bool>(&RigidModelVisualization3D::setShowBBox, this),
		           "Enable/disable visualization of bounding boxes", false);
		r.property("Frame for Bounding Boxes", mBoundingBoxFrame,
		           setter<TransformProperty>(&RigidModelVisualization3D::setBBoxFrame, this),
		           "The frame, the bounding boxes are shown in", TransformProperty());
		r.property("CollisionFilter", mCollisionFilter,
		           setter<std::string>(&RigidModelVisualization3D::setCollisionFilter, this),
		           "The filter for showing bounding box / footprint collision geometries", "");


	}

	virtual void setupScene(IVisualization3DSite* site)
	{
		Ogre::SceneManager* sceneManager = getSite()->getSceneManager();
		mNode = sceneManager->getRootSceneNode()->createChildSceneNode();
	}

	virtual Ogre::SceneNode* getNode()
	{
		return mNode;
	}

	virtual void setEnabled(bool enabled)
	{
		Visualization3D::setEnabled(enabled);
		foreach(auto item, mVisualObjects)
			foreach(auto o, item.objects)
				o->setVisible(enabled & mShowVisual);
		foreach(auto item, mCollisionObjects)
			foreach(auto o, item.objects)
				o->setVisible(enabled & mShowCollision);
		if(mFootprintObject.objects.size() > 0)
			mFootprintObject.objects[0]->setVisible(enabled & mShowFootprint);
		if(mFootprintObject.objects.size() > 1)
			mFootprintObject.objects[1]->setVisible(enabled & mShowFootprintCircles);
		if(mFootprintObject.objects.size() > 2)
			mFootprintObject.objects[2]->setVisible(enabled & mShowFootprintCircles);

		if(mBoundingBoxObject.objects.size() > 0)
			mBoundingBoxObject.objects[0]->setVisible(enabled & mShowBoundingBoxes);
	}

	void setShowCollision(bool show)
	{
		mShowCollision = show;
		foreach(auto item, mCollisionObjects)
			foreach(auto o, item.objects)
				o->setVisible(isEnabled() & mShowCollision);
	}

	void setShowVisual(bool show)
	{
		mShowVisual = show;
		foreach(auto item, mVisualObjects)
			foreach(auto o, item.objects)
				o->setVisible(isEnabled() & mShowVisual);
	}

	void setShowBBox(bool show)
	{
		mShowBoundingBoxes = show;
		if (mBoundingBoxObject.objects.size() > 0)
			mBoundingBoxObject.objects[0]->setVisible(isEnabled() & mShowBoundingBoxes);
	}

	void setBBoxFrame(const TransformProperty& frame)
	{
		mBoundingBoxFrame = frame;
		updateBoundingBoxes();
	}

	void setCollisionFilter(const std::string& filter)
	{
		mCollisionFilter = filter;
		mDirtyModel = true;
	}

	void setShowFootprint(bool show)
	{
		mShowFootprint = show;
		if (mFootprintObject.objects.size() > 0)
			mFootprintObject.objects[0]->setVisible(isEnabled() & mShowFootprint);
	}

	void setShowFootprintCircles(bool show)
	{
		mShowFootprintCircles = show;
		if(mFootprintObject.objects.size() > 1)
			mFootprintObject.objects[1]->setVisible(isEnabled() & mShowFootprintCircles);
		if(mFootprintObject.objects.size() > 2)
			mFootprintObject.objects[2]->setVisible(isEnabled() & mShowFootprintCircles);
	}

protected:

	virtual void update(Duration dt);
	void updateProvider();

	void clearObjects();
	void collisionChanged();
	void updateFootprints();
	void updateBoundingBoxes();

protected:

	ServiceProperty mModelProvider;
	ServiceProperty mModelConsumer;
	std::string mUsedModelProvider;
	RigidModelPtr mModel;
	RigidModelObserverPtr mModelObserver;

	Ogre::SceneNode* mNode;
	std::list<ObjectRepresentation> mVisualObjects;
	std::list<ObjectRepresentation> mCollisionObjects;
	ObjectRepresentation mFootprintObject;
	ObjectRepresentation mBoundingBoxObject;

	bool mShowCollision;
	bool mShowVisual;
	bool mShowBoundingBoxes;
	TransformProperty mBoundingBoxFrame;
	bool mShowFootprint;
	std::string mCollisionFilter;
	bool mShowFootprintCircles;
	bool mDirtyModel;
};

///////////////////////////////////////////////////////////////////////////////

void RigidModelVisualization3D::update(Duration dt)
{
	try
	{
		updateProvider();
	}
	catch(Exception& ex)
	{
		error("Model", ex.message(), "Error in model");
	}
	catch(Ogre::Exception& ex)
	{
		error("Model", ex.what(), "Error in model");
	}
	foreach(auto item, mVisualObjects)
	{
		Pose3 d = getAuthority().getTransform<RigidTransform3f>(item.frame,
		                                                        getSite()->getCameraFrame(),
		                                                        Time::now());
		OgreUtils::setTransform(item.node, d);
	}
	foreach(auto item, mCollisionObjects)
	{
		Pose3 d = getAuthority().getTransform<RigidTransform3f>(item.frame,
		                                                        getSite()->getCameraFrame(),
		                                                        Time::now());
		OgreUtils::setTransform(item.node, d);
	}
	if (mFootprintObject.node)
	{
		if (mDirtyModel)
			updateFootprints();
		Pose3 d = getAuthority().getTransform<RigidTransform3f>(mFootprintObject.frame,
		                                                        getSite()->getCameraFrame(),
		                                                        Time::now());
		OgreUtils::setTransform(mFootprintObject.node, d);
	}
	if (mBoundingBoxObject.node && mShowBoundingBoxes)
	{
		updateBoundingBoxes();
		Pose3 d = getAuthority().getTransform<RigidTransform3f>(mBoundingBoxObject.frame,
		                                                        getSite()->getCameraFrame(),
		                                                        Time::now());
		OgreUtils::setTransform(mBoundingBoxObject.node, d);
	}
	mDirtyModel = false;
}

Ogre::MaterialPtr getMaterial(MaterialPtr material)
{
	Ogre::MaterialPtr m;
	if (material)
	{
		static int count = 0;
		std::string materialName = MakeString() << "RigidModelMaterial " << count++;
		m = Ogre::MaterialManager::getSingleton().create(materialName, "MIRAMaterials");
		m->setReceiveShadows(false);
		m->getTechnique(0)->setLightingEnabled(true);
		if (material->texture.empty())
		{
			const Color::RGBA& c = material->color;
			m->getTechnique(0)->setAmbient(c.r * 0.5, c.g * 0.5,
			                               c.b * 0.5);
			m->getTechnique(0)->setDiffuse(c.r, c.g, c.b, c.a);
			if(c.a>0.0f)
				m->setSceneBlending(Ogre::SBT_TRANSPARENT_ALPHA);
			else
				m->setSceneBlending(Ogre::SBT_REPLACE);
		}
		else
		{
			Path filename = resolvePath(material->texture);
			if (!Ogre::TextureManager::getSingleton().resourceExists(filename.string()))
			{
				if (!boost::filesystem::exists(filename))
					MIRA_THROW(XFileNotFound, "Texture '"
					           << filename << "' could not be found.");
				std::ifstream is(filename.string().c_str());
				Ogre::DataStreamPtr filedatastream(new Ogre::FileStreamDataStream(&is, false));
				Ogre::Image image;
				std::string extension = filename.extension().string();
				extension.erase(0, 1);
				try
				{
					image.load(filedatastream, extension);
					Ogre::TextureManager::getSingleton().loadImage(
						filename.string(),
						"MIRATextures",
						image);
				} catch (Ogre::Exception& e)
				{
					MIRA_THROW(XIO, "Error loading texture '"
					           << filename << "'." << e.what());
				}
			}
			Ogre::Pass* pass = m->getTechnique(0)->getPass(0);
			Ogre::TextureUnitState* tex = pass->createTextureUnitState();
			tex->setTextureName(filename.string());
		}
	}
	return m;
}

boost::shared_ptr<VisualizationObject> createRepresentation(const Pose3& origin,
                                                            GeometryPtr geometry,
                                                            MaterialPtr material,
                                                            Ogre::SceneNode* node,
                                                            Ogre::SceneManager* sceneManager,
                                                            bool visible)
{
	Ogre::Vector3 scale(Ogre::Vector3::UNIT_SCALE);
	Ogre::Vector3 offsetPosition(origin.x(), origin.y(), origin.z());
	Ogre::Quaternion offsetOrientation(origin.r.w(),
	                                   origin.r.x(),
	                                   origin.r.y(),
	                                   origin.r.z());

	boost::shared_ptr<VisualizationObject> object;
	std::string identifier = geometry->getClass().getIdentifier();
	if (identifier == "mira::model::Box")
	{
		boost::shared_ptr<model::Box> p = boost::dynamic_pointer_cast<model::Box>(geometry);
		Ogre::MaterialPtr m = getMaterial(material);
		object.reset(new MeshObject("Box.mesh", m, sceneManager, node));
		scale = Ogre::Vector3(p->size.width(), p->size.height(), p->size.depth());
		offsetPosition.y -= p->size.height()/2.0f;
	}
	else if (identifier == "mira::model::Cone")
	{
		boost::shared_ptr<model::Cone> p = boost::dynamic_pointer_cast<model::Cone>(geometry);
		Ogre::MaterialPtr m = getMaterial(material);
		object.reset(new MeshObject("Cone.mesh", m, sceneManager, node));
		scale = Ogre::Vector3(p->radius, p->length, p->radius);
		Ogre::Quaternion rotX;
		rotX.FromAngleAxis(Ogre::Degree(90), Ogre::Vector3::UNIT_X);
		offsetOrientation = offsetOrientation * rotX;
	}
	else if (identifier == "mira::model::Cylinder")
	{
		boost::shared_ptr<model::Cylinder> p = boost::dynamic_pointer_cast<model::Cylinder>(geometry);
		Ogre::MaterialPtr m = getMaterial(material);
		object.reset(new MeshObject("CylinderCentered.mesh", m, sceneManager, node));
		scale = Ogre::Vector3(p->radius, p->length, p->radius);
		Ogre::Quaternion rotX;
		rotX.FromAngleAxis(Ogre::Degree(90), Ogre::Vector3::UNIT_X);
		offsetOrientation = offsetOrientation * rotX;
	}
	else if (identifier == "mira::model::Sphere")
	{
		boost::shared_ptr<model::Sphere> p = boost::dynamic_pointer_cast<model::Sphere>(geometry);
		Ogre::MaterialPtr m = getMaterial(material);
		object.reset(new MeshObject("Sphere.mesh", m, sceneManager, node));
		scale = Ogre::Vector3(p->radius, p->radius, p->radius);
	}
	else if (identifier == "mira::model::Mesh")
	{
		boost::shared_ptr<model::Mesh> mesh = boost::dynamic_pointer_cast<model::Mesh>(geometry);
		Ogre::MaterialPtr m = getMaterial(material);
		object.reset(new MeshObject(mesh->filename, m, sceneManager, node));
		scale = Ogre::Vector3(mesh->scale, mesh->scale, mesh->scale);
	}
	else if (identifier == "mira::model::Polygon")
	{
		boost::shared_ptr<model::Polygon> shape = boost::dynamic_pointer_cast<model::Polygon>(geometry);
		RigidModelVisualization3D::PolygonObjectType* poly = new RigidModelVisualization3D::PolygonObjectType(sceneManager, node);
		object.reset(poly);
		poly->setPolygon(shape->polygon);
		if (material)
			object->setColor(OgreUtils::toOgreColor(material->color));
	}
	else
		MIRA_THROW(XInvalidConfig, "'" << identifier << "' is not supported by RigidModelVisualization")
	object->setScale(scale);
	object->setPosition(offsetPosition);
	object->setOrientation(offsetOrientation);
	object->setVisible(visible);
	return object;
}

void RigidModelVisualization3D::updateProvider()
{
	std::string c = mModelConsumer.getService();
	if (!c.empty()) {
		std::string s
			= getAuthority().callService<std::string>(c+"#builtin",
		                                              "getProperty", std::string("ModelProvider")).get();
		json::Value v;
		json::read(s, v);
		JSONDeserializer jds(v);
		jds.deserialize(mModelProvider);
	}

	std::string provider;
	if (!mModelProvider.empty())
	{
		// same provider as already used and it still exists
		if (mUsedModelProvider == mModelProvider.getService() && getAuthority().existsService(mUsedModelProvider))
			return;
		clearObjects();
		mUsedModelProvider = "";
		provider = mModelProvider;
	}
	else
	{
		// we already have a model provider and it still exists -> do nothing
		if (!mUsedModelProvider.empty() && getAuthority().existsService(mUsedModelProvider))
			return;
		clearObjects();
		mUsedModelProvider = "";
		auto l = getAuthority().queryServicesForInterface("IRigidModelProvider");
		// no provider found
		if (l.empty())
			return;
		provider = *l.begin();
	}
	ok("Objects");
	auto p = getAuthority().callService<RigidModelPtr>(provider, "getRigidModel");
	if (!p.timedWait(Duration::seconds(1000)))
		MIRA_THROW(XRuntime, "Model could not be retrieved from service '"
		           << provider << "'");
	mUsedModelProvider = provider;

	mModel = p.get();
	assert(mModel);
	Ogre::SceneManager* sceneManager = getSite()->getSceneManager();

	foreach(auto link, mModel->links)
	{
		if (link->visuals.size() > 0)
		{
			ObjectRepresentation r;
			r.node = mNode->createChildSceneNode();
			foreach(VisualPtr v, link->visuals)
			{
				if (!v || !v->geometry)
					continue;
				try
				{
					auto object = createRepresentation(v->origin, v->geometry, v->material,
					                                   r.node, sceneManager,
					                                   isEnabled() & mShowVisual);
					r.objects.push_back(object);
				}
				catch(XInvalidConfig& ex)
				{
					warning("Objects", ex.message());
				}
			}
			r.frame = link->name;
			mVisualObjects.push_back(r);
		}
		if (link->collisions.size() > 0)
		{
			ObjectRepresentation r;
			r.node = mNode->createChildSceneNode();
			foreach(CollisionPtr c, link->collisions)
			{
				if (!c || !c->geometry)
					continue;
				try
				{
					auto object = createRepresentation(c->origin, c->geometry, MaterialPtr(),
					                                   r.node, sceneManager,
					                                   isEnabled() & mShowCollision);
					r.objects.push_back(object);
				}
				catch(XInvalidConfig& ex)
				{
					warning("Objects", ex.message());
				}
			}
			r.frame = link->name;
			mCollisionObjects.push_back(r);
		}
	}
	// Create visualization for footprint
	mFootprintObject.frame = mModel->getRootLink();
	mFootprintObject.node = mNode->createChildSceneNode();

	PolygonObjectType* poly = new PolygonObjectType(sceneManager, mFootprintObject.node);
	mFootprintObject.objects.push_back(boost::shared_ptr<VisualizationObject>(poly));
	poly->setColor(Ogre::ColourValue(1,1,0));
	mFootprintObject.objects[0]->setVisible(isEnabled() & mShowFootprint);

	poly = new PolygonObjectType(sceneManager, mFootprintObject.node);
	mFootprintObject.objects.push_back(boost::shared_ptr<VisualizationObject>(poly));
	poly->setColor(Ogre::ColourValue(0,1,0));
	mFootprintObject.objects[1]->setVisible(isEnabled() & mShowFootprintCircles);

	poly = new PolygonObjectType(sceneManager, mFootprintObject.node);
	mFootprintObject.objects.push_back(boost::shared_ptr<VisualizationObject>(poly));
	poly->setColor(Ogre::ColourValue(0,1,0));
	mFootprintObject.objects[2]->setVisible(isEnabled() & mShowFootprintCircles);

	updateFootprints();

	// Create visualization for bounding boxes
	mBoundingBoxObject.frame = mModel->getRootLink();
	mBoundingBoxObject.node = mNode->createChildSceneNode();

	poly = new PolygonObjectType(sceneManager, mBoundingBoxObject.node);
	mBoundingBoxObject.objects.push_back(boost::shared_ptr<VisualizationObject>(poly));
	poly->setColor(Ogre::ColourValue(1,0,0));
	mBoundingBoxObject.objects[0]->setVisible(isEnabled() & mShowBoundingBoxes);

	updateBoundingBoxes();

	mModelObserver.reset(new RigidModelObserver(*this, mModel));
	mModelObserver->registerCollisionChangedCallback(boost::bind(&RigidModelVisualization3D::collisionChanged, this));

	ok("Model");
}

void RigidModelVisualization3D::clearObjects()
{
	mModelObserver.reset();
	Ogre::SceneManager* sceneManager = getSite()->getSceneManager();
	foreach(auto item, mVisualObjects)
		sceneManager->destroySceneNode(item.node);
	mVisualObjects.clear();
	foreach(auto item, mCollisionObjects)
		sceneManager->destroySceneNode(item.node);
	mCollisionObjects.clear();
	if (mFootprintObject.node != NULL)
		sceneManager->destroySceneNode(mFootprintObject.node);
	mFootprintObject.node = NULL;
	mFootprintObject.objects.clear();

	if (mBoundingBoxObject.node != NULL)
		sceneManager->destroySceneNode(mBoundingBoxObject.node);
	mBoundingBoxObject.node = NULL;
	mBoundingBoxObject.objects.clear();
}

void RigidModelVisualization3D::collisionChanged()
{
	mDirtyModel = true;
}

void RigidModelVisualization3D::updateFootprints()
{
	Footprint footprint = mModel->getFootprint("", Time::now(), mCollisionFilter);
	if (!footprint.empty())
	{
		PolygonObjectType* poly = dynamic_cast<PolygonObjectType*>(mFootprintObject.objects[0].get());
		poly->setPolygons(footprint);

		poly = dynamic_cast<PolygonObjectType*>(mFootprintObject.objects[1].get());
		poly->setPolygon(circleToPolygon(footprint.getInnerRadius()));

		poly = dynamic_cast<PolygonObjectType*>(mFootprintObject.objects[2].get());
		poly->setPolygon(circleToPolygon(footprint.getOuterRadius()));
	}
}

void addBoundingBoxToPolygons(std::vector<Polygon3f>& oPolys, const Box3f& bbox)
{

	Point3f a = bbox.minCorner;
	Point3f b = bbox.maxCorner;

	Polygon3f p1;
	p1.push_back(Point3f(a.x(),a.y(),a.z()));
	p1.push_back(Point3f(b.x(),a.y(),a.z()));
	p1.push_back(Point3f(b.x(),a.y(),b.z()));
	p1.push_back(Point3f(a.x(),a.y(),b.z()));
	oPolys.push_back(p1);

	Polygon3f p2;
	p2.push_back(Point3f(b.x(),a.y(),a.z()));
	p2.push_back(Point3f(b.x(),b.y(),a.z()));
	p2.push_back(Point3f(b.x(),b.y(),b.z()));
	p2.push_back(Point3f(b.x(),a.y(),b.z()));
	oPolys.push_back(p2);

	Polygon3f p3;
	p3.push_back(Point3f(b.x(),b.y(),a.z()));
	p3.push_back(Point3f(a.x(),b.y(),a.z()));
	p3.push_back(Point3f(a.x(),b.y(),b.z()));
	p3.push_back(Point3f(b.x(),b.y(),b.z()));
	oPolys.push_back(p3);

	Polygon3f p4;
	p4.push_back(Point3f(a.x(),b.y(),a.z()));
	p4.push_back(Point3f(a.x(),a.y(),a.z()));
	p4.push_back(Point3f(a.x(),a.y(),b.z()));
	p4.push_back(Point3f(a.x(),b.y(),b.z()));
	oPolys.push_back(p4);
}

void RigidModelVisualization3D::updateBoundingBoxes()
{
	if(mBoundingBoxObject.objects.empty())
		return;

	std::vector<Polygon3f> polys;

	std::list<Box3f> bboxes = mModel->getCollisionBoundingBoxes(mBoundingBoxFrame.getID(), Time::now(), mCollisionFilter);
	if(mBoundingBoxFrame.getID().empty())
		mBoundingBoxObject.frame = mModel->getRootLink();
	else
		mBoundingBoxObject.frame = mBoundingBoxFrame.getID();

	foreach(const Box3f& bbox, bboxes)
	{
		if (bbox.isValid())
			addBoundingBoxToPolygons(polys, bbox);
	}

	PolygonObjectType* poly = dynamic_cast<PolygonObjectType*>(mBoundingBoxObject.objects[0].get());
	poly->setPolygons(polys);
}
//////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::model::RigidModelVisualization3D, mira::Visualization3D);
