/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RigidModelProvider.h
 *    Template helper class that provides a rigid model via rpc service.
 *
 * @author Tim Langner
 * @date   2011/07/01
 */

#ifndef _MIRA_RIGIDMODELPROVIDER_H_
#define _MIRA_RIGIDMODELPROVIDER_H_

#include <fw/Framework.h>

#include <model/IRigidModelProvider.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

/**
 * Inherit from this class by specifying a ModelType that is inherited
 * from RigidModel to build your own model provider.
 */
template <typename ModelType>
class RigidModelProvider : public virtual IRigidModelProvider
{
public:
	typedef boost::shared_ptr<ModelType> ModelTypePtr;
	
	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, IRigidModelProvider);
		r.member("Model", mModel, "The used model");
		r.method("publishJoint", &RigidModelProvider::publishJoint, this,
		         "Publishes state of a joint",
		         "name", "joint name", "DisplayMotorToDisplayArm",
		         "value", "joint value ([rad] for CONTINUOUS/REVOLUTE type, [m] for PRISMATIC)", deg2rad(15.f),
		         "timestamp", "timestamp", Time::now());
	}

	/**
	 * Return the used rigid model pointer
	 */
	virtual model::RigidModelPtr getRigidModel()
	{
		return mModel;
	}

	/**
	 * Publishes the state of the joint with the given name. See
	 * RigidModel::publishJoint() for details.
	 * @throw XInvalidConfig if no model is loaded
	 */
	void publishJoint(const std::string& name, float value,
	                  Time const& t = Time::now())
	{
		if (!mModel)
			MIRA_THROW(XInvalidConfig, "No rigid model loaded");
		mModel->publishJoint(name, value, t);
	}

	/**
	 * Resolves all transform frames, adds transform links and publishes all 
	 * transformations of the loaded model using the given namespace.
	 * @param ns The namespace of the model
	 */
	void publishTransforms(const std::string& ns)
	{
		if (!mModel)
			MIRA_THROW(XInvalidConfig, "No rigid model loaded");
		mModel->resolveAndPublishLinks(ns);
	}

protected:

	ModelTypePtr mModel;
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif
