/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PathGoalTool.C
 *    Tool plugin for miracenter to send a robot along a path using the
 *    INavigation service interface.
 *
 * @author Erik Einhorn
 * @date   2011/07/02
 */

#include <QMouseEvent>
#include <QToolTip>

#include <OGRE/OgreRenderWindow.h>
#include <OGRE/OgreSceneManager.h>

#include <transform/Pose.h>

#include <visualization/3d/SceneQueryTool.h>
#include <visualization/3d/LineStripObject.h>
#include <visualization/3d/AxesObject.h>

#include <navigation/tasks/PathFollowTask.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////



/**
 * Tool plugin for miracenter to send a robot along a path using the
 * INavigation service interface.
 */
class PathGoalTool : public SceneQueryTool
{
	MIRA_META_OBJECT(PathGoalTool,
		("Name", "Path")
		("Category", "Navigation")
		("Description", "Set navigation goal"))

public:
	enum SelectionMode
	{
		Position,
		Orientation,
	};

	PathGoalTool() :
		mIsValid(false),
		mShowToolTips(true),
		mLine(NULL)
	{
	}

	~PathGoalTool()
	{
		if (mLine)
			delete mLine;
	}

public:
	/** @name Implementation of VisualizationTool */
	//@{
	virtual void onMousePressed(QMouseEvent* e);

	virtual void onMouseMoved(QMouseEvent* e);

	//@}

protected:
	/** @name Implementation of VisualizationTool3D */
	//@{

	virtual void setupScene(IVisualization3DSite* site);

	//@}

	/** @name Implementation of VisualizationTool */
	//@{

	virtual QIcon getIcon() {
		QPixmap pixmap(":/icons/ToolGoal.png");
		return QIcon(pixmap);
	}

	virtual void activate() {
		SceneQueryTool::activate();
		doReset();
	}
	//@}


	void doReset()
	{
		mTask.reset(new PathFollowTask);
		mTask->frame = getSite()->getFixedFrame();
		mLine->clear();
		mDoReset = false;
	}


protected:
	/** @name Implementation of PoseTool */
	//@{

	/// new pose is set as goal pose
	void onNewPose(const Pose2& pose)
	{
		if (mDoReset)
			doReset();
		mTask->points.push_back(pose);
		mLine->add(Ogre::Vector3(pose.x(), pose.y(), 0));
	}

private:

	std::pair<std::string, Ogre::Vector3> getIntersection(float x, float y,
	                                                      const std::string& mask="");

	Ogre::Vector3 transformVector(const Ogre::Vector3& v);

	void updateToolTip(QMouseEvent* e, const QString& text);

	boost::shared_ptr<AxesObject> mAxesObject;
	SelectionMode mSelectionMode;
	std::string mSelectedObject;
	Ogre::Vector3 mLastPos;
	bool mIsValid;
	bool mShowToolTips;
	QPoint mLastToolTipPoint;
	QString mLastToolTip;
	Pose2 mPose;
	boost::shared_ptr<PathFollowTask> mTask;
	bool mDoReset;
	LineStripObject* mLine;

	//@}
};

///////////////////////////////////////////////////////////////////////////////

void PathGoalTool::onMousePressed(QMouseEvent* e)
{
	if (e->button() == Qt::RightButton) {
		mIsValid = false;
		mSelectedObject = "";
		mSelectionMode = Position;
		auto s = queryServicesForInterface("INavigation");
		if (!s.empty()) {
			boost::shared_ptr<Task> task(new Task());
			task->addSubTask(mTask);
			auto result = callService<void>(s.front(), "setTask", task);
			result.timedWait(Duration::seconds(10));
			result.get(); // will cause exception if something went wrong
		}
		mDoReset = true;
		return;
	}

	if (e->button() != Qt::LeftButton)
		return;

	switch (mSelectionMode) {
		case Position:
		{
			auto i = getIntersection(e->posF().x(), e->posF().y());
			if (!i.first.empty())
			{
				mAxesObject->setPosition(i.second);
				mSelectedObject = i.first;
				mLastPos = i.second;
				mPose.x() = i.second.x;
				mPose.y() = i.second.y;
				mSelectionMode = Orientation;

				mIsValid = true;
				mAxesObject->setVisible(true);
			}
			break;
		}
		case Orientation:
		{
			auto i = getIntersection(e->posF().x(), e->posF().y(), mSelectedObject);
			if (!i.first.empty())
			{
				mSelectionMode = Position;
				mSelectedObject = "";
				updateToolTip(e, "");
				onNewPose(mPose);
			}
			break;
		}
		default: break;
	}
}

void PathGoalTool::onMouseMoved(QMouseEvent* e)
{
	if (!mSelectedObject.empty())
	{
		auto i = getIntersection(e->posF().x(), e->posF().y(), mSelectedObject);
		if (i.first.empty())
			return;

		Ogre::Vector3 diff = i.second-mLastPos;
		float r = diff.length();

		switch (mSelectionMode) {
			case Orientation: {
				float phi = std::atan2(diff.y, diff.x);
				mPose.phi() = phi;
				mAxesObject->setOrientation(Eigen::Quaternionf(Eigen::AngleAxisf(phi, Eigen::Vector3f::UnitZ())));
				updateToolTip(e, "Click left to set orientation.");
				break;
			}
			default: break;
		}
	} else {
		switch (mSelectionMode) {
		case Position:
			updateToolTip(e, "Click left to set position.");
			break;
		default:
			break;
		}
	}
}

void PathGoalTool::setupScene(IVisualization3DSite* site)
{
	SceneQueryTool::setupScene(site);
	mAxesObject.reset(new AxesObject(site->getSceneManager(), mNode));
	mAxesObject->showZAxis(false);
	mAxesObject->setVisible(false);
	mSelectedObject = "";
	mSelectionMode = Position;
	mLine = new LineStripObject(site->getSceneManager(), mNode);
}

Ogre::Vector3 PathGoalTool::transformVector(const Ogre::Vector3& v)
{
	Eigen::Vector3f e(v.x,v.y,v.z);
	Eigen::Vector3f r;
	r = mTransform.inverse() * e;
	return Ogre::Vector3(r.x(),r.y(),r.z());
}

std::pair<std::string, Ogre::Vector3> PathGoalTool::getIntersection(float x, float y,
                                                                    const std::string& mask)
{
	return mask.empty() ? SceneQueryTool::getIntersection(x, y, "ImageObject", false, true)
	                    : SceneQueryTool::getIntersection(x, y, mask, true, true);
}

void PathGoalTool::updateToolTip(QMouseEvent* e, const QString& text)
{
	if (text.isEmpty()) {
		QToolTip::showText(QPoint(), "");
		return;
	}
	int tDelta = (e->globalPos()-mLastToolTipPoint).manhattanLength();
	if ((tDelta > 10) || (text != mLastToolTip)) {
		QToolTip::showText(QPoint(), "");
		QToolTip::showText(e->globalPos(), text);
		mLastToolTip = text;
		mLastToolTipPoint = e->globalPos();
	}
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::navigation::PathGoalTool, mira::VisualizationTool3D)
