/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PolygonVisualization.C
 *    Visualization for polygons
 *
 * @author Tim Langner
 * @date   2012/10/08
 */

#include <serialization/Serialization.h>

#include <visualization/Visualization3DBasic.h>
#include <visualization/3d/PolygonObject.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>

#include <widgets/OgreUtils.h>
#include <OGRE/OgreSceneManager.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

/**
 * A polygon visualization.
 */
template <typename T, int Dim>
class PolygonVisualization3DBase : public Visualization3DBasic<boost::geometry::model::ring<Point<T, Dim>>>
{
public:
	typedef boost::geometry::model::ring<Point<T, Dim>> Poly;
	typedef Visualization3DBasic<Poly> Base;

public:
	/** @name Constructor, destructor and reflect */
	//@{

	/// The constructor.
	PolygonVisualization3DBase() :
		mPolygon(NULL),
		mOutlineColor(Ogre::ColourValue::Black),
		mFillColor(0.f, 0.f, 0.f, 0.f)
	{
	}

	/// The destructor.
	virtual ~PolygonVisualization3DBase() {
		delete mPolygon;
	}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		Base::reflect(r);
		r.property("OutlineColor", mOutlineColor,
		           setter<Ogre::ColourValue>(&PolygonVisualization3DBase::setOutlineColor, this),
		           "The color of the polygon outline", Ogre::ColourValue::Black);
		r.property("FillColor", mFillColor,
		           setter<Ogre::ColourValue>(&PolygonVisualization3DBase::setFillColor, this),
		           "The color of the polygon area", Ogre::ColourValue(0.f, 0.f, 0.f, 0.f));
	}

	//@}

public:
	/** @name Setters */
	//@{

	/// Set the color of the polygon outline.
	void setOutlineColor(const Ogre::ColourValue& c)
	{
		mOutlineColor = c;
		if (mPolygon != NULL)
			mPolygon->setOutlineColor(mOutlineColor);
	}

	/// Set the color of the polygon area.
	void setFillColor(const Ogre::ColourValue& c)
	{
		mFillColor = c;
		if (mPolygon != NULL)
			mPolygon->setFillColor(mFillColor);
	}

	//@}

public:
	/** @name Public implementation of Visualization3D and Visualization*/
	//@{

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{
		mPolygon = new PolygonObject<T>(mgr, node);
		mPolygon->setOutlineColor(mOutlineColor);
		mPolygon->setFillColor(mFillColor);
		mPolygon->setVisible(this->isEnabled());
	}

	virtual void setEnabled(bool enabled)
	{
		Visualization3DBasic<Poly>::setEnabled(enabled);
		if (mPolygon != NULL)
			mPolygon->setVisible(enabled);
	}

	void dataChanged(ChannelRead<Poly> polygon)
	{
		mPolygon->setPolygon(polygon->value());
	}

	//@}

private:
	PolygonObject<T>* mPolygon;
	Ogre::ColourValue mOutlineColor;
	Ogre::ColourValue mFillColor;
};

///////////////////////////////////////////////////////////////////////////////

class Polygon2fVisualization3D :  public PolygonVisualization3DBase<float, 2>
{
	MIRA_META_OBJECT(Polygon2fVisualization3D,
		("Name", "Polygon2f")
		("Description", "Visualizes a 2D float polygon in the xy-plane")
		("Category", "Geometry"))
};

class Polygon2dVisualization3D :  public PolygonVisualization3DBase<double, 2>
{
	MIRA_META_OBJECT(Polygon2dVisualization3D,
		("Name", "Polygon2d")
		("Description", "Visualizes a 2D double polygon in the xy-plane")
		("Category", "Geometry"))
};

class Polygon3fVisualization3D :  public PolygonVisualization3DBase<float, 3>
{
	MIRA_META_OBJECT(Polygon3fVisualization3D,
		("Name", "Polygon3f")
		("Description", "Visualizes a 3D float polygon")
		("Category", "Geometry"))
};

class Polygon3dVisualization3D :  public PolygonVisualization3DBase<double, 3>
{
	MIRA_META_OBJECT(Polygon3dVisualization3D,
		("Name", "Polygon3d")
		("Description", "Visualizes a 3D double polygon")
		("Category", "Geometry"))
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace


MIRA_CLASS_SERIALIZATION(mira::gui::Polygon2fVisualization3D,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Polygon2dVisualization3D,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Polygon3fVisualization3D,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Polygon3dVisualization3D,
                         mira::Visualization3D);
